/**
* @file SettingsDlgBar.cpp
*
* Implementation of class CSettingsDlgBar
*
* @author Matthias Jngel
* @author Martin Ltzsch
* @author Uwe Dffert
*/

#include <io.h>

#include "StdAfx.h"
#include "SettingsDlgBar.h"
#include "Dialogs/SettingNewDialog.h"
#include "Dialogs/GenericModalBinary.h"

#include "MfcTools/GridCtrl/GridCellCombo.h"
#include "RobotControlQueues.h"
#include "Tools/Location.h"



CSettingsDlgBar::CSettingsDlgBar()
: CRobotControlDialogBar(IDD)
{
  //{{AFX_DATA_INIT(CCSettingsDlgBar)
  //}}AFX_DATA_INIT
  m_columnWidth = 100;
  m_columnWidth = 10;
  
  currentSetting = 0;
}

CSettingsDlgBar::~CSettingsDlgBar() {}

SolutionRequest& CSettingsDlgBar::getSolutionRequestForLocalProcesses()
{
  return solutionRequestLocalList[currentSetting];
}

void CSettingsDlgBar::DoDataExchange(CDataExchange* pDX)
{
  CDialog::DoDataExchange(pDX);
  //{{AFX_DATA_MAP(CSettingsDlgBar)
  DDX_Control(pDX, IDC_SETTINGS_SETTINGS_COMBO, m_settingsCombo);
  DDX_Control(pDX, IDC_SETTINGS_TOALL_COMBO, m_toAllRobots);
  DDX_Control(pDX, IDC_SETTINGS_TOSELECTED_COMBO, m_toSelected);
  DDX_Control(pDX, IDC_SETTINGS_GRID, m_Grid);             // associate the grid window with a C++ object
  DDX_Control(pDX, IDC_SETTINGS_NEW, m_buttonNew);
  DDX_Control(pDX, IDC_SETTINGS_DELETE, m_buttonDel);
  DDX_Control(pDX, IDC_SETTINGS_RENAME, m_buttonRen);
  DDX_Control(pDX, IDC_SETTINGS_SAVE, m_buttonSave);
  DDX_Control(pDX, IDC_SETTINGS_MKDEFAULT, m_buttonMakeDefault);
  DDX_Control(pDX, IDC_SETTINGS_STATIC1, m_SelectedRobotsText);
  //}}AFX_DATA_MAP
}

BEGIN_MESSAGE_MAP(CSettingsDlgBar, CDynamicBarDlg)
//{{AFX_MSG_MAP(CSettingsDlgBar)
ON_WM_SIZE()
ON_CBN_SELCHANGE(IDC_SETTINGS_SETTINGS_COMBO, OnSelchangeSettingsCombo)
ON_BN_CLICKED(IDC_SETTINGS_NEW, OnSettingsNew)
ON_BN_CLICKED(IDC_SETTINGS_DELETE, OnSettingsDelete)
ON_BN_CLICKED(IDC_SETTINGS_RENAME, OnSettingsRename)
ON_BN_CLICKED(IDC_SETTINGS_SAVE, OnSettingsSave)
ON_BN_CLICKED(IDC_SETTINGS_MKDEFAULT, OnSettingsMakeDefault)
ON_BN_CLICKED(IDC_SETTINGS_TOSELECTED_COMBO, OnStaCheck)
ON_BN_CLICKED(IDC_SETTINGS_TOALL_COMBO, OnStaCheck)
//}}AFX_MSG_MAP
ON_NOTIFY(NM_CLICK, IDC_SETTINGS_GRID, OnGridClick)
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CSettingsDlgBar message handlers

BOOL CSettingsDlgBar::OnInitDialog()
{
  CDynamicBarDlg::OnInitDialog();
  	
 	CButton *btn;
  btn = (CButton*) GetDlgItem(IDC_SETTINGS_NEW);
	btn->SetBitmap((HBITMAP)::LoadImage(AfxGetInstanceHandle(), MAKEINTRESOURCE(IDB_CREATENEW), IMAGE_BITMAP, 0, 0, LR_MONOCHROME | LR_LOADTRANSPARENT));
  btn = (CButton*) GetDlgItem(IDC_SETTINGS_RENAME);
	btn->SetBitmap((HBITMAP)::LoadImage(AfxGetInstanceHandle(), MAKEINTRESOURCE(IDB_RENAME), IMAGE_BITMAP, 0, 0, LR_MONOCHROME | LR_LOADTRANSPARENT));
  btn = (CButton*) GetDlgItem(IDC_SETTINGS_DELETE);
	btn->SetBitmap((HBITMAP)::LoadImage(AfxGetInstanceHandle(), MAKEINTRESOURCE(IDB_DELETE), IMAGE_BITMAP, 0, 0, LR_COLOR | LR_LOADTRANSPARENT));
  btn = (CButton*) GetDlgItem(IDC_SETTINGS_SAVE);
	btn->SetBitmap((HBITMAP)::LoadImage(AfxGetInstanceHandle(), MAKEINTRESOURCE(IDB_FILESAVE), IMAGE_BITMAP, 0, 0, LR_COLOR | LR_LOADTRANSPARENT));

  AddSzControl(m_Grid, mdResize, mdResize);
  AddSzControl(m_settingsCombo, mdResize, mdNone);
  AddSzControl(m_toSelected, mdRelative, mdNone);
  AddSzControl(m_toAllRobots, mdRelative, mdNone);
  AddSzControl(m_buttonNew, mdRepos, mdNone);
  AddSzControl(m_buttonDel, mdRepos, mdNone);
  AddSzControl(m_buttonRen, mdRepos, mdNone);
  AddSzControl(m_buttonSave, mdRepos, mdNone);
  AddSzControl(m_SelectedRobotsText, mdRelative, mdNone);
  AddSzControl(m_buttonMakeDefault, mdRepos, mdNone);

  loadSettings();
  sendLocal();
  sendToRobot();
  updateComboBox();
  
  CRect rect;
  GetClientRect(rect);
  
  /////////////////////////////////////////////////////////////////////////
  // initialise grid properties
  /////////////////////////////////////////////////////////////////////////
  
  m_Grid.EnableDragAndDrop(false);
  m_Grid.GetDefaultCell(FALSE, FALSE)->SetBackClr(RGB(0xFF, 0xFF, 0xE0));
  m_Grid.EnableToolTips(false);
  m_Grid.EnableTitleTips(false);
  m_Grid.EnableTrackingToolTips(false);
  m_Grid.EnableHiddenColUnhide(false);
  
  CFont font;
  font.CreateFont(
    12,                        // nHeight
    0,                         // nWidth
    0,                         // nEscapement
    0,                         // nOrientation
    FW_NORMAL,                 // nWeight
    FALSE,                     // bItalic
    FALSE,                     // bUnderline
    0,                         // cStrikeOut
    ANSI_CHARSET,              // nCharSet
    OUT_DEFAULT_PRECIS,        // nOutPrecision
    CLIP_DEFAULT_PRECIS,       // nClipPrecision
    DEFAULT_QUALITY,           // nQuality
    DEFAULT_PITCH | FF_SWISS,  // nPitchAndFamily
    "Arial");                 // lpszFacename
  m_Grid.SetFont(&font);
  
  initGrid();
  
  m_Grid.SetFixedColumnSelection(false);
  m_Grid.SetFixedRowSelection(false);
  
  //  m_Grid.AutoSize();
  m_Grid.SetEditable(true);
  m_Grid.SetSingleRowSelection(false);
  m_Grid.SetSingleColSelection(false);

  if (AfxGetApp()->GetProfileInt("SETTINGS", "STA", 0))
  {
    m_toSelected.SetCheck(0);
    m_toAllRobots.SetCheck(1);
  }
  else
  {
    m_toSelected.SetCheck(1);
    m_toAllRobots.SetCheck(0);
  }  
  return TRUE;  // return TRUE  unless you set the focus to a control
}

void CSettingsDlgBar::OnSize(UINT nType, int cx, int cy) 
{
  CDynamicBarDlg::OnSize(nType, cx, cy);
  if (::IsWindow(m_Grid.GetSafeHwnd()))
  {
    // resize the grid very small ...
    CRect rect;
    m_Grid.GetWindowRect(rect);
    m_columnWidth = (rect.right-rect.left - 20) / 3;
    m_rowHeight = (rect.bottom-rect.top - 10) / m_Grid.GetRowCount();
    m_Grid.SetColumnWidth(0,m_columnWidth);
    m_Grid.SetColumnWidth(1,m_columnWidth);
    m_Grid.SetColumnWidth(2,m_columnWidth);
    bool toSmall = false;
    if(m_rowHeight < 11) 
    {
      toSmall = true;
      m_rowHeight = 11;
    }
    
    for(int row = 0; row < m_Grid.GetRowCount(); row++)
    {
      m_Grid.SetRowHeight(row,m_rowHeight);
    }
    
    //... and expand to fit.
    if(toSmall)
    {
      //      m_Grid.ExpandColumnsToFit();
    }
    else
      m_Grid.ExpandToFit();
    
  }
}

// (Thanks to Koay Kah Hoe for this)
BOOL CSettingsDlgBar::PreTranslateMessage(MSG* pMsg) 
{
  if( pMsg->message == WM_KEYDOWN )
  {
    if(pMsg->wParam == VK_RETURN
      || pMsg->wParam == VK_ESCAPE )
    {
      ::TranslateMessage(pMsg);
      ::DispatchMessage(pMsg);
      return TRUE;                    // DO NOT process further
    }
  }
  return CDialog::PreTranslateMessage(pMsg);
}	

void CSettingsDlgBar::OnGridClick(NMHDR *pNotifyStruct, LRESULT* /*pResult*/)
{
  NM_GRIDVIEW* pItem = (NM_GRIDVIEW*) pNotifyStruct;
  if (pItem->iRow < 0)
    return;
  //    TRACE2("Clicked on row %d, col %d\n", pItem->iRow, pItem->iColumn);
}

void CSettingsDlgBar::initGrid() 
{
  UpdateData();
  m_Grid.SetFixedColumnCount(0); 
  m_Grid.SetFixedRowCount(1); 
  m_Grid.SetColumnCount(3);
  m_Grid.SetRowCount(SolutionRequest::numOfModules + 1);
  
  //  m_Grid.SetAutoSizeStyle();
  
  // fill rows/cols with text
  for (int row = 0; row < m_Grid.GetRowCount(); row++)
  {
    SolutionRequest::ModuleID currentModule = (SolutionRequest::ModuleID)(row - 1);
    GV_ITEM Item;
    Item.mask = GVIF_TEXT | GVIF_FGCLR;
    Item.nState = GVIS_READONLY;
    Item.row = row;
    Item.col = 0;
    Item.crFgClr = RGB(0,0,0);    
    if (row == 0)
      Item.strText = "modules";
    else
    {
      Item.strText = SolutionRequest::getModuleName(currentModule);
      CGridCellCombo* pCell = (CGridCellCombo*) m_Grid.GetCell(row,0);
      pCell->SetState(GVIS_READONLY);
    }
    m_Grid.SetItem(&Item);
    
    ///////////////////////////////////////////
    Item.col = 1;
    if (row == 0)
    {
      Item.strText = "solutions robot";
      Item.crFgClr = RGB(0,0,0);    
    }
    else
    {
      Item.strText = SolutionRequest::getModuleSolutionName(
        currentModule, solutionRequestRobotList[currentSetting].solutions[row - 1]);
      
      if (SolutionRequest::getDefaultSolution(currentModule) == 
        solutionRequestRobotList[currentSetting].solutions[row - 1])
      {
        Item.crFgClr = RGB(0,0,0);    
      }
      else
      {
        Item.crFgClr = RGB(0,0,255);    
      }
      
      m_Grid.SetCellType(row,1, RUNTIME_CLASS(CGridCellCombo));
      
      CStringArray options;
      
      if(SolutionRequest::getDefaultSolution(currentModule) == SolutionRequest::disabled)
        options.Add("disabled *");
      else
        options.Add("disabled");
      
      CString string;
      for (int i=0;i<SolutionRequest::getNumOfSolutions(currentModule);i++)
      {
        string = SolutionRequest::getModuleSolutionName(
          currentModule,(SolutionRequest::ModuleSolutionID)i);
        if (i == SolutionRequest::getDefaultSolution(currentModule))
          string += " *";
        options.Add(string);
      }
      
      CGridCellCombo *pCell = (CGridCellCombo*) m_Grid.GetCell(row,1);
      pCell->SetOptions(options);
    }
    m_Grid.SetItem(&Item);
    
    /////////////////////////////////////////////
    Item.col = 2;
    if (row == 0)
    {
      Item.strText = "solutions local";
      Item.crFgClr = RGB(0,0,0);
    }
    else
    {
      Item.strText = SolutionRequest::getModuleSolutionName(
        currentModule, solutionRequestLocalList[currentSetting].solutions[row - 1]);
      
      if (SolutionRequest::getDefaultSolution(currentModule) == 
        solutionRequestLocalList[currentSetting].solutions[row - 1])
      {
        Item.crFgClr = RGB(0,0,0);    
      }
      else
      {
        Item.crFgClr = RGB(0,0,255);    
      }
      
      m_Grid.SetCellType(row,2, RUNTIME_CLASS(CGridCellCombo));
      
      CStringArray options;
      
      if(SolutionRequest::getDefaultSolution(currentModule) == SolutionRequest::disabled)
        options.Add("disabled *");
      else
        options.Add("disabled");
      
      CString string;
      for (int i=0;i<SolutionRequest::getNumOfSolutions(currentModule);i++)
      {
        string = SolutionRequest::getModuleSolutionName(
          currentModule,(SolutionRequest::ModuleSolutionID)i);
        if (i == SolutionRequest::getDefaultSolution(currentModule))
          string += " *";
        options.Add(string);
      }
      
      CGridCellCombo *pCell = (CGridCellCombo*) m_Grid.GetCell(row,2);
      pCell->SetOptions(options);
    }
    m_Grid.SetItem(&Item);
  }
  UpdateData(FALSE);
}

BOOL CSettingsDlgBar::OnNotify(WPARAM wParam, LPARAM lParam, LRESULT* pResult) 
{
  if (wParam == (WPARAM)m_Grid.GetDlgCtrlID())
  {
    *pResult = 1;
    GV_DISPINFO *pDispInfo = (GV_DISPINFO*)lParam;
    switch (pDispInfo->hdr.code)
    {
    case GVN_SELCHANGED: 
      break;
    case GVN_SELCHANGING: 
      break;
    case GVN_COLUMNCLICK: 
      break;
    case GVN_ENDLABELEDIT: 
      if (modeOfSetting[currentSetting]==readOnly)
      {
        if (numberOfSettings == maxNumberOfSettings) break;
            
        newSetting();
       /*
        {
          //reset everything to "default"
          OnSelchangeSettingsCombo();
          return CDialog::OnNotify(wParam, lParam, pResult);
        }*/

      }
      CGridCellCombo* pCellCombo = (CGridCellCombo*)(m_Grid.GetCell(pDispInfo->item.row, pDispInfo->item.col));
      
      CString cellText, defaultString, solutionString;
      SolutionRequest::ModuleID currentModule = (SolutionRequest::ModuleID)(pDispInfo->item.row - 1);
      int selectedSolution = -1; //disabled
      if(pDispInfo->item.col == 1 || pDispInfo->item.col == 2)
      {
        cellText = m_Grid.GetCell(pDispInfo->item.row,pDispInfo->item.col)->GetText();
        defaultString = 
          SolutionRequest::getModuleSolutionName(
          currentModule, SolutionRequest::getDefaultSolution(currentModule) );
        if(cellText == (defaultString + " *") )
        {
          pCellCombo->SetText(defaultString);
          pCellCombo->SetTextClr(RGB(0,0,0));
          cellText = defaultString;
        }
        else
          pCellCombo->SetTextClr(RGB(0,0,255));
        
        for (int i=0;i<SolutionRequest::getNumOfSolutions(currentModule);i++)
        {
          solutionString = SolutionRequest::getModuleSolutionName(
            currentModule,(SolutionRequest::ModuleSolutionID)i);
          if(solutionString == cellText)
            selectedSolution = i;
        }
      }
      
      switch(pDispInfo->item.col)
      {
      case 1:
        solutionRequestRobotList[currentSetting].solutions[currentModule] = (SolutionRequest::ModuleSolutionID)selectedSolution;
        sendToRobot();
        break;
      case 2:
        solutionRequestLocalList[currentSetting].solutions[currentModule] = (SolutionRequest::ModuleSolutionID)selectedSolution;
        sendLocal();
        break;
      }
      //mark as changed
      if (modeOfSetting[currentSetting]==isSaved) modeOfSetting[currentSetting]=isChanged;
      updateComboBox();
      break;
    }
  }
  return CDialog::OnNotify(wParam, lParam, pResult);
}

void CSettingsDlgBar::sendLocal() 
{
  if (m_toAllRobots.GetCheck() ==0) {
    getQueues().toSimulated.selectedRobot.out.bin << solutionRequestLocalList[currentSetting];
    getQueues().toSimulated.selectedRobot.out.finishMessage(idSolutionRequest);
  } else {
    getQueues().toSimulated.allRobots.out.bin << solutionRequestLocalList[currentSetting];
    getQueues().toSimulated.allRobots.out.finishMessage(idSolutionRequest);
  }
}

void CSettingsDlgBar::sendToRobot() 
{
  if (m_toAllRobots.GetCheck() ==0) {
    getQueues().toPhysical.selectedRobot.out.bin << solutionRequestRobotList[currentSetting];
    getQueues().toPhysical.selectedRobot.out.finishMessage(idSolutionRequest);
  } else {
    getQueues().toPhysical.allRobots.out.bin << solutionRequestRobotList[currentSetting];
    getQueues().toPhysical.allRobots.out.finishMessage(idSolutionRequest);
  }
}

void CSettingsDlgBar::loadSettings()
{
  nameOfSetting[0] = "default";
  modeOfSetting[0] = readOnly;
  
  nameOfSetting[1] = "disable all";
  modeOfSetting[1] = readOnly;
  
  solutionRequestRobotList[0].setDefaultSolutions();
  solutionRequestLocalList[0].setDefaultSolutions(); 

  numberOfSettings = 2;
  struct _finddata_t ff;
  char ffname[512];
  sprintf(ffname,"%s/Config/Settings/*.*",File::getGTDir());
  long fd=(long)_findfirst(ffname,&ff);
  bool thereAreMore = (fd>0);
  while (thereAreMore)
  {
    if ((strncmp(ff.name,".",1)!=0)&&((ff.attrib & 0x10)== 0))
    {
      if(numberOfSettings==maxNumberOfSettings)
      {
        AfxMessageBox("The settings directory contains to many settings\n-> not all of them were loaded!", MB_OK);
        break;
      }
      else
      {
        nameOfSetting[numberOfSettings++] = ff.name;
      }
    }
    thereAreMore=(_findnext(fd,&ff)==0);
  }
  _findclose(fd);
  
  for(int i=2; i<numberOfSettings; i++)
  {
    CString pathAndFileName = "Settings/";
    pathAndFileName += nameOfSetting[i];
    
    InConfigFile finRobot(pathAndFileName,"Robot");
    if(finRobot.exists())
      finRobot >> solutionRequestRobotList[i];

    InConfigFile finLocal(pathAndFileName,"Local");
    if(finLocal.exists())
      finLocal >> solutionRequestLocalList[i];

    if (solutionRequestRobotList[i].errorWhileReading || solutionRequestLocalList[i].errorWhileReading)
    {
      getQueues().toGUI.out.text << "SettingsDlgBar : Settings file " 
        << nameOfSetting[i] << " was corrupt. Rewriting file.";
      getQueues().toGUI.out.finishMessage(idText);
      //write back to file to correct corrupt setting files
      saveSetting(i);
    } else
      modeOfSetting[i]=isSaved;
  }
  currentSetting = 0;
}

void CSettingsDlgBar::updateComboBox()
{
  m_settingsCombo.ResetContent();
  for(int i = 0; i < numberOfSettings; i++) 
  {
    char s[128];
    switch (modeOfSetting[i])
    {
      case isChanged: sprintf(s,"%s *",nameOfSetting[i]); break;
      case isNew: sprintf(s,"%s  (N)",nameOfSetting[i]); break;
      case readOnly: sprintf(s,"%s  (R)",nameOfSetting[i]); break;
      case isSaved:
      default: sprintf(s,nameOfSetting[i]); break;
    }
    m_settingsCombo.AddString(s);
  }
  m_settingsCombo.SetCurSel(currentSetting);
}


void CSettingsDlgBar::OnSelchangeSettingsCombo() 
{
  currentSetting = m_settingsCombo.GetCurSel();
  initGrid();
  m_Grid.Invalidate();
  sendLocal();
  sendToRobot();
}

void CSettingsDlgBar::OnSettingsNew() 
{
  newSetting();
  OnSettingsRename();
}

void CSettingsDlgBar::newSetting() 
{
  if (numberOfSettings == maxNumberOfSettings) return;

  int i;
  for (i=0; i<numberOfSettings;i++)
  {
    if (strcmp("new setting",nameOfSetting[i])==0 && modeOfSetting[i] == isNew)
      break;
  }

  int previousSetting = currentSetting;
  currentSetting = i;
  if (i == numberOfSettings)
  {
    numberOfSettings++;
    nameOfSetting[currentSetting] = "new setting";
  }
  
  //mark as changed
  modeOfSetting[currentSetting] = isNew;
  updateComboBox();

  solutionRequestLocalList[currentSetting] = solutionRequestLocalList[previousSetting];
  solutionRequestRobotList[currentSetting] = solutionRequestRobotList[previousSetting];

  //OnSettingsRename();
  //initGrid();
  //m_Grid.Invalidate();

/*
  CSettingNewDialog newDialog;
  newDialog.m_nameEdit = "new setting";
  if (newDialog.DoModal()!=IDOK) return;
  if (newDialog.m_nameEdit.GetLength()==0) return;
  
  for (int i=0; i<numberOfSettings;i++)
  {
    if (strcmp(newDialog.m_nameEdit,nameOfSetting[currentSetting])==0)
    {
      AfxMessageBox("There is already another setting with that name -> creating forbidden!", MB_OK);
      return;
    }
  }
  
  int previousSetting = currentSetting;
  currentSetting = numberOfSettings;
  numberOfSettings++;
  nameOfSetting[currentSetting] = newDialog.m_nameEdit;
  //mark as changed
  modeOfSetting[currentSetting] = isNew;
  updateComboBox();
  
  solutionRequestLocalList[currentSetting] = solutionRequestLocalList[previousSetting];
  solutionRequestRobotList[currentSetting] = solutionRequestRobotList[previousSetting];
  
  //  debugKeyTableLocalList[currentSetting] = debugKeyTableLocalList[previousSetting];
  //  debugKeyTableRobotList[currentSetting] = debugKeyTableRobotList[previousSetting];
  
  initGrid();
  m_Grid.Invalidate();
  
  sendLocal();
  sendToRobot();
*/
}

void CSettingsDlgBar::OnSettingsRename() 
{
  if (modeOfSetting[currentSetting]==readOnly)
  {
    AfxMessageBox("Sorry, renaming the read-only default setting\nis not possible.", MB_OK);
  }
  else
  {
    CSettingNewDialog newDialog;
    newDialog.m_nameEdit = nameOfSetting[currentSetting];
    newDialog.DoModal();
    if (newDialog.m_nameEdit.GetLength()==0) return;

    if(newDialog.m_nameEdit != nameOfSetting[currentSetting])
    {
      for (int i=0; i<numberOfSettings;i++)
      {
        if ((strcmp(newDialog.m_nameEdit,nameOfSetting[i])==0)&&
          (i!=currentSetting))
        {
          AfxMessageBox("There is already another setting \nby that name -> renaming canceled!", MB_OK);
          return;
        }
      }
    }

    if (modeOfSetting[currentSetting]!=isNew)
    {
      char sOld[128];
      char sNew[128];
      sprintf(sOld,"%s/Config/Settings/%s",File::getGTDir(),nameOfSetting[currentSetting]);
      sprintf(sNew,"%s/Config/Settings/%s",File::getGTDir(),newDialog.m_nameEdit);
      rename(sOld,sNew);
    } else
      modeOfSetting[currentSetting]=isChanged;
    
    nameOfSetting[currentSetting] = newDialog.m_nameEdit;
    updateComboBox();
    
    initGrid();
    m_Grid.Invalidate();
  }
}

void CSettingsDlgBar::OnSettingsDelete() 
{
  if (modeOfSetting[currentSetting]==readOnly)
  {
    AfxMessageBox("You can't delete a read-only default setting!", MB_OK);
    return;
  }

  CString message; 
  message.Format("Please confirm permanent removal of module setting: \n\n        \"%s\"\n\nDelete?", nameOfSetting[currentSetting]);
  
  CGenericModalBinary deleteDialog("Delete", "Cancel", message);
  if (IDCANCEL == deleteDialog.DoModal())
    return;

  //if (IDCANCEL == AfxMessageBox( message, MB_OKCANCEL))
    //return;

  numberOfSettings--;
  
  if (modeOfSetting[currentSetting]!=isNew)
  {
    char s[128];
    sprintf(s,"%s/Config/Settings/%s",File::getGTDir(),nameOfSetting[currentSetting]);
    remove(s);
  }
  
  for (int i = currentSetting; i < numberOfSettings; i++)
  {
    nameOfSetting[i] = nameOfSetting[i + 1];
    modeOfSetting[i] = modeOfSetting[i + 1];
    solutionRequestLocalList[i] = solutionRequestLocalList[i + 1];
    solutionRequestRobotList[i] = solutionRequestRobotList[i + 1];
    
    //    debugKeyTableLocalList[i] = debugKeyTableLocalList[i + 1];
    //    debugKeyTableRobotList[i] = debugKeyTableRobotList[i + 1];
  }
  m_settingsCombo.DeleteString(currentSetting);
  m_settingsCombo.SetCurSel((currentSetting >= numberOfSettings)?numberOfSettings - 1:currentSetting);
  
  OnSelchangeSettingsCombo();
  /*
  initGrid();
  m_Grid.Invalidate();
  sendLocal();
  sendToRobot();
  */
}

void CSettingsDlgBar::OnSettingsSave() 
{
  switch (modeOfSetting[currentSetting])
  {
  case isNew:
    OnSettingsRename();
    if (modeOfSetting[currentSetting]==isNew) break;
  case isChanged:
    {
      saveSetting(currentSetting);
      break;
    }
  case readOnly:
    {
      AfxMessageBox("This is a read-only default\nsetting, it can't be saved!", MB_OK);
      break;
    }
  }
}

void CSettingsDlgBar::saveSetting(int index)
{
  CString pathAndFileName = "Settings/";
  pathAndFileName += nameOfSetting[index];
  OutTextFile fout(pathAndFileName);
  if(fout.exists())
  {
    fout << "[Robot]" << endl;
    fout << solutionRequestRobotList[index];
    fout << "[Local]" << endl;
    fout << solutionRequestLocalList[index];
    //      fout << debugKeyTableRobotList[index];
    //      fout << debugKeyTableLocalList[index];
    
    //mark as unchanged
    modeOfSetting[index]=isSaved;
    updateComboBox();
  }
  else
  {
    CString message;
    message.Format( "File %s could not be created", pathAndFileName);
    AfxMessageBox( message, MB_OK);
  }
}

void CSettingsDlgBar::OnSettingsMakeDefault()
{
  if (AfxMessageBox("This writes current robot module solutions to \"modules.cfg\". The default solutions will be changed permanently. The new settings apply when restarting RobotControl and when copying to a memory stick and rebooting robot.", MB_OKCANCEL)==1)
  {
    OutTextFile modulesFile(getLocation().getFilename("modules.cfg"));
    modulesFile << "/*"<<"Module"<<"solutions"<<"config"<<"file" << endl;
    modulesFile << ""<<"*"<<"Contains"<<"one"<<"entry"<<"for"<<"each"<<"module"<<"specifying" << endl;
    modulesFile << ""<<"*"<<"which"<<"solution"<<"is"<<"default"<<"and"<<"is"<<"selected"<<"on"<<"startup." << endl;
    modulesFile << ""<<"*/" << endl << endl;
    modulesFile << solutionRequestRobotList[currentSetting];
  }
}

void CSettingsDlgBar::OnStaCheck() 
{
  AfxGetApp()->WriteProfileInt("SETTINGS", "STA",(int)m_toAllRobots.GetCheck());
}


/*
* Change log :
* 
* $Log: SettingsDlgBar.cpp,v $
* Revision 1.10  2004/03/19 18:57:08  jhoffman
* improved drop down combos in the settings dialog
*
* Revision 1.9  2004/03/16 14:00:23  juengel
* Integrated Improvments from "Gnne"
* -ATH2004ERS7Behavior
* -ATHHeadControl
* -KickSelectionTable
* -KickEditor
*
* Revision 1.2  2004/03/15 17:11:40  hoffmann
* - added ATH2004HeadControl
* - added ATH2004LEDControl
* - headmotiontester shows "tilt2"
* - motion process updates odometry while no new robotPose is received, added to motion request
* - some ui adjustments
* - added member function to "field" to find out if robot is in own penalty area for use in the obstacles locator
*
* Revision 1.1.1.1  2004/03/05 10:10:37  loetzsch
* created local cvs for Gnne
*
* Revision 1.8  2004/02/29 17:32:21  jhoffman
* settings dialog resizing behavior improved
*
* Revision 1.7  2004/02/23 13:27:25  jhoffman
* - changes to Main-menu are shown ... this currently results in one additional menu (!) somebody please fix this!
* - added GenericModalBinary-dialog: can be used to create a modal dialog where you can specify the button labels and the descriptive text (like APX...ModalDialog but you can specify what is written on the buttons)
* - CameraToolBar: mode, shutter, gain extracted from dropdown
* - ToolBarEx: added AddString method to write const. strings on a toolbar
*
* Revision 1.6  2004/02/02 15:15:53  jhoffman
* UI improvements, user feedback integrated
*
* Revision 1.5  2004/01/29 12:03:24  jhoffman
* UI changes:
* - Main menu entries renamed, reordered, added some new icons
* - delete-confirmation in settings and WLAN dialog
* - settings dialog uses icons, radio buttons added
* - WLAN dialog restructured
* - motion tester and head motion tested layout improved, stop button added
* - check boxes replaced by radio buttons where appropriate
*
* Revision 1.4  2004/01/19 16:59:39  kerdels
* send to all Checkbox im Settings-Dialog hinzugefgt
*
* Revision 1.3  2003/12/09 19:49:25  loetzsch
* Renamed some of the main queues of RobotControl.
*
* Added possibility to send messages to specific simulated or physical robots.
*
* Revision 1.2  2003/11/30 01:53:19  loetzsch
* prepared RobotControl port to Visual C++ .Net
*
* Revision 1.1  2003/10/07 10:09:38  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/09/26 11:40:12  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.12  2003/05/11 23:46:35  dueffert
* Depend now works with RobotControl too
*
* Revision 1.11  2003/05/11 17:03:00  risler
* added location.cfg
*
* Revision 1.10  2003/03/23 19:11:06  loetzsch
* OUTPUT not allowed in the RobotControl thread anymore.
* Use getQueues().toGUI.out instead.
*
* Revision 1.9  2003/03/17 15:16:32  risler
* grid not locked in readonly settings, create new setting on change in readonly grid instead
*
* Revision 1.8  2003/03/05 17:09:27  loetzsch
* redesign of the queues and debug key tables in RobotControl
*
* Revision 1.7  2002/11/18 17:21:18  dueffert
* RobotControl should be startable in any path now
*
* Revision 1.6  2002/11/09 11:22:51  risler
* added errorWhileReading flag to SolutionRequest
*
* Revision 1.5  2002/11/09 10:50:49  risler
* corrupt settings now get corrected
*
* Revision 1.4  2002/09/24 19:24:20  risler
* added make default button to settings dialog
*
* Revision 1.3  2002/09/24 18:41:57  risler
* human readable solutionrequest streaming operator
* default module solutions read from modules.cfg
*
* Revision 1.2  2002/09/24 16:28:19  risler
* preparation for reading default module solutions from file
*
* Revision 1.1  2002/09/10 15:49:05  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
*
* Revision 1.8  2002/09/03 16:02:35  juengel
* Read only settings are not editable now.
*
* Revision 1.7  2002/09/02 11:43:53  juengel
* Read only settings are not editable now.
*
* Revision 1.6  2002/08/27 17:07:18  dueffert
* improved handling significantly
*
* Revision 1.5  2002/08/07 10:50:21  loetzsch
* - removed direct access to the dialogs
* - faster starting of RobotControl application and main window
* - general clean up
*
* Revision 1.4  2002/07/23 13:43:37  loetzsch
* - new streaming classes
* - removed many #include statements
* - 5 instead of 3 debug queues in RobotControl
* - exchanged StaticQueue with MessageQueue
* - new debug message handling
* - empty constructors in bars / dialogs
* - access to debugkeytables and queues via RobotControlQueues.h and RobotControlDebugKeyTables.h
* - general clean up
*
* Revision 1.3  2002/06/17 08:12:28  jhoffman
* disabled tool tips
*
* Revision 1.2  2002/06/04 00:13:26  loetzsch
* added access to solution request for local processes
*
* Revision 1.1.1.1  2002/05/10 12:40:21  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.4  2002/04/23 17:45:16  loetzsch
* - splitted debugKeyTable into debugKeyTableForRobot and debugKeyTableForLocalProcesses
* - removed Modules Toolbar
* - removed access for dialogs and toolbars to solutionRequest.
* - changed access for dialogs and toolbars to debug queues
* - removed the instance of SolutionRequest in CRobotControlApp
* - only the log player, local processes and the robot put messages into queueToRobotControl
*
* Revision 1.3  2002/02/24 15:50:54  juengel
* SettingsDlgBar completed.
*
* Revision 1.2  2002/02/22 17:17:33  loetzsch
* DebugKey "SendAllStopWatches" removed
*
* Revision 1.1  2002/02/18 14:03:57  juengel
* SettingsDlgBar added.
*
*/
