/**
* @file TSLColorTableToolDlgBar.cpp
*
* Implementation of class CTSLColorTableToolDlgBar
*
* @author <a href="mailto:robocup@andreosterhues.de">Andr Osterhues</a>
*/

#include "StdAfx.h"
#include "TSLColorTableToolDlgBar.h"

#include "Visualization/DebugDrawingManager/DebugDrawingManager.h"
#include "Visualization/ImageMethods.h"
#include "Visualization/PaintMethodsWin32.h"

#include "RobotControlQueues.h"
#include "Platform/Sensors.h"

#include "Tools/Math/Common.h"

CTSLColorTableToolDlgBar::CTSLColorTableToolDlgBar()
: CRobotControlDialogBar(IDD),corrector()
{
	//{{AFX_DATA_INIT(CTSLColorTableToolDlgBar)
	//}}AFX_DATA_INIT
}

CTSLColorTableToolDlgBar::~CTSLColorTableToolDlgBar()
{
  deleteBitmaps();
}

bool CTSLColorTableToolDlgBar::handleMessage(InMessage& message)
{
  if (message.getMessageID() == idImage)
  {
    message.bin >> rawImage;

    /* writing color corrected pixels in the image (a kind of hack) */
		doColorCorrection(rawImage);

		/* Generate noise image */
    generateNoiseImage(rawImage, noiseImage, imageNoise);

    if (this->IsWindowVisible())
    {
      /* Generate segmented image */
      colorTableTSL.generateTSLDialogImage(noiseImage, segmentedImage);

      /* Create TSL histograms */
      createHistograms(noiseImage);

      /* Redraw windows */
      RedrawWindow(NULL, NULL, RDW_INVALIDATE);
    }
    return true;
  }
  else return false;
}

void CTSLColorTableToolDlgBar::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CTSLColorTableToolDlgBar)
	DDX_Control(pDX, IDC_TSLCOLORTABLETOOL_COLOR_SELECT, m_ColorSelect);
	DDX_Control(pDX, IDC_TSLCOLORTABLETOOL_ORDER_LIST,   m_OrderList);
	DDX_Control(pDX, IDC_TSLCOLORTABLETOOL_ORDER_UP,     m_OrderUp);
	DDX_Control(pDX, IDC_TSLCOLORTABLETOOL_ORDER_DOWN,   m_OrderDown);
	DDX_Control(pDX, IDC_TSLCOLORTABLETOOL_LOAD,         m_Load);
	DDX_Control(pDX, IDC_TSLCOLORTABLETOOL_SAVE,         m_Save);
	DDX_Control(pDX, IDC_TSLCOLORTABLETOOL_SAVE_YUV,     m_SaveYUV);
	DDX_Control(pDX, IDC_TSLCOLORTABLETOOL_SEND,         m_Send);
	DDX_Control(pDX, IDC_TSLCOLORTABLETOOL_AUTO,         m_Auto);
	DDX_Control(pDX, IDC_TSLCOLORTABLETOOL_T_MIN_SLIDER, m_TMinSlider);
	DDX_Control(pDX, IDC_TSLCOLORTABLETOOL_T_MAX_SLIDER, m_TMaxSlider);
	DDX_Control(pDX, IDC_TSLCOLORTABLETOOL_T_MIN_NUMBER, m_TMinNumber);
	DDX_Control(pDX, IDC_TSLCOLORTABLETOOL_T_MAX_NUMBER, m_TMaxNumber);
	DDX_Control(pDX, IDC_TSLCOLORTABLETOOL_S_MIN_SLIDER, m_SMinSlider);
	DDX_Control(pDX, IDC_TSLCOLORTABLETOOL_S_MAX_SLIDER, m_SMaxSlider);
	DDX_Control(pDX, IDC_TSLCOLORTABLETOOL_S_MIN_NUMBER, m_SMinNumber);
	DDX_Control(pDX, IDC_TSLCOLORTABLETOOL_S_MAX_NUMBER, m_SMaxNumber);
	DDX_Control(pDX, IDC_TSLCOLORTABLETOOL_L_MIN_SLIDER, m_LMinSlider);
	DDX_Control(pDX, IDC_TSLCOLORTABLETOOL_L_MAX_SLIDER, m_LMaxSlider);
	DDX_Control(pDX, IDC_TSLCOLORTABLETOOL_L_MIN_NUMBER, m_LMinNumber);
	DDX_Control(pDX, IDC_TSLCOLORTABLETOOL_L_MAX_NUMBER, m_LMaxNumber);
	DDX_Control(pDX, IDC_TSLCOLORTABLETOOL_NOISE_SLIDER, m_NoiseSlider);
	DDX_Control(pDX, IDC_TSLCOLORTABLETOOL_NOISE_NUMBER, m_NoiseNumber);
	//}}AFX_DATA_MAP
}


BEGIN_MESSAGE_MAP(CTSLColorTableToolDlgBar, CDialog)
	//{{AFX_MSG_MAP(CTSLColorTableToolDlgBar)
    ON_WM_PAINT()
    ON_WM_SIZE()
    ON_WM_HSCROLL()
    ON_WM_LBUTTONDOWN()
    ON_WM_RBUTTONDOWN()
    ON_WM_TIMER()
    ON_CBN_SELCHANGE(IDC_TSLCOLORTABLETOOL_COLOR_SELECT, OnTSLColorTableToolColorSelect)
    ON_CBN_SELCHANGE(IDC_TSLCOLORTABLETOOL_ORDER_LIST,   OnTSLColorTableToolOrderList)
    ON_BN_CLICKED(IDC_TSLCOLORTABLETOOL_LOAD,            OnTSLColorTableToolLoad)
    ON_BN_CLICKED(IDC_TSLCOLORTABLETOOL_SAVE,            OnTSLColorTableToolSave)
    ON_BN_CLICKED(IDC_TSLCOLORTABLETOOL_SAVE_YUV,        OnTSLColorTableToolSaveYUV)
    ON_BN_CLICKED(IDC_TSLCOLORTABLETOOL_SEND,            OnTSLColorTableToolSend)
    ON_BN_CLICKED(IDC_TSLCOLORTABLETOOL_AUTO,            OnTSLColorTableToolAuto)
    ON_BN_CLICKED(IDC_TSLCOLORTABLETOOL_ORDER_UP,        OnTSLColorTableToolOrderUp)
    ON_BN_CLICKED(IDC_TSLCOLORTABLETOOL_ORDER_DOWN,      OnTSLColorTableToolOrderDown)
    ON_EN_KILLFOCUS(IDC_TSLCOLORTABLETOOL_T_MIN_NUMBER,  OnTSLColorTableToolTMinNumber)
    ON_EN_KILLFOCUS(IDC_TSLCOLORTABLETOOL_T_MAX_NUMBER,  OnTSLColorTableToolTMaxNumber)
    ON_EN_KILLFOCUS(IDC_TSLCOLORTABLETOOL_S_MIN_NUMBER,  OnTSLColorTableToolSMinNumber)
    ON_EN_KILLFOCUS(IDC_TSLCOLORTABLETOOL_S_MAX_NUMBER,  OnTSLColorTableToolSMaxNumber)
    ON_EN_KILLFOCUS(IDC_TSLCOLORTABLETOOL_L_MIN_NUMBER,  OnTSLColorTableToolLMinNumber)
    ON_EN_KILLFOCUS(IDC_TSLCOLORTABLETOOL_L_MAX_NUMBER,  OnTSLColorTableToolLMaxNumber)
    ON_EN_KILLFOCUS(IDC_TSLCOLORTABLETOOL_NOISE_NUMBER,  OnTSLColorTableToolNoiseNumber)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

BOOL CTSLColorTableToolDlgBar::OnInitDialog() 
{
  CString string;
  int     i;

  CDynamicBarDlg::OnInitDialog();

  /* Set slider min/max values */
  m_TMinSlider.SetRange(0, 255, TRUE);
  m_TMaxSlider.SetRange(0, 255, TRUE);
  m_SMinSlider.SetRange(0, 255, TRUE);
  m_SMaxSlider.SetRange(0, 255, TRUE);
  m_LMinSlider.SetRange(0, 255, TRUE);
  m_LMaxSlider.SetRange(0, 255, TRUE);

  /* Add strings to color select */
  for (i = 0; i < numOfColors; i++)
  {
    m_ColorSelect.AddString(ColorClasses::getColorName((colorClass) i));
  }
  m_ColorSelect.SetCurSel(0);

  /* Init TSL color class */
  selectedColorClass = noColor;
  colorTableTSL.clear();

  /* Reset TSL color table to default values */
  colorTableTSL.reset();  

  /* Add strings to order list */
  for (i = 0; i < numOfColors; i++)
  {
    m_OrderList.AddString(ColorClasses::getColorName(colorTableTSL.tsl_order[i]));
  }

  /* Set sliders and number values */
  OnTSLColorTableToolColorSelect();

  /* Create bitmaps for histograms */
  createBitmaps();

  /* Set noise slider to zero */
  imageNoise = 0;
  m_NoiseSlider.SetRange(0, 100, TRUE);
  m_NoiseNumber.SetWindowText("0");

  /* Unset region of interest */
  regionX1 = -1;
  regionY1 = -1;
  regionX2 = -1;
  regionY2 = -1;

  /* Generate segmented image */
  colorTableTSL.generateTSLDialogImage(noiseImage, segmentedImage);

  /* Create TSL histograms */
  createHistograms(noiseImage);

  return TRUE;
}

void CTSLColorTableToolDlgBar::OnPaint() 
{
  CPen blackPen(PS_SOLID, 1, RGB(0, 0, 0));
  CPen checkerPen(PS_DOT, 1, RGB(0, 0, 0));
  CPen *oldPen;
  RECT rectT, rectS, rectL;
  CPaintDC dc(this); // device context for painting
  unsigned char r, g, b;
  int i;

  /* Setup rectangles */
  rectT.left = 216;
  rectT.top = 160;
  rectT.right = rectT.left + 256;
  rectT.bottom = rectT.top + 32;

  rectS.left = rectT.left;
  rectS.top = rectT.top+80;
  rectS.right = rectS.left + 256;
  rectS.bottom = rectS.top + 32;

  rectL.left = rectS.left;
  rectL.top = rectS.top+80;
  rectL.right = rectL.left + 256;
  rectL.bottom = rectL.top + 32;

  /* Draw histogram backgrounds */
	for (i = 0; i < 256; i += 3)
  {
    convertTSLToRGB(i, avgS, avgL, &r, &g, &b);
    CPen histogramPen(PS_SOLID, 3, RGB(r, g, b));
    oldPen = dc.SelectObject(&histogramPen);
	  dc.MoveTo(rectT.left + i, rectT.top);
	  dc.LineTo(rectT.left + i, rectT.top + 32);
    dc.SelectObject(oldPen);
  }
	for (i = 0; i < 256; i += 3)
  {
    convertTSLToRGB(avgT, i, avgL, &r, &g, &b);
    CPen histogramPen(PS_SOLID, 3, RGB(r, g, b));
    oldPen = dc.SelectObject(&histogramPen);
	  dc.MoveTo(rectS.left + i, rectS.top);
	  dc.LineTo(rectS.left + i, rectS.top + 32);
    dc.SelectObject(oldPen);
  }
	for (i = 0; i < 256; i += 3)
  {
    convertTSLToRGB(avgT, avgS, i, &r, &g, &b);
    CPen histogramPen(PS_SOLID, 3, RGB(r, g, b));
    oldPen = dc.SelectObject(&histogramPen);
	  dc.MoveTo(rectL.left + i, rectL.top);
	  dc.LineTo(rectL.left + i, rectL.top + 32);
    dc.SelectObject(oldPen);
  }

  /* Draw histograms */
  oldPen = dc.SelectObject(&blackPen);
	for (i = 0; i < 256; i++) 
  {
	  dc.MoveTo(rectT.left + i, rectT.bottom);
	  dc.LineTo(rectT.left + i, rectT.top + 32 - THistogram[i]);
	  dc.MoveTo(rectS.left + i, rectS.bottom);
	  dc.LineTo(rectS.left + i, rectS.top + 32 - SHistogram[i]);
	  dc.MoveTo(rectL.left + i, rectL.bottom);
	  dc.LineTo(rectL.left + i, rectL.top + 32 - LHistogram[i]);
  }
  dc.SelectObject(oldPen);

  /* Display raw image */
  ImageMethods::paintImage2CDCAsYUV(
    dc, noiseImage, 
    CRect(0, 0, 176, 144));

  /* Display segmented image */
  ImageMethods::paintColorClassImage2CDC(
    dc, segmentedImage, 
    CRect(0, 145, 176, 145 + 144));

  /* Draw region of interest */
  if (regionX2 != -1)
  {
    oldPen = dc.SelectObject(&checkerPen);
    dc.MoveTo(regionX1, regionY1);
    dc.LineTo(regionX2, regionY1);
    dc.LineTo(regionX2, regionY2);
    dc.LineTo(regionX1, regionY2);
    dc.LineTo(regionX1, regionY1);
    dc.MoveTo(regionX1, regionY1 + 145);
    dc.LineTo(regionX2, regionY1 + 145);
    dc.LineTo(regionX2, regionY2 + 145);
    dc.LineTo(regionX1, regionY2 + 145);
    dc.LineTo(regionX1, regionY1 + 145);
    dc.SelectObject(oldPen);
  }
  else if (regionX1 != -1)
  {
    oldPen = dc.SelectObject(&checkerPen);
    dc.MoveTo(regionX1,   0);
    dc.LineTo(regionX1, 144);
    dc.MoveTo(  0, regionY1);
    dc.LineTo(176, regionY1);
    dc.MoveTo(regionX1, 145);
    dc.LineTo(regionX1, 144 + 145);
    dc.MoveTo(  0, regionY1 + 145);
    dc.LineTo(176, regionY1 + 145);
    dc.SelectObject(oldPen);
  }
}

void CTSLColorTableToolDlgBar::OnSize(UINT nType, int cx, int cy) 
{
}

void CTSLColorTableToolDlgBar::OnHScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar) 
{
  CString string;
  selectedColorClass = (colorClass) m_ColorSelect.GetCurSel();

  /* Handle sliders */
  switch (pScrollBar->GetDlgCtrlID())
  {
    case IDC_TSLCOLORTABLETOOL_T_MIN_SLIDER:
    {
      colorTableTSL.tsl_threshold[selectedColorClass][0][0] = m_TMinSlider.GetPos();
      string.Format("%i", m_TMinSlider.GetPos());
      m_TMinNumber.SetWindowText(string);
      break;
    }
    case IDC_TSLCOLORTABLETOOL_T_MAX_SLIDER:
    {
      colorTableTSL.tsl_threshold[selectedColorClass][1][0] = m_TMaxSlider.GetPos();
      string.Format("%i", m_TMaxSlider.GetPos());
      m_TMaxNumber.SetWindowText(string);
      break;
    }
    case IDC_TSLCOLORTABLETOOL_S_MIN_SLIDER:
    {
      colorTableTSL.tsl_threshold[selectedColorClass][0][1] = m_SMinSlider.GetPos();
      string.Format("%i", m_SMinSlider.GetPos());
      m_SMinNumber.SetWindowText(string);
      break;
    }
    case IDC_TSLCOLORTABLETOOL_S_MAX_SLIDER:
    {
      colorTableTSL.tsl_threshold[selectedColorClass][1][1] = m_SMaxSlider.GetPos();
      string.Format("%i", m_SMaxSlider.GetPos());
      m_SMaxNumber.SetWindowText(string);
      break;
    }
    case IDC_TSLCOLORTABLETOOL_L_MIN_SLIDER:
    {
      colorTableTSL.tsl_threshold[selectedColorClass][0][2] = m_LMinSlider.GetPos();
      string.Format("%i", m_LMinSlider.GetPos());
      m_LMinNumber.SetWindowText(string);
      break;
    }
    case IDC_TSLCOLORTABLETOOL_L_MAX_SLIDER:
    {
      colorTableTSL.tsl_threshold[selectedColorClass][1][2] = m_LMaxSlider.GetPos();
      string.Format("%i", m_LMaxSlider.GetPos());
      m_LMaxNumber.SetWindowText(string);
      break;
    }
    case IDC_TSLCOLORTABLETOOL_NOISE_SLIDER:
    {
      imageNoise = m_NoiseSlider.GetPos();
      string.Format("%i", m_NoiseSlider.GetPos());
      m_NoiseNumber.SetWindowText(string);

      /* Generate noise image with new noise value*/
      generateNoiseImage(rawImage, noiseImage, imageNoise);

      /* Create TSL histograms */
      createHistograms(noiseImage);
      break;
    }
    default:
    {
      break;
    }
  }

  /* Calculate average TSL values from slider position */
  if (m_TMinSlider.GetPos() > m_TMaxSlider.GetPos())
  {
    avgT = (m_TMinSlider.GetPos() + m_TMaxSlider.GetPos() + 255) >> 1;
    if (avgT > 255)
    {
      avgT -= 256;
    }
  }
  else
  {
    avgT = (m_TMinSlider.GetPos() + m_TMaxSlider.GetPos()) >> 1;
  }
  avgS = (m_SMinSlider.GetPos() + m_SMaxSlider.GetPos()) >> 1;
  avgL = (m_LMinSlider.GetPos() + m_LMaxSlider.GetPos()) >> 1;

  /* Redraw window */
  RedrawWindow(NULL, NULL, RDW_INVALIDATE);

  /* Update segmented image */
  updateSegmentedImage();

  CDynamicBarDlg::OnHScroll(nSBCode, nPos, pScrollBar);
}

void CTSLColorTableToolDlgBar::OnLButtonDown(UINT nFlags, CPoint point) 
{
  /* Did the user click in the image area? */
  if ((point.x >= 0) && (point.y >= 0) && (point.x < 176) && (point.y < (144 + 145)))
  {
    /* Did the user click on the segmented image? */
    if (point.y >= 145)
    {
      /* Set coords relative to raw image */
      point.y -= 145;
    }

    if ((regionX1 == -1) || (regionX2 != -1))
    {
      /* Set first point of region of interest */
      regionX1 = point.x;
      regionY1 = point.y;
      regionX2 = -1;
      regionY2 = -1;
    }
    else
    {
      /* Set second point of region of interest */
      regionX2 = point.x;
      regionY2 = point.y;

      /* Assure that region1 is the left top edge */
      if (regionX1 > regionX2)
      {
          regionX2 = regionX1;
          regionX1 = point.x;
      }
      if (regionY1 > regionY2)
      {
          regionY2 = regionY1;
          regionY1 = point.y;
      }
    }

    /* Create TSL histograms */
    createHistograms(noiseImage);

    /* Redraw window */
    RedrawWindow(NULL, NULL, RDW_INVALIDATE);
  }

  CDynamicBarDlg::OnLButtonDown(nFlags, point);
}

void CTSLColorTableToolDlgBar::OnRButtonDown(UINT nFlags, CPoint point) 
{
  /* Did the user click in the image area? */
  if ((point.x >= 0) && (point.y >= 0) && (point.x < 176) && (point.y < (144 + 145)))
  {
    /* Unset region of interest */
    regionX1 = -1;
    regionY1 = -1;
    regionX2 = -1;
    regionY2 = -1;

    /* Create TSL histograms */
    createHistograms(noiseImage);

    /* Redraw window */
    RedrawWindow(NULL, NULL, RDW_INVALIDATE);
  }

  CDynamicBarDlg::OnRButtonDown(nFlags, point);
}

void CTSLColorTableToolDlgBar::OnTSLColorTableToolColorSelect()
{
  CString string;

  /* Get selected color class */
  selectedColorClass = (colorClass) m_ColorSelect.GetCurSel();

  /* Set selected color class in TSL order list */
  for (int k = 0; k < numOfColors; k++)
  {
    if (colorTableTSL.tsl_order[k] == selectedColorClass)
    {
      m_OrderList.SetCurSel(k);
      break;
    }
  }

  /* Set sliders */
  m_TMinSlider.SetPos(colorTableTSL.tsl_threshold[selectedColorClass][0][0]);
  m_TMaxSlider.SetPos(colorTableTSL.tsl_threshold[selectedColorClass][1][0]);
  m_SMinSlider.SetPos(colorTableTSL.tsl_threshold[selectedColorClass][0][1]);
  m_SMaxSlider.SetPos(colorTableTSL.tsl_threshold[selectedColorClass][1][1]);
  m_LMinSlider.SetPos(colorTableTSL.tsl_threshold[selectedColorClass][0][2]);
  m_LMaxSlider.SetPos(colorTableTSL.tsl_threshold[selectedColorClass][1][2]);

  /* Set numbers */
  string.Format("%i", m_TMinSlider.GetPos());
  m_TMinNumber.SetWindowText(string);
  string.Format("%i", m_TMaxSlider.GetPos());
  m_TMaxNumber.SetWindowText(string);
  string.Format("%i", m_SMinSlider.GetPos());
  m_SMinNumber.SetWindowText(string);
  string.Format("%i", m_SMaxSlider.GetPos());
  m_SMaxNumber.SetWindowText(string);
  string.Format("%i", m_LMinSlider.GetPos());
  m_LMinNumber.SetWindowText(string);
  string.Format("%i", m_LMaxSlider.GetPos());
  m_LMaxNumber.SetWindowText(string);

  /* Calculate average TSL values from slider position */
  if (m_TMinSlider.GetPos() > m_TMaxSlider.GetPos())
  {
    avgT = (m_TMinSlider.GetPos() + m_TMaxSlider.GetPos() + 255) >> 1;
    if (avgT > 255)
    {
      avgT -= 256;
    }
  }
  else
  {
    avgT = (m_TMinSlider.GetPos() + m_TMaxSlider.GetPos()) >> 1;
  }
  avgS = (m_SMinSlider.GetPos() + m_SMaxSlider.GetPos()) >> 1;
  avgL = (m_LMinSlider.GetPos() + m_LMaxSlider.GetPos()) >> 1;

  /* Redraw windows */
  RedrawWindow(NULL, NULL, RDW_INVALIDATE);
}

void CTSLColorTableToolDlgBar::OnTSLColorTableToolOrderList()
{
  orderColorClass = (colorClass) m_OrderList.GetCurSel();
}

void CTSLColorTableToolDlgBar::OnTSLColorTableToolLoad()
{
  int i;
  CString defaultPath = File::getGTDir();
  defaultPath += "/Config/";
  defaultPath.Replace('/','\\');
  CString pathName =
    AfxGetApp()->GetProfileString("TSLColorTableToolDlg", "openPath", defaultPath);
  pathName += "*.tsl";
  
  CFileDialog fileDialog(true, ".tsl",pathName,
    OFN_HIDEREADONLY | OFN_EXPLORER | OFN_ENABLESIZING | OFN_NOCHANGEDIR | OFN_NONETWORKBUTTON,
    "color tables (*.tsl)|*.tsl||", this);
  
  if (fileDialog.DoModal() == IDOK)
  {
    CString fileName = fileDialog.GetFileName();
    CString pathAndFileName = fileDialog.GetPathName();
    pathName = 
      pathAndFileName.Left(
      pathAndFileName.GetLength() -
      fileName.GetLength()
      );
    AfxGetApp()->WriteProfileString("TSLColorTableToolDlg", "openPath", pathName);
    
    InBinaryFile fin(pathAndFileName);
    if (fin.exists())
    {
      fin >> colorTableTSL;
      colorTableTSL.generateTSLDialogImage(noiseImage, segmentedImage);
      OnTSLColorTableToolColorSelect();

      Invalidate(true);
      UpdateWindow();

      for (i = 0; i < numOfColors; i++)
      {
        m_OrderList.DeleteString(0);
      }
      for (i = 0; i < numOfColors; i++)
      {
        m_OrderList.AddString(ColorClasses::getColorName(colorTableTSL.tsl_order[i]));
      }

      m_OrderList.Invalidate(true);
      m_OrderList.UpdateWindow();
      
      m_ColorSelect.Invalidate(true);
      m_ColorSelect.UpdateWindow();

      getQueues().toSimulated.allRobots.out.bin << colorTableTSL;
      getQueues().toSimulated.allRobots.out.finishMessage(idColorTableTSL);
    }
    else
    {
      CString message;
      message.Format( "File %s not found.", pathAndFileName);
      AfxMessageBox(message, MB_OK);
    }
  }
}

void CTSLColorTableToolDlgBar::OnTSLColorTableToolSave()
{
  CString defaultPath = File::getGTDir();
  defaultPath += "/Config/";
  defaultPath.Replace('/','\\');
  CString pathName = 
    AfxGetApp()->GetProfileString("TSLColorTableToolDlg", "savePath", defaultPath);
  pathName += "*.tsl";
  
  CFileDialog fileDialog(false, ".tsl",pathName,
    OFN_HIDEREADONLY | OFN_EXPLORER | OFN_ENABLESIZING | OFN_NOCHANGEDIR | OFN_NONETWORKBUTTON
    , "color tables (*.tsl)|*.tsl||", this);
  
  if (fileDialog.DoModal() == IDOK)
  {
    CString fileName = fileDialog.GetFileName();
    CString pathAndFileName = fileDialog.GetPathName();
    CString pathName = 
      pathAndFileName.Left(
      pathAndFileName.GetLength() -
      fileName.GetLength()
      );
    AfxGetApp()->WriteProfileString("TSLColorTableToolDlg", "savePath", pathName);
    
    OutBinaryFile fout(pathAndFileName);
    if (fout.exists())
    {
      fout << colorTableTSL;
    }
    else
    {
      CString message;
      message.Format( "File %s not found.", pathAndFileName);
      AfxMessageBox(message, MB_OK);
    }
  }
}

void CTSLColorTableToolDlgBar::OnTSLColorTableToolSaveYUV()
{
  CString defaultPath = File::getGTDir();
  defaultPath += "/Config/";
  defaultPath.Replace('/','\\');
  CString pathName = 
    AfxGetApp()->GetProfileString("TSLColorTableToolDlg", "savePath", defaultPath);
  pathName += "*.yuv";
  
  CFileDialog fileDialog(false, ".yuv",pathName,
    OFN_HIDEREADONLY | OFN_EXPLORER | OFN_ENABLESIZING | OFN_NOCHANGEDIR | OFN_NONETWORKBUTTON
    , "YUV (*.yuv)|*.yuv||", this);
  
  if (fileDialog.DoModal() == IDOK)
  {
    CString fileName = fileDialog.GetFileName();
    CString pathAndFileName = fileDialog.GetPathName();
    CString pathName = 
      pathAndFileName.Left(
      pathAndFileName.GetLength() -
      fileName.GetLength()
      );
    AfxGetApp()->WriteProfileString("TSLColorTableToolDlg", "savePath", pathName);
    
    OutBinaryFile fout(pathAndFileName);
    if (fout.exists())
    {
      unsigned long tmpLong;
      unsigned char tmpByte;
      int saveX1, saveX2, saveY1, saveY2;

      if (regionX2 == -1)
      {
        saveX1 = 0;
        saveY1 = 0;
        saveX2 = 176;
        saveY2 = 144;
      }
      else
      {
        saveX1 = regionX1;
        saveY1 = regionY1;
        saveX2 = regionX2;
        saveY2 = regionY2;
      }

      tmpLong = 0x00565559;           /* Magic 'YUV\0' */
      fout << tmpLong;
      tmpLong = saveX2 - saveX1;      /* Width */
      fout << tmpLong;
      tmpLong = saveY2 - saveY1;      /* Height */
      fout << tmpLong;

      for (int j = saveY1; j < saveY2; j++)
      {
        for (int i = saveX1; i < saveX2; i++)
        {
          for (int c = 0; c < 3; c++)
          {
            tmpByte = noiseImage.image[j][c][i];
            fout << tmpByte;
          }
        }
      }
    }
    else
    {
      CString message;
      message.Format( "File %s not found.", pathAndFileName);
      AfxMessageBox(message, MB_OK);
    }
  }
}

void CTSLColorTableToolDlgBar::OnTSLColorTableToolSend()
{
  getQueues().toSimulated.allRobots.out.bin << colorTableTSL;
  getQueues().toSimulated.allRobots.out.finishMessage(idColorTableTSL);

  getQueues().toPhysical.allRobots.out.bin << colorTableTSL;
  getQueues().toPhysical.allRobots.out.finishMessage(idColorTableTSL);
}

void CTSLColorTableToolDlgBar::OnTSLColorTableToolAuto()
{
  CString string;

  colorTableTSL.tsl_threshold[selectedColorClass][0][0] = leftT;
  m_TMinSlider.SetPos(leftT);
  string.Format("%i", leftT);
  m_TMinNumber.SetWindowText(string);
  colorTableTSL.tsl_threshold[selectedColorClass][1][0] = rightT;
  m_TMaxSlider.SetPos(rightT);
  string.Format("%i", rightT);
  m_TMaxNumber.SetWindowText(string);
  colorTableTSL.tsl_threshold[selectedColorClass][0][1] = leftS;
  m_SMinSlider.SetPos(leftS);
  string.Format("%i", leftS);
  m_SMinNumber.SetWindowText(string);
  colorTableTSL.tsl_threshold[selectedColorClass][1][1] = rightS;
  m_SMaxSlider.SetPos(rightS);
  string.Format("%i", rightS);
  m_SMaxNumber.SetWindowText(string);
  colorTableTSL.tsl_threshold[selectedColorClass][0][2] = leftL;
  m_LMinSlider.SetPos(leftL);
  string.Format("%i", leftL);
  m_LMinNumber.SetWindowText(string);
  colorTableTSL.tsl_threshold[selectedColorClass][1][2] = rightL;
  m_LMaxSlider.SetPos(rightL);
  string.Format("%i", rightL);
  m_LMaxNumber.SetWindowText(string);
}

void CTSLColorTableToolDlgBar::OnTSLColorTableToolOrderUp()
{
  colorClass  tmp;
  int         i;

  orderColorClass = (colorClass) m_OrderList.GetCurSel();
  if (orderColorClass > 0)
  {
    tmp                                          = colorTableTSL.tsl_order[orderColorClass - 1];
    colorTableTSL.tsl_order[orderColorClass - 1] = colorTableTSL.tsl_order[orderColorClass];
    colorTableTSL.tsl_order[orderColorClass]     = tmp;
    for (i = 0; i < numOfColors; i++)
    {
      m_OrderList.DeleteString(0);
    }
    for (i = 0; i < numOfColors; i++)
    {
      m_OrderList.AddString(ColorClasses::getColorName(colorTableTSL.tsl_order[i]));
    }
    m_OrderList.SetCurSel(orderColorClass - 1);
    updateSegmentedImage();
  }
}

void CTSLColorTableToolDlgBar::OnTSLColorTableToolOrderDown()
{
  colorClass  tmp;
  int         i;

  orderColorClass = (colorClass) m_OrderList.GetCurSel();
  if (orderColorClass < (colorTableTSL.tsl_index - 1))
  {
    tmp                                          = colorTableTSL.tsl_order[orderColorClass + 1];
    colorTableTSL.tsl_order[orderColorClass + 1] = colorTableTSL.tsl_order[orderColorClass];
    colorTableTSL.tsl_order[orderColorClass]     = tmp;
    for (i = 0; i < numOfColors; i++)
    {
      m_OrderList.DeleteString(0);
    }
    for (i = 0; i < numOfColors; i++)
    {
      m_OrderList.AddString(ColorClasses::getColorName(colorTableTSL.tsl_order[i]));
    }
    m_OrderList.SetCurSel(orderColorClass + 1);
    updateSegmentedImage();
  }
}

void CTSLColorTableToolDlgBar::OnTSLColorTableToolTMinNumber()
{
  CString string;
  int number;

  m_TMinNumber.GetWindowText(string);
  number = atoi(string);
  if (number < 0)
  {
    number = 0;
  }
  else if (number > 255)
  {
    number = 255;
  }
  colorTableTSL.tsl_threshold[selectedColorClass][0][0] = number;
  m_TMinSlider.SetPos(number);
  string.Format("%i", number);
  m_TMinNumber.SetWindowText(string);
}

void CTSLColorTableToolDlgBar::OnTSLColorTableToolTMaxNumber()
{
  CString string;
  int number;

  m_TMaxNumber.GetWindowText(string);
  number = atoi(string);
  if (number < 0)
  {
    number = 0;
  }
  else if (number > 255)
  {
    number = 255;
  }
  colorTableTSL.tsl_threshold[selectedColorClass][1][0] = number;
  m_TMaxSlider.SetPos(number);
  string.Format("%i", number);
  m_TMaxNumber.SetWindowText(string);
}

void CTSLColorTableToolDlgBar::OnTSLColorTableToolSMinNumber()
{
  CString string;
  int number;

  m_SMinNumber.GetWindowText(string);
  number = atoi(string);
  if (number < 0)
  {
    number = 0;
  }
  else if (number > 255)
  {
    number = 255;
  }
  colorTableTSL.tsl_threshold[selectedColorClass][0][1] = number;
  m_SMinSlider.SetPos(number);
  string.Format("%i", number);
  m_SMinNumber.SetWindowText(string);
}

void CTSLColorTableToolDlgBar::OnTSLColorTableToolSMaxNumber()
{
  CString string;
  int number;

  m_SMaxNumber.GetWindowText(string);
  number = atoi(string);
  if (number < 0)
  {
    number = 0;
  }
  else if (number > 255)
  {
    number = 255;
  }
  colorTableTSL.tsl_threshold[selectedColorClass][1][1] = number;
  m_SMaxSlider.SetPos(number);
  string.Format("%i", number);
  m_SMaxNumber.SetWindowText(string);
}

void CTSLColorTableToolDlgBar::OnTSLColorTableToolLMinNumber()
{
  CString string;
  int number;

  m_LMinNumber.GetWindowText(string);
  number = atoi(string);
  if (number < 0)
  {
    number = 0;
  }
  else if (number > 255)
  {
    number = 255;
  }
  colorTableTSL.tsl_threshold[selectedColorClass][0][2] = number;
  m_LMinSlider.SetPos(number);
  string.Format("%i", number);
  m_LMinNumber.SetWindowText(string);
}

void CTSLColorTableToolDlgBar::OnTSLColorTableToolLMaxNumber()
{
  CString string;
  int number;

  m_LMaxNumber.GetWindowText(string);
  number = atoi(string);
  if (number < 0)
  {
    number = 0;
  }
  else if (number > 255)
  {
    number = 255;
  }
  colorTableTSL.tsl_threshold[selectedColorClass][1][2] = number;
  m_LMaxSlider.SetPos(number);
  string.Format("%i", number);
  m_LMaxNumber.SetWindowText(string);
}

void CTSLColorTableToolDlgBar::OnTSLColorTableToolNoiseNumber()
{
  CString string;

  m_NoiseNumber.GetWindowText(string);
  imageNoise = atoi(string);
  if (imageNoise < 0)
  {
    imageNoise = 0;
  }
  else if (imageNoise > 100)
  {
    imageNoise = 100;
  }
  m_NoiseSlider.SetPos(imageNoise);
  string.Format("%i", imageNoise);
  m_NoiseNumber.SetWindowText(string);
}

void CTSLColorTableToolDlgBar::createBitmaps()
{
  int xSize = 256;
  int ySize = 32;

  pBufferTHistogram = (BITMAPINFOHEADER*) new
    char[sizeof(BITMAPINFOHEADER) + ((xSize * 3 + 3) & 0xfffffffc) * ySize];
  pBufferTHistogram->biSize          = sizeof(BITMAPINFOHEADER);
  pBufferTHistogram->biWidth         = xSize;
  pBufferTHistogram->biHeight        = ySize;
  pBufferTHistogram->biPlanes        = 1;
  pBufferTHistogram->biBitCount      = 24;
  pBufferTHistogram->biCompression   = BI_RGB;
  pBufferTHistogram->biSizeImage     = 0;
  pBufferTHistogram->biXPelsPerMeter = 0;
  pBufferTHistogram->biYPelsPerMeter = 0;
  pBufferTHistogram->biClrUsed       = 0;
  pBufferTHistogram->biClrImportant  = 0;
}

void CTSLColorTableToolDlgBar::deleteBitmaps()
{
  delete [] (char*) pBufferTHistogram;
}

void CTSLColorTableToolDlgBar::updateSegmentedImage()
{
  RECT rect;
  colorTableTSL.generateTSLDialogImage(noiseImage, segmentedImage);
  rect.left   = 0;
  rect.top    = 145;
  rect.right  = 176;  
  rect.bottom = 145 + 144;
  InvalidateRect(&rect, false);
  UpdateWindow();
}

void CTSLColorTableToolDlgBar::createHistograms(Image& image)
{
  unsigned char y, u, v, t, s, l;
  unsigned int maxT, maxS, maxL;
  int startX, startY, endX, endY;
  register int i, j;
  unsigned int threshold;
  
  /* Reset max values */
  maxT = 1;
  maxS = 1;
  maxL = 1;

  /* Reset left and right values */
  leftT  = 0;
  rightT = 0;
  leftS  = 0;
  rightS = 0;
  leftL  = 0;
  rightL = 0;

  /* Reset histogram tables */
  for (i = 0; i < 256; i++)
  {
    THistogram[i] = 0;
    SHistogram[i] = 0;
    LHistogram[i] = 0;
  }
  
  /* Define region */
  if (regionX2 == -1)
  {
    startX = 0;
    startY = 0;
    endX   = 176;
    endY   = 144;
  }
  else
  {
    startX = regionX1;
    startY = regionY1;
    endX   = regionX2;
    endY   = regionY2;
  }

  /* Gather statistics */
  for (j = startY; j < endY; j++) 
  {
    for (i = startX; i < endX; i++) 
    {
      y = image.image[j][0][i];
      u = image.image[j][1][i];
      v = image.image[j][2][i];
      colorTableTSL.convertYUVToTSL(y, u, v, &t, &s, &l);

      THistogram[t]++;
      SHistogram[s]++;
      LHistogram[l]++;

      if (THistogram[t] > maxT) 
      {
        maxT = THistogram[t];
        leftT = t;
      }
      if (SHistogram[s] > maxS)
      {
        maxS = SHistogram[s];
        leftS = s;
      }
      if (LHistogram[l] > maxL)
      {
        maxL = LHistogram[l];
        leftL = l;
      }
    }
  }

  /* Normalize */
  for (i = 0; i < 256; i++)
  {
    THistogram[i] = (THistogram[i] << 5) / maxT;
    SHistogram[i] = (SHistogram[i] << 5) / maxS;
    LHistogram[i] = (LHistogram[i] << 5) / maxL;
  }

  /* Find left and right boundaries */
  rightT = leftT;
  rightS = leftS;
  rightL = leftL;
  threshold = 1;
  while ((THistogram[leftT] > threshold) && (leftT > 0))
  {
    leftT--;
  }
  while ((THistogram[rightT] > threshold) && (rightT < 255))
  {
    rightT++;
  }
  while ((SHistogram[leftS] > threshold) && (leftS > 0))
  {
    leftS--;
  }
  while ((SHistogram[rightS] > threshold) && (rightS < 255))
  {
    rightS++;
  }
  while ((LHistogram[leftL] > threshold) && (leftL > 0))
  {
    leftL--;
  }
  while ((LHistogram[rightL] > threshold) && (rightL < 255))
  {
    rightL++;
  }

}

void CTSLColorTableToolDlgBar::convertTSLToRGB(
    unsigned char t, 
    unsigned char s, 
    unsigned char l,
    unsigned char *r,
    unsigned char *g,
    unsigned char *b
  ) 
{
  double y1, u1, v1, r1, g1, b1, rad, phi;
  
  /* Convert TSL to YUV */
  rad = (((double) s) * 0.555) / 255.0;
  phi = (((double) t) * 2.0 * 3.14159) / 255.0;
  phi += 0.25 * 3.14159;
  y1 = (double) l;
  u1 =  (rad * cos(phi)) * 255.0;
  v1 = -(rad * sin(phi)) * 255.0;

  /* Crop UV */
  if (u1 < -128.0)
  {
    u1 = -128.0;
  }
  else if (u1 > 127.0)
  {
    u1 = 127.0;
  }
  if (v1 < -128.0)
  {
    v1 = -128.0;
  }
  else if (v1 > 127.0)
  {
    v1 = 127.0;
  }

  /* Convert YUV to RGB */
  r1 = y1 + 1.371 * v1;
  g1 = y1 - 0.336 * u1 - 0.698 * v1;
  b1 = y1 + 1.732 * u1;

  /* Crop RGB */
  if (r1 < 0.0)
  {
    r1 = 0.0;
  }
  else if (r1 > 255.0)
  {
    r1 = 255.0;
  }
  if (g1 < 0.0)
  {
    g1 = 0.0;
  }
  else if (g1 > 255.0)
  {
    g1 = 255.0;
  }
  if (b1 < 0.0)
  {
    b1 = 0.0;
  }
  else if (b1 > 255.0)
  {
    b1 = 255.0;
  }

  /* Return values */
  *r = (unsigned char) r1;
  *g = (unsigned char) g1;
  *b = (unsigned char) b1;
}

void CTSLColorTableToolDlgBar::generateNoiseImage(Image& rawImage, Image& noiseImage, unsigned int noise)
{
  double v;
  register int i, j, c;

  /* If noise is zero, just copy image */
  if (noise == 0)
  {
    for (j = 0; j < 144; j++) 
    {
      for (c = 0; c < 3; c++)
      {
        for (i = 0; i < 176; i++) 
        {
          noiseImage.image[j][c][i] = rawImage.image[j][c][i];
        }
      }
    }
    return;
  }

  /* Generate noise image */  
  for (j = 0; j < 144; j++) 
  {
    for (c = 0; c < 3; c++)
    {
      for (i = 0; i < 176; i++) 
      {
        /* Get value */
        v = (double) rawImage.image[j][c][i];

        /* Add noise in the range [-noise/2, ..., noise/2] */
        v += random() * noise - (noise >> 1);

        /* Crop */
        if (v < 0.0)
        {
          v = 0.0;
        }
        else if (v > 255.0)
        {
          v = 255.0;
        }

        /* Write back modified value */
        noiseImage.image[j][c][i] = (unsigned char) v;
      }
    }
  }
}

void CTSLColorTableToolDlgBar::doColorCorrection(Image& image){
	int width = image.cameraInfo.resolutionWidth;
	int height = image.cameraInfo.resolutionHeight;
	for (int x = 0; x < width; x++)
		for(int y = 0; y < height; y++){
			corrector.getCorrectedPixel(x,y,
				image.image[y][0][x],image.image[y][1][x],image.image[y][2][x]);
		}

}


/*
* Change log :
* 
* $Log: TSLColorTableToolDlgBar.cpp,v $
* Revision 1.4  2004/03/04 09:55:49  schmidtb
* color correction integrated
*
* Revision 1.3  2003/12/09 21:56:43  schmidtb
* speed up of generateColorTableImage() and
* method generateTSLDialogImage() to ColorTableTSL
* changes made in TSLColorTableToolDlgBar
*
* Revision 1.2  2003/12/09 19:49:26  loetzsch
* Renamed some of the main queues of RobotControl.
*
* Added possibility to send messages to specific simulated or physical robots.
*
* Revision 1.1  2003/10/07 10:09:38  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/09/26 11:40:13  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.13  2003/05/11 23:46:35  dueffert
* Depend now works with RobotControl too
*
* Revision 1.12  2003/04/07 15:03:07  osterhues
* Load now changes TSL order list properly
* Region of interest is reset correctly now
*
* Revision 1.11  2003/03/31 03:56:22  osterhues
* "Load" sends loaded TSL color table to all simulated robots
* "Send" sends current TSL color table to all simulated and physical robots
*
* Revision 1.10  2003/03/05 17:09:28  loetzsch
* redesign of the queues and debug key tables in RobotControl
*
* Revision 1.9  2003/02/21 18:50:53  osterhues
* Bugfix: Save YUV didn't save correct width and height when no region of interest is used
*
* Revision 1.8  2003/02/20 12:35:27  osterhues
* Added auto functionality
*
* Revision 1.7  2003/02/20 03:16:40  deutsch
* Save image function added
*
* Revision 1.6  2003/02/12 06:12:18  osterhues
* Fixed number editing (works now)
* Colorized histograms
*
* Revision 1.5  2003/02/11 17:49:32  osterhues
* Added noise slider
* Added region of interest
*
*/