/**
* @file TimeDiagramDlgBar.cpp
*
* Implementation of class CTimeDiagramDlgBar
*
* @author Matthias Jngel
*/

#include "StdAfx.h"
#include "TimeDiagramDlgBar.h"

#include "RobotControlDebugKeyTables.h"
#include "RobotControlQueues.h"

CTimeDiagramDlgBar::CTimeDiagramDlgBar()
	: CRobotControlDialogBar(IDD)
{
  //{{AFX_DATA_INIT(CTimeDiagramDlgBar)
  //}}AFX_DATA_INIT
  bmpOffScreen = NULL;
  oldBitmap = NULL;
}

CTimeDiagramDlgBar::~CTimeDiagramDlgBar()
{
  if(bmpOffScreen)
  {
    dcOffScreen.SelectObject(oldBitmap);
    bmpOffScreen->DeleteObject();
    delete(bmpOffScreen);
  }
  deleteBrushes();
  saveCurrentSelection();
  AfxGetApp()->WriteProfileInt("TimeDiagram","average range",averageRange);
}


void CTimeDiagramDlgBar::DoDataExchange(CDataExchange* pDX)
{
  CDialog::DoDataExchange(pDX);
  //{{AFX_DATA_MAP(CTimeDiagramDlgBar)
  DDX_Control(pDX, IDC_TD_STATIC1, m_static1);
  DDX_Control(pDX, IDC_TD_AVERAGE_SLIDER, m_averageSlider);
  DDX_Control(pDX, IDC_TD_STOP_COMBO, m_stopCombo);
  //}}AFX_DATA_MAP
}


BEGIN_MESSAGE_MAP(CTimeDiagramDlgBar, CDynamicBarDlg)
//{{AFX_MSG_MAP(CTimeDiagramDlgBar)
ON_WM_PAINT()
ON_WM_SIZE()
ON_WM_CONTEXTMENU()
ON_CBN_SELCHANGE(IDC_TD_STOP_COMBO, OnSelchangeStopCombo)
ON_WM_HSCROLL()
//}}AFX_MSG_MAP
END_MESSAGE_MAP()


BOOL CTimeDiagramDlgBar::OnInitDialog() 
{
  CDynamicBarDlg::OnInitDialog();
  
  averageRange = AfxGetApp()->GetProfileInt("TimeDiagram","average range",10);
  
  m_averageSlider.SetRange(2,500);
  m_averageSlider.SetPos(averageRange);
  m_averageSlider.SetTicFreq(10);
  m_averageSlider.SetLineSize(1);
  
  CString text;
  text.Format("avg: %i", averageRange);
  m_static1.SetWindowText(text.GetBuffer(text.GetLength()));
  
  AddSzControl(m_averageSlider,mdResize,mdNone);
  //init the ring buffers
  clearBuffers();
  
  m_stopCombo.SetCurSel(0);
  stopMode = displayLogFiles;
  unselectAll();
  loadSelection();
  createBrushes();
  return TRUE;
}

void CTimeDiagramDlgBar::OnPaint() 
{
  CPaintDC dc(this); // device context for painting
  
  //  CBrush backGroundBrush(GetSysColor(COLOR_WINDOWFRAME));
  CRect currentRect;
  currentRect.top = 0;
  currentRect.left = 0;
  currentRect.bottom = rect.bottom;
  currentRect.right = rect.right;
  
  CBrush backGroundBrush(RGB(255,255,255));
  dcOffScreen.FillRect(&currentRect, &backGroundBrush);
  
  enum Layout{compact, normal, large} layout = normal;
  
  int minHeightPerTime = 18;
  int maxHeightPerTime = 100;
  int heightPerTime;
  if(numberOfSelectedTimes)
    heightPerTime = (rect.bottom - rect.top) / numberOfSelectedTimes;
  else
    heightPerTime = minHeightPerTime;
  
  if(heightPerTime > 50)
  {
    layout = large;
  }
  
  if(heightPerTime > maxHeightPerTime) heightPerTime = maxHeightPerTime;
  
  if(rect.right < 340) layout = normal;
  if(rect.right < 240) layout = compact;
  
  if(heightPerTime < 30)
  {
    if(heightPerTime < minHeightPerTime) heightPerTime = 18;
    layout = compact;
  }
  
  int fontSize = 14;
  if(layout == large) fontSize = 19;
  
  CFont font;
  CFont* oldFont;
  font.CreateFont(
    fontSize,                  // nHeight
    0,                         // nWidth
    0,                         // nEscapement
    0,                         // nOrientation
    FW_NORMAL,                 // nWeight
    FALSE,                     // bItalic
    FALSE,                     // bUnderline
    0,                         // cStrikeOut
    ANSI_CHARSET,              // nCharSet
    OUT_DEFAULT_PRECIS,        // nOutPrecision
    CLIP_DEFAULT_PRECIS,       // nClipPrecision
    DEFAULT_QUALITY,           // nQuality
    DEFAULT_PITCH | FF_SWISS,  // nPitchAndFamily
    "Arial");                 // lpszFacename
  oldFont = dcOffScreen.SelectObject(&font);
  
  CPen greyPen, blackPen;
  greyPen.CreatePen(PS_SOLID, 1, RGB(127,127,127));
  blackPen.CreatePen(PS_SOLID, 1, RGB(0,0,0));
  CPen* oldPen;
  
  oldPen = dcOffScreen.SelectObject(&blackPen);
  
  int currentY = 10;
  int namePosX, namePosY, timePosX, timePosY, perSecondPosX, perSecondPosY;
  int currentTimeXPos, minTimeXPos, maxTimeXPos;
  for(int i = 0; i < Stopwatch::numberOfStopwatchEventIDs; i++)
  {
    Stopwatch::StopwatchEventID stopwatchEventID = Stopwatch::StopwatchEventID(i);
    if(showTime[stopwatchEventID])
    {
      if(layout == normal)
      {
        currentRect.left = 120;
        currentRect.right = (long)(
          (rect.right - rect.left - currentRect.left - 10) * 
          averageMeasuredTime[i] /
          millisecondsPerView +
          currentRect.left + 2);
        currentTimeXPos = (long)(
          (rect.right - rect.left - currentRect.left - 10) * 
          measuredTime[stopwatchEventID].getEntry(0) /
          millisecondsPerView +
          currentRect.left + 2);
        minTimeXPos = (long)(
          (rect.right - rect.left - currentRect.left - 10) * 
          minMeasuredTime[i] /
          millisecondsPerView +
          currentRect.left + 2);
        maxTimeXPos = (long)(
          (rect.right - rect.left - currentRect.left - 10) * 
          maxMeasuredTime[i] /
          millisecondsPerView +
          currentRect.left + 2);
        
        currentRect.top = currentY;
        currentRect.bottom = currentY + heightPerTime - 5;
        namePosX = 10;
        namePosY = currentY;
        timePosX = 10;
        timePosY = currentY + 15;
        perSecondPosX = 10 + 50;
        perSecondPosY = currentY + 15;
      }
      if(layout == compact)
      {
        currentRect.left = 5;
        currentRect.right = (long)(
          (rect.right - rect.left - currentRect.left - 5) * 
          averageMeasuredTime[i] /
          millisecondsPerView +
          currentRect.left + 2);
        currentTimeXPos = (long)(
          (rect.right - rect.left - currentRect.left - 5) * 
          measuredTime[stopwatchEventID].getEntry(0) /
          millisecondsPerView +
          currentRect.left + 2);
        minTimeXPos = (long)(
          (rect.right - rect.left - currentRect.left - 5) * 
          minMeasuredTime[i] /
          millisecondsPerView +
          currentRect.left + 2);    
        maxTimeXPos = (long)(
          (rect.right - rect.left - currentRect.left - 5) * 
          maxMeasuredTime[i] /
          millisecondsPerView +
          currentRect.left + 2);    
        
        currentRect.top = currentY;
        currentRect.bottom = currentY + heightPerTime - 16;
        namePosX = 5;
        namePosY = currentY + heightPerTime - 17;
        timePosX = 5 + 105;
        timePosY = currentY + heightPerTime - 17;
        perSecondPosX = 5 + 105 + 35;
        perSecondPosY = currentY + heightPerTime - 17;
      }
      if(layout == large)
      {
        currentRect.left = 170;
        currentRect.right = (long)(
          (rect.right - rect.left - currentRect.left - 10) * 
          averageMeasuredTime[i] /
          millisecondsPerView +
          currentRect.left + 2);
        currentTimeXPos = (long)(
          (rect.right - rect.left - currentRect.left - 10) * 
          measuredTime[stopwatchEventID].getEntry(0) /
          millisecondsPerView +
          currentRect.left + 2);
        minTimeXPos = (long)(
          (rect.right - rect.left - currentRect.left - 10) * 
          minMeasuredTime[i] /
          millisecondsPerView +
          currentRect.left + 2);
        maxTimeXPos = (long)(
          (rect.right - rect.left - currentRect.left - 10) * 
          maxMeasuredTime[i] /
          millisecondsPerView +
          currentRect.left + 2);
        currentRect.top = currentY;
        currentRect.bottom = currentY + heightPerTime - 5;
        namePosX = 10;
        namePosY = currentY;
        timePosX = 10;
        timePosY = currentY + 15;
        perSecondPosX = 10 + 60;
        perSecondPosY = currentY + 15;
      }
      
      dcOffScreen.FillRect(&currentRect, &brush[i]);
      
      dcOffScreen.SelectObject(&greyPen);
      dcOffScreen.MoveTo(minTimeXPos, currentRect.top);
      dcOffScreen.LineTo(minTimeXPos, currentRect.bottom);
      
      dcOffScreen.MoveTo(maxTimeXPos, currentRect.top);
      dcOffScreen.LineTo(maxTimeXPos, currentRect.bottom);
      
      dcOffScreen.SelectObject(&blackPen);
      dcOffScreen.MoveTo(currentTimeXPos, currentRect.top);
      dcOffScreen.LineTo(currentTimeXPos, currentRect.bottom);
      
      dcOffScreen.SetBkMode(TRANSPARENT);
      
      CString timeString;
      timeString.Format("%2.1f ms", averageMeasuredTime[i]);
      
      int timeDiff =         
        startTime[stopwatchEventID].getEntry(0) - 
        startTime[stopwatchEventID].getEntry(1);
      
      CString perSecondString;
      perSecondString.Format("%5.1f/sec", averageFrequency[stopwatchEventID]);
      
      dcOffScreen.TextOut(
        namePosX,
        namePosY,
        Stopwatch::getStopwatchEventIDName(stopwatchEventID));
      
      dcOffScreen.TextOut(
        timePosX,
        timePosY,
        timeString);
      
      dcOffScreen.TextOut(
        perSecondPosX,
        perSecondPosY,
        perSecondString);
      
      currentY += heightPerTime;
    }
  }
  
  dcOffScreen.SelectObject(oldFont);
  dcOffScreen.SelectObject(oldPen);
  
  greyPen.DeleteObject();
  blackPen.DeleteObject();
  font.DeleteObject();
  
  dc.BitBlt(rect.left, rect.top, rect.right, rect.bottom,
    &dcOffScreen, 0, 0, SRCCOPY);
}

void CTimeDiagramDlgBar::OnSize(UINT nType, int cx, int cy) 
{
  CDynamicBarDlg::OnSize(nType,cx,cy);
  
  CPaintDC dc(this); // device context for painting
  
  rect.left = 0;
  rect.right = cx;
  rect.top = 25;
  rect.bottom = cy - 25;
  
  if(!dcOffScreen)
    dcOffScreen.CreateCompatibleDC(&dc);
  
  if(bmpOffScreen)
  {
    dcOffScreen.SelectObject(oldBitmap);
    bmpOffScreen->DeleteObject();
    delete(bmpOffScreen);
  }
  bmpOffScreen = new CBitmap;
  bmpOffScreen->CreateCompatibleBitmap(
    &dc, cx, cy);
  oldBitmap = dcOffScreen.SelectObject(bmpOffScreen);
  RedrawWindow(NULL, NULL, RDW_INVALIDATE);
}

bool CTimeDiagramDlgBar::handleMessage(InMessage& message)
{
  if (message.getMessageID() == idStopwatch)
  {
    unsigned long time1, time2, counter1;
    char id;

    message.bin >> id >> time1 >> time2 >> counter1;
    
    Stopwatch::StopwatchEventID stopwatchEventID = Stopwatch::StopwatchEventID(id);
    
    // erase all old times if a measured time is older than the last one 
    if (startTime[stopwatchEventID].getNumberOfEntries() > 0 
      && time1 + 500 < startTime[stopwatchEventID].getEntry(0))
    {
      clearBuffers();
    }
    
    if (stopMode == displayLogFiles)
    {
      if (showTime[stopwatchEventID] == false)
      {
        showTime[stopwatchEventID] = true;
        numberOfSelectedTimes++;
      }
    }
    
    startTime[stopwatchEventID].add(time1);
    measuredTime[stopwatchEventID].add(time2 - time1);
    counter[stopwatchEventID].add(counter1);
    
    //recalculate average of times
    averageMeasuredTime[stopwatchEventID] = 0;
    minMeasuredTime[stopwatchEventID] = 100000;
    maxMeasuredTime[stopwatchEventID] = 0;
    
    calculateAverageValues(stopwatchEventID);
    
    RedrawWindow(NULL, NULL, RDW_INVALIDATE);
    return true;
  }
  return false;
}

void CTimeDiagramDlgBar::calculateAverageValues(Stopwatch::StopwatchEventID stopwatchEventID)
{
  int numberOfEntriesForAverage = 
    (averageRange < measuredTime[stopwatchEventID].getNumberOfEntries() ? averageRange 
    :measuredTime[stopwatchEventID].getNumberOfEntries());
  
  for(int i = 0; i < numberOfEntriesForAverage; i++)
  {
    averageMeasuredTime[stopwatchEventID] += measuredTime[stopwatchEventID].getEntry(i);
    if(measuredTime[stopwatchEventID].getEntry(i) > maxMeasuredTime[stopwatchEventID])
      maxMeasuredTime[stopwatchEventID] = measuredTime[stopwatchEventID].getEntry(i);
    if(measuredTime[stopwatchEventID].getEntry(i) < minMeasuredTime[stopwatchEventID])
      minMeasuredTime[stopwatchEventID] = measuredTime[stopwatchEventID].getEntry(i);
  }
  if (numberOfEntriesForAverage > 1)
  {
    averageMeasuredTime[stopwatchEventID] /= numberOfEntriesForAverage;
    
    averageFrequency[stopwatchEventID] = 1000.0 * 
      (counter[stopwatchEventID].getEntry(0) - counter[stopwatchEventID].getEntry(numberOfEntriesForAverage - 1)) /
      (startTime[stopwatchEventID].getEntry(0) - startTime[stopwatchEventID].getEntry(numberOfEntriesForAverage - 1));
  }   
}

void CTimeDiagramDlgBar::clearBuffers()
{
  for(int i = 0; i < Stopwatch::numberOfStopwatchEventIDs; i++)
  {
    Stopwatch::StopwatchEventID stopwatchEventID = Stopwatch::StopwatchEventID(i);
    startTime[stopwatchEventID].init();
    measuredTime[stopwatchEventID].init();
    counter[stopwatchEventID].init();
    
    averageMeasuredTime[stopwatchEventID] = 0;
    averageFrequency[stopwatchEventID] = 0;
    maxMeasuredTime[stopwatchEventID] = 0;
    minMeasuredTime[stopwatchEventID] = 0;
    
    if (stopMode == displayLogFiles)
    {
      showTime[stopwatchEventID] = false;
      loadSelection();
    }
  }
}

void CTimeDiagramDlgBar::OnContextMenu(CWnd* pWnd, CPoint point) 
{
  CMenu menu;
  VERIFY( menu.CreatePopupMenu() );
  
  UINT nFlags = 0;
  
  if(stopMode != displayLogFiles) 
  {
    // add time StopwatchEventIDs
    for(int i = 0; i < Stopwatch::numberOfStopwatchEventIDs; i++)
    {
      if (showTime[i])
      {
        nFlags = MF_CHECKED;
      }
      else
      {
        nFlags = 0;
      }
      
      VERIFY( menu.AppendMenu( MF_BYCOMMAND | nFlags,
        i + 3000, Stopwatch::getStopwatchEventIDName((Stopwatch::StopwatchEventID)i)) );
    }
    menu.AppendMenu(MF_SEPARATOR);
  }
  menu.AppendMenu(0,3100,"Export values to csv file");
  
  
  // Track menu
  int nID = menu.TrackPopupMenu( TPM_LEFTALIGN | TPM_RIGHTBUTTON | TPM_RETURNCMD | TPM_NONOTIFY,
    point.x, point.y, this);
  
  if (nID == 3100)
  {
    exportToFileAsCSV();
  }
  else
  {
    nID -= 3000;
    if(0 <= nID && nID < Stopwatch::numberOfStopwatchEventIDs)
    {
      showTime[nID] = !showTime[nID];
      if(showTime[nID]) numberOfSelectedTimes++;
      else numberOfSelectedTimes--;
      
      switch(stopMode)
      {
      case displayLogFiles:
        getDebugKeyTables().forPhysicalRobots.set(Stopwatch::getDebugKeyID(
          (Stopwatch::StopwatchEventID)nID),DebugKey::disabled);
        
        getDebugKeyTables().forSimulatedRobots.set(Stopwatch::getDebugKeyID(
          (Stopwatch::StopwatchEventID)nID),DebugKey::disabled);
        break;
      case stopTimesOnRobot:
        getDebugKeyTables().forPhysicalRobots.set(Stopwatch::getDebugKeyID(
          (Stopwatch::StopwatchEventID)nID),
          (showTime[nID] ==true? DebugKey::always : DebugKey::disabled) 
          );
        
        getDebugKeyTables().forSimulatedRobots.set(Stopwatch::getDebugKeyID(
          (Stopwatch::StopwatchEventID)nID),DebugKey::disabled); 
        
        break;
      case stopTimesLocal:
        getDebugKeyTables().forPhysicalRobots.set(Stopwatch::getDebugKeyID(
          (Stopwatch::StopwatchEventID)nID),DebugKey::disabled);
        
        getDebugKeyTables().forSimulatedRobots.set(Stopwatch::getDebugKeyID(
          (Stopwatch::StopwatchEventID)nID),
          (showTime[nID] ==true? DebugKey::always : DebugKey::disabled) 
          );
        
        break;
      }
      getDebugKeyTables().sendForSimulatedRobots();
      getDebugKeyTables().sendForPhysicalRobots();
 
      RedrawWindow(NULL, NULL, RDW_INVALIDATE);
    }
  }
}

void CTimeDiagramDlgBar::createBrushes()
{
  crColor[Stopwatch::imageProcessor] = RGB(127,255,127);
  crColor[Stopwatch::selfLocator] = RGB(255,63,63);
  crColor[Stopwatch::ballLocator] = RGB(255,127,0);
  //crColor[Stopwatch::] = RGB(,,);
  //crColor[Stopwatch::] = RGB(,,);
  
  for(int i = 0; i < Stopwatch::numberOfStopwatchEventIDs; i++)
  {
    brush[i].CreateSolidBrush(crColor[i]);
  }
}

void CTimeDiagramDlgBar::deleteBrushes()
{
  for(int i = 0; i < Stopwatch::numberOfStopwatchEventIDs; i++)
  {
    brush[i].DeleteObject();
  }
}


void CTimeDiagramDlgBar::OnSelchangeStopCombo() 
{
  saveCurrentSelection();
  
  stopMode = (StopMode)(m_stopCombo.GetCurSel());
  
  unselectAll();
  clearBuffers();
  switch(stopMode)
  {
  case displayLogFiles:
    //stop stopwatch on robot and local
    getDebugKeyTables().sendForSimulatedRobots();
    getDebugKeyTables().sendForPhysicalRobots();
    loadSelection();
    break;
  case stopTimesOnRobot:
    //stop stopwatch local
    getDebugKeyTables().sendForSimulatedRobots();
    //start stopwatch on robot
    loadSelection();
    getDebugKeyTables().sendForPhysicalRobots();
    break;
  case stopTimesLocal:
    //stop stopwatch on robot
    getDebugKeyTables().sendForPhysicalRobots();
    //start stopwatch local
    loadSelection();
    getDebugKeyTables().sendForSimulatedRobots();
    break;
  }
  RedrawWindow(NULL, NULL, RDW_INVALIDATE);
}


void CTimeDiagramDlgBar::unselectAll()
{
  for(int i = 0; i < Stopwatch::numberOfStopwatchEventIDs; i++)
  {
    showTime[i] = false;
    getDebugKeyTables().forPhysicalRobots.set(Stopwatch::getDebugKeyID(
      (Stopwatch::StopwatchEventID)i),
      DebugKey::disabled
      );
    getDebugKeyTables().forSimulatedRobots.set(Stopwatch::getDebugKeyID(
      (Stopwatch::StopwatchEventID)i),
      DebugKey::disabled
      );
  }
  numberOfSelectedTimes = 0;
}

void CTimeDiagramDlgBar::loadSelection()
{
  if(stopMode == displayLogFiles)
  {
    showTime[Stopwatch::imageProcessor] = true;
    showTime[Stopwatch::selfLocator] = true;
    numberOfSelectedTimes = 2;
    
    return;
  }
  
  CString stopString;
  if(stopMode == stopTimesOnRobot)
    stopString = "TimeDiagram\\stopTimesOnRobot";
  
  if(stopMode == stopTimesLocal)
    stopString = "TimeDiagram\\stopTimesLocal";
  
  // set some default values for the first use of RobotControl
  showTime[Stopwatch::imageProcessor] 
    = (AfxGetApp()->GetProfileInt(stopString,"imageProcessor",1) == 1);
  AfxGetApp()->WriteProfileInt(stopString,"imageProcessor",
    (showTime[Stopwatch::imageProcessor]?1:0));
  
  showTime[Stopwatch::selfLocator] 
    = (AfxGetApp()->GetProfileInt(stopString,"selfLocator",1) == 1);
  AfxGetApp()->WriteProfileInt(stopString,"selfLocator",
    (showTime[Stopwatch::imageProcessor]?1:0));
  
  numberOfSelectedTimes = 0;
  for(int i = 0; i < Stopwatch::numberOfStopwatchEventIDs; i++)
  {
    Stopwatch::StopwatchEventID stopwatchEventID = Stopwatch::StopwatchEventID(i);
    
    int showTimeInt = AfxGetApp()->GetProfileInt(stopString,
      Stopwatch::getStopwatchEventIDName(stopwatchEventID),0);
    
    if(showTimeInt)
    {
      numberOfSelectedTimes ++;
      showTime[i] = true;
    }
    else
    {
      showTime[i] = false;
    }
    
  if(stopMode == stopTimesOnRobot)
    getDebugKeyTables().forPhysicalRobots.set(
      Stopwatch::getDebugKeyID(stopwatchEventID),
      (showTime[i] ==true? DebugKey::always : DebugKey::disabled) 
      );
  
  if(stopMode == stopTimesLocal)
    getDebugKeyTables().forSimulatedRobots.set(
      Stopwatch::getDebugKeyID(stopwatchEventID),
      (showTime[i] ==true? DebugKey::always : DebugKey::disabled) 
      );
  }
}

void CTimeDiagramDlgBar::saveCurrentSelection()
{
  CString stopString;
  switch(stopMode)
  {
  case displayLogFiles:
    break;
  case stopTimesOnRobot:
    stopString = "TimeDiagram\\stopTimesOnRobot";
    break;
  case stopTimesLocal:
    stopString = "TimeDiagram\\stopTimesLocal";
    break;
  }
  for(int i = 0; i < Stopwatch::numberOfStopwatchEventIDs; i++)
  {
    Stopwatch::StopwatchEventID stopwatchEventID = Stopwatch::StopwatchEventID(i);
    AfxGetApp()->WriteProfileInt(
      stopString,
      Stopwatch::getStopwatchEventIDName(stopwatchEventID),
      showTime[stopwatchEventID]
      );
  }
}

void CTimeDiagramDlgBar::exportToFileAsCSV()
{
  CString defaultPath = File::getGTDir();
  defaultPath += "/Config/";
  defaultPath.Replace('/','\\');
  CString pathName = 
    AfxGetApp()->GetProfileString("TimeDiagram", "csvWritePath", defaultPath);
  pathName += "*.csv";
  
  CFileDialog fileDialog(false, ".csv",pathName,
    OFN_HIDEREADONLY | OFN_EXPLORER | OFN_ENABLESIZING | OFN_NOCHANGEDIR | OFN_NONETWORKBUTTON
    , "csv files (*.csv)|*.csv||", this);
  
  if (fileDialog.DoModal()==IDOK)
  {
    CString fileName = fileDialog.GetFileName();
    CString pathAndFileName = fileDialog.GetPathName();
    CString pathName = 
      pathAndFileName.Left(
      pathAndFileName.GetLength() -
      fileName.GetLength()
      );
    AfxGetApp()->WriteProfileString("TimeDiagram", "csvWritePath", pathName);
    
    OutTextFile fout(pathAndFileName);
    
    fout << "id,averageTime,minTime,maxTime,frequency" << endl;
    for(int i = 0; i < Stopwatch::numberOfStopwatchEventIDs; i++)
    {
      Stopwatch::StopwatchEventID stopwatchEventID = Stopwatch::StopwatchEventID(i);
      
      if(showTime[stopwatchEventID])
      {
        CString frequency;
        frequency.Format("%.1f",averageFrequency[stopwatchEventID]);
        
        fout << Stopwatch::getStopwatchEventIDName(stopwatchEventID) << "," 
          << averageMeasuredTime[stopwatchEventID] << ","
          << minMeasuredTime[stopwatchEventID] << ","
          << maxMeasuredTime[stopwatchEventID] << ","
          << frequency.GetBuffer(frequency.GetLength())
          << endl;
      }
    }
  }
}

void CTimeDiagramDlgBar::OnHScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar)  
{
  averageRange = m_averageSlider.GetPos();
  CString text;
  text.Format("avg: %i", averageRange);
  m_static1.SetWindowText(text.GetBuffer(text.GetLength()));
  RedrawWindow(NULL, NULL, RDW_INVALIDATE);
  
  for (int i=0;i<Stopwatch::numberOfStopwatchEventIDs;i++)
  {
    calculateAverageValues((Stopwatch::StopwatchEventID)i);
  }
  CDynamicBarDlg::OnHScroll(nSBCode, nPos, pScrollBar);
}

/*
* Change log :
* 
* $Log: TimeDiagramDlgBar.cpp,v $
* Revision 1.2  2003/11/07 11:19:56  juengel
* TimeDiagramDlgBar is smaller now.
*
* Revision 1.1  2003/10/07 10:09:38  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/09/26 11:40:12  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.8  2003/06/24 08:13:12  dueffert
* average run time is now %2.1f
*
* Revision 1.7  2003/06/22 12:55:24  roefer
* Stopwatch RT compatible (2nd try)
*
* Revision 1.6  2003/06/22 09:55:12  roefer
* Stopwatch now RT compatible
*
* Revision 1.5  2003/05/11 23:46:35  dueffert
* Depend now works with RobotControl too
*
* Revision 1.4  2003/03/05 17:09:28  loetzsch
* redesign of the queues and debug key tables in RobotControl
*
* Revision 1.3  2002/10/01 11:14:34  loetzsch
* Redesigned DebugKey and DebugKeyTable
*
* Revision 1.2  2002/09/22 18:40:52  risler
* added new math functions, removed GTMath library
*
* Revision 1.1  2002/09/10 15:49:05  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
*
* Revision 1.3  2002/09/07 13:36:55  loetzsch
* unified the vision modules into one module "ImageProcessor"
* - FloodFillRLEImageProcessor, BallPerceptor, LandmarksPerceptor
*   and PlayersPerceptor were are combined to the new solution
*   "BlobImageProcessor"
* - The GridImageProcessor and the SubPixelGradientCalculator became
*   a solution of "ImageProcessor"
*
* Revision 1.2  2002/07/23 13:43:37  loetzsch
* - new streaming classes
* - removed many #include statements
* - 5 instead of 3 debug queues in RobotControl
* - exchanged StaticQueue with MessageQueue
* - new debug message handling
* - empty constructors in bars / dialogs
* - access to debugkeytables and queues via RobotControlQueues.h and RobotControlDebugKeyTables.h
* - general clean up
*
* Revision 1.1.1.1  2002/05/10 12:40:21  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.18  2002/04/25 14:50:34  kallnik
* changed double/float to double
* added several #include GTMath
*
* PLEASE use double
*
* Revision 1.17  2002/04/23 17:45:15  loetzsch
* - splitted debugKeyTable into debugKeyTableForRobot and debugKeyTableForLocalProcesses
* - removed Modules Toolbar
* - removed access for dialogs and toolbars to solutionRequest.
* - changed access for dialogs and toolbars to debug queues
* - removed the instance of SolutionRequest in CRobotControlApp
* - only the log player, local processes and the robot put messages into queueToRobotControl
*
* Revision 1.16  2002/04/20 15:53:36  roefer
* Calculation of call frequency corrected
*
* Revision 1.15  2002/03/16 13:44:49  juengel
* GridImageProcessor created.
*
* Revision 1.14  2002/02/27 19:52:49  loetzsch
* TimeDiagram dialog continued
*
* Revision 1.13  2002/02/26 00:00:56  loetzsch
* Dies & Das
*
* Revision 1.12  2002/02/25 20:19:07  loetzsch
* CSV output
*
* Revision 1.11  2002/02/25 20:01:18  loetzsch
* CSV output
*
* Revision 1.10  2002/02/25 19:27:49  juengel
* no message
*
* Revision 1.9  2002/02/25 19:14:36  loetzsch
* writeAsCSV started
*
* Revision 1.8  2002/02/25 15:20:46  loetzsch
* changed keys
*
* Revision 1.7  2002/02/24 10:02:21  juengel
* TimeDiagramDlgBar completed.
*
* Revision 1.6  2002/01/26 18:09:20  juengel
* DebugDrawingManager umstrukturiert.
*
* Revision 1.5  2002/01/22 22:41:58  loetzsch
* OUTPUT for the measured time removed
*
* Revision 1.4  2002/01/19 08:07:29  juengel
* stopwatch weiterentwickelt
*
* Revision 1.3  2002/01/16 16:34:10  juengel
* stopwatch eingebaut
*
*
*/

