/**
* @file RobotControlDialogBar.h
*
* Definition of class CRobotControlDialogBar
*
* @author Martin Ltzsch
*/

#ifndef __RobotControlDialogBar_h_
#define __RobotControlDialogBar_h_

#include "resource.h"
#include "StdAfx.h"
#include "MfcTools/DockingControlBars/DynamicBar.h"
#include "Tools/MessageQueue/InMessage.h"
#include "Visualization/DebugDrawingManager/DebugDrawingManagerEnums.h"
#include "Representations/Perception/CameraInfo.h"

/**
* @class CRobotControlDialogBar
*
* The base class of all dialog bars in RobotControl.
*
* @author Martin Ltzsch.
*/
class CRobotControlDialogBar : public CDynamicBarDlg, public MessageHandler
{
public:
  /**
  * Constructor.
  * @param dialogID The id of the dialog (from resource.h)
  */
  CRobotControlDialogBar(UINT dialogID) : CDynamicBarDlg(dialogID) {};

  /**
  * Is called for incoming debug messages.
  * @param message The message to handle.
  * @return If the message was handled.
  */
  virtual bool handleMessage(InMessage& message)  { return false; }


  /** 
  * Handles control notifications which arrived in the main frame 
  * @param command The id of the control, menu, accelerator etc.
  * @return If the command was handled.
  */
  virtual bool handleCommand(UINT command) { return 0; }

  /** 
  * Enables the controls in the dialog bar. 
  * This function is called from the main window for all controls. 
  * @param pCmdUI An interface to the control that allows enabling/disabling, checking etc.
  */
  virtual void updateUI(CCmdUI* pCmdUI) {};

  /** Is called when the application is in idle state  */
  virtual void OnIdle() {};

  /** 
  * Is called when a configuration is saved. Can be used by the dialogs and 
  * bars to save configuration dependand settings.
  * @param sectionName The registry section for the current configuration
  */
  virtual void OnConfigurationSave(CString sectionName) {};

  /** 
  * Is called when a configuration is loaded. Can be used by the dialogs and 
  * bars to load configuration dependand settings.
  * @param sectionName The registry section for the current configuration
  */
  virtual void OnConfigurationLoad(CString sectionName) {};

protected:
  /**
  * Registers a CWnd that hosts a certain drawing paper at the debug drawing manager.
  * @param fieldDrawingPaper The field drawing paper.
  * @param pWnd A pointer to the window.
  */
  void registerDrawingPaper(DrawingPapers::FieldDrawingPaper fieldDrawingPaper, CWnd* pWnd);

  /**
  * Registers a CWnd that hosts a certain drawing paper at the debug drawing manager.
  * @param imageDrawingPaper The image drawing paper.
  * @param pWnd A pointer to the window.
  */
  void registerDrawingPaper(DrawingPapers::ImageDrawingPaper imageDrawingPaper, CWnd* pWnd);

  /**
  * Calls the paint method of the DebugDrawingManager for a certain drawing paper
  * @param pDC The DC to draw to.
  * @param rect Specifies the rect in the device context to paint to.
  * @param fieldDrawingPaper Specifies the drawing paper to paint.
  */
  void paintDrawingPaper(CDC* pDC, CRect* rect, DrawingPapers::FieldDrawingPaper fieldDrawingPaper);

  /**
  * Calls the paint method of the DebugDrawingManager for a certain drawing paper
  * @param rect Specifies the rect in the device context to paint to.
  * @param imageDrawingPaper Specifies the drawing paper to paint.
  */
  void paintDrawingPaper(CRect* rect, DrawingPapers::ImageDrawingPaper imageDrawingPaper);
  
  /**
  * Returns the cameraInfo of the first image in the debugDrawingManager
  */
  CameraInfo getDebugImageCameraInfo();
};


/** Creates a dialog bar inside the main frame */
#define CREATE_DIALOG_BAR(classname,title,idd) \
  splashScreen.setMessage(CString("Creating the ") + CString(title) + CString(" dialog bar ..."));\
  CDynamicBarT<classname>* p##classname = new CDynamicBarT<classname>(); \
  p##classname->Create(_T(title),this,CSize(50,50),TRUE,idd,WS_CHILD|CBRS_TOP); \
  p##classname->SetBarStyle(p##classname->GetBarStyle() | CBRS_TOOLTIPS | CBRS_FLYBY | CBRS_SIZE_DYNAMIC); \
  p##classname->EnableDocking(CBRS_ALIGN_ANY);\
  DockControlBar(p##classname, AFX_IDW_DOCKBAR_LEFT);\
  dialogBarMap[idd]=(&p##classname->m_wndDlg);\
  dialogBarArray.Add(&p##classname->m_wndDlg);\
  dynamicBarArray.Add(reinterpret_cast<CDynamicBarT<CRobotControlDialogBar>*>(p##classname));



#endif //__RobotControlDialogBar_h_

/*
* Change Log:
*
* $Log: RobotControlDialogBar.h,v $
* Revision 1.8  2004/02/25 13:41:28  jhoffman
* - removed include of "resource.h" in "stdafx.h" to force compile when "resource.h" is changed
* - layout change should work for everybody now
* - cancel works for "new" layout dialog
*
* Revision 1.7  2003/12/19 17:35:30  loetzsch
* fixed a doxygen bug
*
* Revision 1.6  2003/12/15 13:05:04  loetzsch
* RobotControl now displays a splash screen at the beginning
*
* Revision 1.5  2003/12/15 11:35:05  juengel
* Added method getDebugImageCameraInfo.
*
* Revision 1.4  2003/12/09 18:17:01  loetzsch
* changed macro CREATE_DIALOG
*
* Revision 1.3  2003/12/06 06:31:19  loetzsch
* no message
*
* Revision 1.2  2003/11/30 01:53:19  loetzsch
* prepared RobotControl port to Visual C++ .Net
*
* Revision 1.1  2003/10/07 10:09:37  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.5  2003/05/11 23:51:02  dueffert
* Depend now works with RobotControl too
*
* Revision 1.4  2002/12/18 16:22:56  dueffert
* doxygen docu corrected
*
* Revision 1.3  2002/11/26 12:21:22  dueffert
* doxygen bug fixed
*
* Revision 1.2  2002/09/20 23:28:33  juengel
* Moved instance of DebugDrawingManager to RobotControlMainFrame.
*
* Revision 1.1  2002/09/10 15:49:04  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
*
* Revision 1.3  2002/09/01 17:20:34  loetzsch
* tidied up the MfcTools/ directory. Created directories
* MfcTools/DockingControlBars and MfcTools/IEStyleToolbars.
*
* Revision 1.2  2002/08/29 15:03:47  dueffert
* documentation bug fixed
*
* Revision 1.1  2002/07/23 13:45:14  loetzsch
* - new streaming classes
* - removed many #include statements
* - 5 instead of 3 debug queues in RobotControl
* - new debug message handling
* - access to debugkeytables and queues via RobotControlQueues.h and RobotControlDebugKeyTables.h
* - general clean up
*
*/
