/**
* @file RobotControlPhysicalRobots.cpp 
*
* Implementation of class CRobotControlPhysicalRobots.
*
* @author Uwe Dffert
* @author Martin Ltzsch
*/

#include "StdAfx.h"
#include "RobotControlPhysicalRobots.h"
#include "RobotControl.h"
#include "Tools/Debugging/QueueFillRequest.h"

#include <stdlib.h>

CRobotControlWLanThread::CRobotControlWLanThread(int number):myNumber(number)
{
  tcpConnection = 0;
  wlanNeedsToBeConnected = false;
  start(this,communicate);
}

void CRobotControlWLanThread::communicate()
{
  while(isRunning())
  {
    if(wlanNeedsToBeConnected)
    {
      if(tcpConnection)
      delete tcpConnection;
      char strRemoteIP[32];
      sprintf(strRemoteIP, "%i.%i.%i.%i", remoteIP >> 24,
                                      remoteIP >> 16 & 255,
                                      remoteIP >> 8 & 255,
                                      remoteIP & 255);
      tcpConnection = new TcpConnection(strRemoteIP, remotePort);

      SYNC_WITH(getRobotControlApp());
      if(wlanIsConnected())
      {
         getQueues().toGUI.out.text << "CRobotControlWLanThread : connected to " << strRemoteIP << ":" << remotePort;
      }
      else
      {
         getQueues().toGUI.out.text << "CRobotControlWLanThread : couldn't connect to " << strRemoteIP << ":" << remotePort;
      }
      getQueues().toGUI.out.finishMessage(idText);
      wlanNeedsToBeConnected = false;
    }

    if(wlanIsConnected())
    {
      char* sendData,
          * receivedData;
      int sendSize = 0,
          receivedSize = 0;

      sendData = 0;

      // If there is something to send, prepare a package
      {
        SYNC;
        if(!queueToRobot.isEmpty())
        {
          OutBinarySize size;
          size << queueToRobot;
          sendSize = size.getSize();
          sendData = new char[sendSize];
          OutBinaryMemory stream(sendData);
          stream << queueToRobot;
        }
      }

      // exchange data with the router
      if(tcpConnection->sendAndReceive(sendData,sendSize,receivedData,receivedSize))
        queueToRobot.clear();

      // If a package was prepared, remove it
      if(sendSize)
        delete [] sendData;

      // If a package was received from the router program, add it to receiver queue
      if(receivedSize > 0)
      {
        InBinaryMemory stream(receivedData,receivedSize);
        stream >> queueFromRobot;
        delete [] receivedData;

        // move all messages to the queue from physical robots
      
        SYNC_WITH(getRobotControlApp());
        queueFromRobot.moveAllMessages(getQueues().fromPhysicalRobots);
      
      }
    }
    Sleep(10);
  }
}

/** Returns whether WLAN connection is established */
bool CRobotControlWLanThread::wlanIsConnected()
{
  if(tcpConnection && !tcpConnection->isConnected())
  {
    delete tcpConnection;
    tcpConnection = 0;
  }
  return tcpConnection != 0;
}

/** This function establishes the connection over the WLan interface and must be called before using it, of course **/
void CRobotControlWLanThread::wlanConnect(const unsigned long theRemoteIP, int theRemotePort)
{
  SYNC;
  remoteIP = theRemoteIP;
  remotePort = theRemotePort;
  wlanNeedsToBeConnected = true;
    char strRemoteIP[32];
      sprintf(strRemoteIP, "%i.%i.%i.%i", remoteIP >> 24,
                                      remoteIP >> 16 & 255,
                                      remoteIP >> 8 & 255,
                                      remoteIP & 255);
  getQueues().toGUI.out.text << "CRobotControlWLanThread : trying to connect to " << strRemoteIP << ":" << remotePort;
  getQueues().toGUI.out.finishMessage(idText);
}

/** This function terminates an WLan connection **/
void CRobotControlWLanThread::wlanDisconnect()
{
  {
    SYNC;
    if(wlanIsConnected())
    {
      /** send empty DebugKeyTable and immediateReadWrite to prevent useless full queues on robot
      * @todo this might be not usefull when in QueueFillRequest::overwriteOlder mode
      */
      QueueFillRequest queueFillRequest;
      queueToRobot.out.bin << queueFillRequest;
      queueToRobot.out.finishMessage(idQueueFillRequest);
      DebugKeyTable debugKeyTable;
      queueToRobot.out.bin << debugKeyTable;
      queueToRobot.out.finishMessage(idDebugKeyTable);
    }  
  }
  Sleep(800);
  SYNC;
  if(wlanIsConnected())
  {
    delete tcpConnection;
    tcpConnection = 0;
    getQueues().toGUI.out.text << "CRobotControlWLanThread : disconnected";
    getQueues().toGUI.out.finishMessage(idText);
  }
}

CRobotControlWLanThread::~CRobotControlWLanThread()
{
}




CRobotControlPhysicalRobots::CRobotControlPhysicalRobots(): selectedRobot(-1)
{
  for (int i=0;i<4;i++)
  {
    wLanThreadRed[i]=0;
    wLanThreadBlue[i]=0;
  }
}

CRobotControlPhysicalRobots::~CRobotControlPhysicalRobots()
{
  disconnect();
}

void CRobotControlPhysicalRobots::setSelectedRobot(int robot)
{
  selectedRobot=robot;
}

int CRobotControlPhysicalRobots::getSelectedRobot() const
{
  return selectedRobot;
}

bool CRobotControlPhysicalRobots::isConnected(int robot) const
{
  if (robot<4)
  {
    return (wLanThreadRed[robot] && wLanThreadRed[robot]->wlanIsConnected());
  }
  else
  {
    return (wLanThreadBlue[robot-4] && wLanThreadBlue[robot-4]->wlanIsConnected());
  }
}

bool CRobotControlPhysicalRobots::isSomeoneConnected()
{
  int first=-1;
  int selectedRob=(selectedRobot==-1)?0:selectedRobot;
  for (int i=0;i<8;i++)
  {
    if (isConnected((selectedRob+i)%8))
    {
      first=((selectedRob+i)%8);
      break;
    }
  }
  if (selectedRobot!=first)
  {
    selectedRobot=first;
  }
  return (first!=-1);
}

void CRobotControlPhysicalRobots::connect(CRobotControlWLANConfiguration& config)
{
  for (int i=0;i<4;i++)
  {
    if (config.useRed[i] && (wLanThreadRed[i]==0))
    {
      wLanThreadRed[i]=new CRobotControlWLanThread(i);
    }
    else if (!config.useRed[i] && (wLanThreadRed[i]!=0))
    {
      delete(wLanThreadRed[i]);
      wLanThreadRed[i]=0;
    }
    if (config.useBlue[i] && (wLanThreadBlue[i]==0))
    {
      wLanThreadBlue[i]=new CRobotControlWLanThread(i+4);
    }
    else if (!config.useBlue[i] && (wLanThreadBlue[i]!=0))
    {
      delete(wLanThreadBlue[i]);
      wLanThreadBlue[i]=0;
    }
  }
  
  if (config.autostartRed)
  {
    //2do: richtig machen
    //phys->start("141.20.26.100");
    //_sleep(2000);
  }
  else if (config.autostartBlue)
  {
    //2do: richtig machen
    //phys->start("141.20.26 111 112 113");
    //_sleep(2000);
  }
  
  for (i=0;i<4;i++)
  {
    if (wLanThreadRed[i] && !wLanThreadRed[i]->wlanIsConnected())
    {
      wLanThreadRed[i]->wlanConnect(config.ipRed[i],0xA1BD);
    }
    if (wLanThreadBlue[i] && !wLanThreadBlue[i]->wlanIsConnected())
    {
      wLanThreadBlue[i]->wlanConnect(config.ipBlue[i],0xA1BD);
    }
  }
}

void CRobotControlPhysicalRobots::disconnect()
{
  for (int i=0;i<4;i++)
  {
    if (wLanThreadRed[i] && wLanThreadRed[i]->wlanIsConnected())
    {
      wLanThreadRed[i]->wlanDisconnect();
    }
    if (wLanThreadBlue[i] && wLanThreadBlue[i]->wlanIsConnected())
    {
      wLanThreadBlue[i]->wlanDisconnect();
    }
  }
  
  for (i=0;i<4;i++)
  {
    if (wLanThreadRed[i])
    {
      delete(wLanThreadRed[i]);
      wLanThreadRed[i]=0;
    }
    if (wLanThreadBlue[i])
    {
      delete(wLanThreadBlue[i]);
      wLanThreadBlue[i]=0;
    }
  }
  
  /** @todo stop the RouterCtrl if it runs */
  //stop();
}

void CRobotControlPhysicalRobots::onIdle()
{
  SYNC_WITH(getRobotControlApp()); //without that people can modify queues.toAllPhysicalRobots during sending, so different robots get different subsets of messages, but all are deleted afterwards...
 
  for (int i=0;i<8;i++)
  {
    CRobotControlWLanThread* wlanThread=i<4?wLanThreadRed[i]:wLanThreadBlue[i-4];
 
    if (wlanThread)
    {
      SYNC_WITH(*wlanThread);
    
      if (wlanThread->isRunning())
      {
        if (wlanThread->wlanIsConnected())
        {
          getQueues().toPhysical.allRobots.copyAllMessages(wlanThread->queueToRobot);
          getQueues().toPhysical.robot[i].moveAllMessages(wlanThread->queueToRobot);
          if (getSelectedRobot() == i)
          {
            getQueues().toPhysical.selectedRobot.moveAllMessages(wlanThread->queueToRobot);
          }
        }
        else
        {
          getQueues().toPhysical.robot[i].moveAllMessages(getQueues().toPhysical.selectedRobot);
          getQueues().toPhysical.allRobots.copyAllMessages(getQueues().toPhysical.selectedRobot);
        }
      }
      else
      {
        getQueues().toPhysical.robot[i].moveAllMessages(getQueues().toPhysical.selectedRobot);
        getQueues().toPhysical.allRobots.copyAllMessages(getQueues().toPhysical.selectedRobot);
      }
    }
  }
  getQueues().toPhysical.allRobots.clear();
  
}

void CRobotControlPhysicalRobots::onLineReceived(const char* line) {
  getQueues().toGUI.out.text << line;
  getQueues().toGUI.out.finishMessage(idText);
}


CRobotControlPhysicalRobots& getPhysicalRobots()
{
  return getRobotControlApp().physicalRobots;
}

CRobotControlWLANConfiguration::CRobotControlWLANConfiguration()
{
  readFromProfile(99);
}

void CRobotControlWLANConfiguration::readFromProfile(int index)
{
  char entry[256];
  char test[256];
  sprintf(entry,"Name%d",index);
  strcpy(test,AfxGetApp()->GetProfileString("WLAN",entry,"there_is_no_old_config"));
  if (strcmp(test,"there_is_no_old_config")!=0)
  {
    HKEY rc;
    strcpy(name,test);
    RegOpenKey(HKEY_CURRENT_USER,"Software\\GermanTeam\\RobotControl\\WLAN",&rc);
    RegDeleteValue(rc,entry);
    
    sprintf(entry,"IP%d",index);
    strcpy(test,AfxGetApp()->GetProfileString("WLAN",entry,"127.0.0.1").GetBuffer(255));
    ipRedRouter=ntohl(inet_addr(test));
    ipBlueRouter=ipRedRouter;
    RegDeleteValue(rc,entry);
    
    sprintf(entry,"IPRobot%d",index);
    strcpy(test,AfxGetApp()->GetProfileString("WLAN",entry,"192.168.18.1").GetBuffer(255));
    ipRed[0]=ntohl(inet_addr(test));
    ipRed[1]=0;
    ipRed[2]=0;
    ipRed[3]=0;
    ipBlue[0]=0;
    ipBlue[1]=0;
    ipBlue[2]=0;
    ipBlue[3]=0;
    useRed[0]=true;
    useRed[1]=false;
    useRed[2]=false;
    useRed[3]=false;
    useBlue[0]=false;
    useBlue[1]=false;
    useBlue[2]=false;
    useBlue[3]=false;
    RegDeleteValue(rc,entry);
    
    sprintf(entry,"PortTo%d",index);
    portRedRouter = AfxGetApp()->GetProfileInt("WLAN",entry,15043);
    portBlueRouter = portRedRouter;
    RegDeleteValue(rc,entry);
    
    sprintf(entry,"Channel%d",index);
    channelRed = AfxGetApp()->GetProfileInt("WLAN",entry,3);
    channelBlue = channelRed;
    RegDeleteValue(rc,entry);
    
    sprintf(entry,"APMode%d",index);
    apmodeRed = AfxGetApp()->GetProfileInt("WLAN",entry,2);
    apmodeBlue = apmodeRed;
    RegDeleteValue(rc,entry);
    
    sprintf(entry,"WepKey%d",index);
    strcpy(wepkeyRed,AfxGetApp()->GetProfileString("WLAN",entry,""));
    strcpy(wepkeyBlue,wepkeyRed);
    RegDeleteValue(rc,entry);
    
    sprintf(entry,"NetMask%d",index);
    strcpy(test,AfxGetApp()->GetProfileString("WLAN",entry,"255.255.0.0").GetBuffer(255));
    netmaskRed=ntohl(inet_addr(test));
    netmaskBlue = netmaskRed;
    RegDeleteValue(rc,entry);
    
    sprintf(entry,"ESSID%d",index);
    strcpy(essidRed ,AfxGetApp()->GetProfileString("WLAN",entry,"SONY_LEGGED_A"));
    strcpy(essidBlue,essidRed);
    RegDeleteValue(rc,entry);
    
    sprintf(entry,"AutoIP%d",index); RegDeleteValue(rc,entry);
    sprintf(entry,"ChangeStick%d",index); RegDeleteValue(rc,entry);
    sprintf(entry,"UseRouter%d",index); RegDeleteValue(rc,entry);
    sprintf(entry,"PortFrom%d",index); RegDeleteValue(rc,entry);

    writeToProfile(index);
  }
  else
  {
    char path[256];
    sprintf(path,"WLAN\\Configuration%02i",index);
    strcpy(name,AfxGetApp()->GetProfileString(path,"Name","unnamed"));
    ipRedRouter=AfxGetApp()->GetProfileInt(path,"IPRedRouter",0x7f000001);
    ipBlueRouter=AfxGetApp()->GetProfileInt(path,"IPBlueRouter",0x7f000001);
    portRedRouter=AfxGetApp()->GetProfileInt(path,"PortRedRouter",15000);
    portBlueRouter=AfxGetApp()->GetProfileInt(path,"PortBlueRouter",15000);
    autostartRed=(AfxGetApp()->GetProfileInt(path,"AutostartRed",0)==1);
    autostartBlue=(AfxGetApp()->GetProfileInt(path,"AutostartBlue",0)==1);
    ipRed[0]=AfxGetApp()->GetProfileInt(path,"IPRed1",0x0a000064);
    ipRed[1]=AfxGetApp()->GetProfileInt(path,"IPRed2",0x0a000065);
    ipRed[2]=AfxGetApp()->GetProfileInt(path,"IPRed3",0x0a000066);
    ipRed[3]=AfxGetApp()->GetProfileInt(path,"IPRed4",0x0a000067);
    ipBlue[0]=AfxGetApp()->GetProfileInt(path,"IPBlue1",0x0a00006e);
    ipBlue[1]=AfxGetApp()->GetProfileInt(path,"IPBlue2",0x0a00006f);
    ipBlue[2]=AfxGetApp()->GetProfileInt(path,"IPBlue3",0x0a000070);
    ipBlue[3]=AfxGetApp()->GetProfileInt(path,"IPBlue4",0x0a000071);
    useRed[0]=(AfxGetApp()->GetProfileInt(path,"UseRed1",0)==1);
    useRed[1]=(AfxGetApp()->GetProfileInt(path,"UseRed2",0)==1);
    useRed[2]=(AfxGetApp()->GetProfileInt(path,"UseRed3",0)==1);
    useRed[3]=(AfxGetApp()->GetProfileInt(path,"UseRed4",0)==1);
    useBlue[0]=(AfxGetApp()->GetProfileInt(path,"UseBlue1",0)==1);
    useBlue[1]=(AfxGetApp()->GetProfileInt(path,"UseBlue2",0)==1);
    useBlue[2]=(AfxGetApp()->GetProfileInt(path,"UseBlue3",0)==1);
    useBlue[3]=(AfxGetApp()->GetProfileInt(path,"UseBlue4",0)==1);
    strcpy(essidRed,AfxGetApp()->GetProfileString(path,"ESSIDRed","SONY_LEGGED_A"));
    strcpy(essidBlue,AfxGetApp()->GetProfileString(path,"ESSIDBlue","SONY_LEGGED_B"));
    netmaskRed=AfxGetApp()->GetProfileInt(path,"NetMaskRed",0xffffff00);
    netmaskBlue=AfxGetApp()->GetProfileInt(path,"NetMaskBlue",0xffffff00);
    apmodeRed=AfxGetApp()->GetProfileInt(path,"APModeRed",2);
    apmodeBlue=AfxGetApp()->GetProfileInt(path,"APModeBlue",2);
    channelRed=AfxGetApp()->GetProfileInt(path,"ChannelRed",3);
    channelBlue=AfxGetApp()->GetProfileInt(path,"ChannelBlue",3);
    strcpy(wepkeyRed,AfxGetApp()->GetProfileString(path,"WEPKeyRed",""));
    strcpy(wepkeyBlue,AfxGetApp()->GetProfileString(path,"WEPKeyBlue",""));
    strcpy(teamIdentifierRed,AfxGetApp()->GetProfileString(path,"TeamIdentifierRed","undefRed"));
    strcpy(teamIdentifierBlue,AfxGetApp()->GetProfileString(path,"TeamIdentifierBlue","undefBlue"));
  }
}

void CRobotControlWLANConfiguration::writeToProfile(int index)
{
  char path[256];
  sprintf(path,"WLAN\\Configuration%02i",index);
  AfxGetApp()->WriteProfileString(path,"Name",name);
  AfxGetApp()->WriteProfileInt(path,"IPRedRouter",ipRedRouter);
  AfxGetApp()->WriteProfileInt(path,"IPBlueRouter",ipBlueRouter);
  AfxGetApp()->WriteProfileInt(path,"PortRedRouter",portRedRouter);
  AfxGetApp()->WriteProfileInt(path,"PortBlueRouter",portBlueRouter);
  AfxGetApp()->WriteProfileInt(path,"AutostartRed", autostartRed?1:0);
  AfxGetApp()->WriteProfileInt(path,"AutostartBlue", autostartBlue?1:0);
  AfxGetApp()->WriteProfileInt(path,"IPRed1",ipRed[0]);
  AfxGetApp()->WriteProfileInt(path,"IPRed2",ipRed[1]);
  AfxGetApp()->WriteProfileInt(path,"IPRed3",ipRed[2]);
  AfxGetApp()->WriteProfileInt(path,"IPRed4",ipRed[3]);
  AfxGetApp()->WriteProfileInt(path,"IPBlue1",ipBlue[0]);
  AfxGetApp()->WriteProfileInt(path,"IPBlue2",ipBlue[1]);
  AfxGetApp()->WriteProfileInt(path,"IPBlue3",ipBlue[2]);
  AfxGetApp()->WriteProfileInt(path,"IPBlue4",ipBlue[3]);
  AfxGetApp()->WriteProfileInt(path,"UseRed1",useRed[0]?1:0);
  AfxGetApp()->WriteProfileInt(path,"UseRed2",useRed[1]?1:0);
  AfxGetApp()->WriteProfileInt(path,"UseRed3",useRed[2]?1:0);
  AfxGetApp()->WriteProfileInt(path,"UseRed4",useRed[3]?1:0);
  AfxGetApp()->WriteProfileInt(path,"UseBlue1",useBlue[0]?1:0);
  AfxGetApp()->WriteProfileInt(path,"UseBlue2",useBlue[1]?1:0);
  AfxGetApp()->WriteProfileInt(path,"UseBlue3",useBlue[2]?1:0);
  AfxGetApp()->WriteProfileInt(path,"UseBlue4",useBlue[3]?1:0);
  AfxGetApp()->WriteProfileString(path,"ESSIDRed",essidRed);
  AfxGetApp()->WriteProfileString(path,"ESSIDBlue",essidBlue);
  AfxGetApp()->WriteProfileInt(path,"NetMaskRed",netmaskRed);
  AfxGetApp()->WriteProfileInt(path,"NetMaskBlue",netmaskBlue);
  AfxGetApp()->WriteProfileInt(path,"APModeRed",apmodeRed);
  AfxGetApp()->WriteProfileInt(path,"APModeBlue",apmodeBlue);
  AfxGetApp()->WriteProfileInt(path,"ChannelRed",channelRed);
  AfxGetApp()->WriteProfileInt(path,"ChannelBlue",channelBlue);
  AfxGetApp()->WriteProfileString(path,"WEPKeyRed",wepkeyRed);
  AfxGetApp()->WriteProfileString(path,"WEPKeyBlue",wepkeyBlue);
  AfxGetApp()->WriteProfileString(path,"TeamIdentifierRed",teamIdentifierRed);
  AfxGetApp()->WriteProfileString(path,"TeamIdentifierBlue",teamIdentifierBlue);
}

void CRobotControlWLANConfiguration::deleteProfile(int index)
{
  char key[256];
  sprintf(key,"Configuration%02i",index);
  HKEY rc;
  RegOpenKey(HKEY_CURRENT_USER,"Software\\GermanTeam\\RobotControl\\WLAN",&rc);
  RegDeleteKey(rc,key);
  RegCloseKey(rc);
}

/*
* Change Log:
*
* $Log: RobotControlPhysicalRobots.cpp,v $
* Revision 1.10  2004/01/27 23:20:23  wachter
* Added  teamIdentifier to WLANConfigurationDialog
*
* Revision 1.9  2004/01/04 19:06:12  wachter
* Moved connect(..) into WLan-thread.
*
* Revision 1.8  2004/01/03 19:16:29  wachter
* - Debug-communication without router working now
* - Router-IP not longer needed
*
* Revision 1.7  2003/12/15 11:34:14  juengel
* Initialized sendData.
*
* Revision 1.6  2003/12/13 11:32:06  loetzsch
* renamed WLANConfiguration to CRobotControlWLANConfiguration
* and moved to RobotControlPhysicalRobots.h
*
* Revision 1.5  2003/12/09 20:27:52  loetzsch
* sending of messages to specific robots improved
*
* Revision 1.4  2003/12/09 19:49:24  loetzsch
* Renamed some of the main queues of RobotControl.
*
* Added possibility to send messages to specific simulated or physical robots.
*
* Revision 1.3  2003/12/07 18:57:34  loetzsch
* all messages but not only from the selected robot are sent to the "queue from physical robot"
*
* Revision 1.2  2003/12/04 00:52:05  loetzsch
* small beautification
*
* Revision 1.1  2003/10/07 10:09:37  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.3  2003/09/26 11:40:12  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.2  2003/08/17 18:35:37  roefer
* Communication with router standardized and synchronized
*
* Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.6  2003/05/11 23:51:02  dueffert
* Depend now works with RobotControl too
*
* Revision 1.5  2003/04/14 16:15:37  loetzsch
* ATH after GermanOpen CVS merge
* bug fix
*
* Revision 1.2  2003/04/10 22:53:13  dueffert
* blue4 connection bug fixed
*
* Revision 1.1.1.1  2003/04/09 14:22:42  loetzsch
* started Aibo Team Humboldt's GermanOpen CVS
*
* Revision 1.4  2003/03/28 14:04:19  dueffert
* team wlan support improved
*
* Revision 1.3  2003/03/25 12:48:43  loetzsch
* moved the variable wlanHeartbeatInterval from RobotControl.h
* to RobotControlPhysicalRobots.cpp
*
* Revision 1.2  2003/03/25 11:46:53  dueffert
* team wlan support improved
*
* Revision 1.1  2003/03/24 14:06:51  loetzsch
* added class CRobotControlPhysicalRobot
*
*/
