/**
* @file RobotControlPhysicalRobots.h
*
* Definition of class CRobotControlWLanThread and class CRobotControlPhysicalRobots
*
* @author Uwe Dffert
* @author Martin Ltzsch
*/

#ifndef __RobotControlPhysicalRobots_h_
#define __RobotControlPhysicalRobots_h_

#include "Tools/MessageQueue/MessageQueue.h"
#include "Platform/Win32/RouterCtrl.h"
#include "Platform/Win32/Thread.h"
#include "Platform/Win32Linux/TcpConnection.h"
#include "RobotControlQueues.h"

/** 
* @class CRobotControlWLANConfiguration
*
* Stores all information of a (team) wlan configuration 
*
* @author Uwe Dffert
* @author Max Risler
*/
class CRobotControlWLANConfiguration
{
public:
  /** constructor */
  CRobotControlWLANConfiguration();

  /** 
  * Fills this configuration with data read from the Registry
  * @param index number of the WLAN configuration in the Registry to read from
  */
  void readFromProfile(int index);

  /** 
  * Writes the data in this configuration to the Registry
  * @param index number of the WLAN configuration in the Registry to write to
  */
  void writeToProfile(int index);

  /** 
  * deletes a WLAN configuration in the Registry
  * @param index number of the WLAN configuration in the Registry to delete
  */
  void deleteProfile(int index);

  char name[65]; /*!< The name of the configuration */
  unsigned long ipRedRouter; /*!< The ip of the router for the red robots */
	unsigned long ipBlueRouter; /*!< The ip of the router for the blue robots */
	int portRedRouter; /*!< The port of the router for the red robots */
	int portBlueRouter; /*!< The port of the router for the blue robots */
  bool autostartRed; /*!< ??? */
  bool autostartBlue; /*!< ??? */
  unsigned long ipRed[4]; /*!< The ips for the red team */
  unsigned long ipBlue[4]; /*!< The ips for the blue team */
	bool useRed[4]; /*!< Which of the red robots shall be connected */
	bool useBlue[4]; /*!< Which of the red robots shall be connected */
	char essidRed[65]; /*!< The essid of the red team */
	char essidBlue[65]; /*!< The essid of the blue team */
	unsigned long netmaskRed; /*!< The netmask of the red team */
	unsigned long netmaskBlue; /*!< The netmask of the blue team */
	int apmodeRed; /*!< The apmode of the red team */
	int apmodeBlue; /*!< The apmode of the blue team */
	int channelRed; /*!< The WLAN channel of the red team */
	int channelBlue; /*!< The WLAN channel of the blue team */
  char wepkeyRed[17]; /*!< The wepkey of the red team */
  char wepkeyBlue[17]; /*!< The wepkey of the blue team */
  char teamIdentifierBlue[17];
  char teamIdentifierRed[17];
};

/**
* @class CRobotControlWLanThread
*
* A thread that manages the communication between the queues and the Wireless Lan
*
* @author Martin Ltzsch
*/
class CRobotControlWLanThread : public Thread<CRobotControlWLanThread>
{
public:
  DECLARE_SYNC; /**< Make this object synchronizable. */
  
  /** 
  * Constructor 
  * @param number for the CRobotControlWLanThread to know which robot it belongs to
  */
  CRobotControlWLanThread(int number);
  
  /** Destructor */
  ~CRobotControlWLanThread();
  
  /** A queue for messages to be sent via the WLAN to the robot */
  MessageQueue queueToRobot;
  
  /** A queue for messages to be received from the WLAN */
  MessageQueue queueFromRobot;
  
  /** Returns whether WLAN connection is established */
  bool wlanIsConnected();
  
  /** This function establishes the connection over the WLan interface and must be called before using it, of course **/
  void wlanConnect(const unsigned long remoteIP, int remotePort);
  
  /** This function terminates an WLan connection **/
  void wlanDisconnect();
  
private:  
  /** The thread main function */
  void communicate();
  
  /** a TCP/IP client class handling the WLan interface **/
  TcpConnection* tcpConnection;
  unsigned long remoteIP;
  int remotePort;
  bool wlanNeedsToBeConnected;
  
  /** for the thread to know which robot it belongs to */
  int myNumber;
};


/** 
* @class CRobotControlPhysicalRobots
*
* Encapsulates the WLAN interface to the physical robots.
* 
* @author Uwe Dffert
* @author Martin Ltzsch
*/
class CRobotControlPhysicalRobots: public RouterCtrl
{
public:
  /** Default constructor */
  CRobotControlPhysicalRobots();
  
  /** Destructor */
  ~CRobotControlPhysicalRobots();
  
  /** Should be called when the application is idle, used for communication */
  void onIdle();
  
  /** Connects all robots contained in config */
  void connect(CRobotControlWLANConfiguration& config);
  
  /** Disconnects all connected robots */
  void disconnect();
  
  /** Selects a physical robot. If -1, then no robot is selected. */
  void setSelectedRobot(int robot);
  
  /** Returns the selected physical robot. If -1, then no robot is selected. */
  int getSelectedRobot() const;
  
  /** return whether a certain robot is connected via WLAN */
  bool isConnected(int robot) const;
  
  /** return whether at least one robot is connected and select that if no one else is selected */
  bool isSomeoneConnected();
  
protected:
  /** Four instances of the WLan thread for red robots */
  CRobotControlWLanThread* wLanThreadRed[4];
  
  /** Four instances of the WLan thread for blue robots */
  CRobotControlWLanThread* wLanThreadBlue[4];
  
  /**
  * The function is called for each line received from the router.
  * @param line The line of text that has been received.
  */
  virtual void onLineReceived(const char* line); 
  
  /** The robot that is connected to the gui */
  int selectedRobot;
};

/** Returns a reference to the physical robots instance inside RobotControl */
CRobotControlPhysicalRobots& getPhysicalRobots();

#endif //__RobotControlPhysicalRobots_h_

/*
* Change Log:
*
* $Log: RobotControlPhysicalRobots.h,v $
* Revision 1.5  2004/01/27 23:20:23  wachter
* Added  teamIdentifier to WLANConfigurationDialog
*
* Revision 1.4  2004/01/04 19:06:12  wachter
* Moved connect(..) into WLan-thread.
*
* Revision 1.3  2003/12/13 11:32:06  loetzsch
* renamed WLANConfiguration to CRobotControlWLANConfiguration
* and moved to RobotControlPhysicalRobots.h
*
* Revision 1.2  2003/12/04 00:52:05  loetzsch
* small beautification
*
* Revision 1.1  2003/10/07 10:09:37  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/08/17 18:35:37  roefer
* Communication with router standardized and synchronized
*
* Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.3  2003/03/28 14:04:19  dueffert
* team wlan support improved
*
* Revision 1.2  2003/03/25 11:46:53  dueffert
* team wlan support improved
*
* Revision 1.1  2003/03/24 14:06:51  loetzsch
* added class CRobotControlPhysicalRobot
*
*/
