/**
* @file RobotControlSimulatedRobots.cpp 
*
* Implementation of class CRobotControlSimulatedRobots.
*
* @author Martin Ltzsch
* @author Thomas Rfer
*/

#include "StdAfx.h"
#include "RobotControlSimulatedRobots.h"
#include "RobotControl/Bars/SimulatorObjectViewerDlgBar.h"
#include "RobotControl.h"

CRobotControlSimulatedRobots::CRobotControlSimulatedRobots()
: selectedRobot(-1), doRestart(false), sendOracle(false),
simulatedRobotsAreDisabled(false)
{
  for (int i=0; i<8; i++)
  {
    design[i] = RobotDesign::ERS210;
  }
}

CRobotControlSimulatedRobots::~CRobotControlSimulatedRobots()
{
  CString key;
  for (int i=0;i<8;i++)
  {
    key.Format("state%i",i+1);
    AfxGetApp()->WriteProfileInt("SimulatedRobots",key,(int)state[i]);
    key.Format("design%i",i+1);
    AfxGetApp()->WriteProfileInt("SimulatedRobots",key,(int)design[i]);
  }
  AfxGetApp()->WriteProfileInt("SimulatedRobots","sendOracle",sendOracle?1:0);
  AfxGetApp()->WriteProfileInt("SimulatedRobots","selectedRobot",selectedRobot);
  AfxGetApp()->WriteProfileInt("SimulatedRobots","simulatedRobotsAreDisabled",
    simulatedRobotsAreDisabled?1:0);


  if (pSimRobotDocument!=0) delete pSimRobotDocument;
}

void CRobotControlSimulatedRobots::create()
{
  state[0] = (State)AfxGetApp()->GetProfileInt("SimulatedRobots","state1",activeWithImages);
  AfxGetApp()->WriteProfileInt("SimulatedRobots","state1",state[0]);

  CString key;
  for (int i=0;i<8;i++)
  {
    key.Format("state%i",i+1);
    state[i] = (State)AfxGetApp()->GetProfileInt("SimulatedRobots",key,disabled);
    key.Format("design%i",i+1);
    design[i] = (RobotDesign::Design)AfxGetApp()->GetProfileInt("SimulatedRobots",key,RobotDesign::ERS7);
  }
  sendOracle = (AfxGetApp()->GetProfileInt("SimulatedRobots","sendOracle",0) == 1);
  selectedRobot = AfxGetApp()->GetProfileInt("SimulatedRobots","selectedRobot",0);
  simulatedRobotsAreDisabled 
    = (AfxGetApp()->GetProfileInt("SimulatedRobots","simulatedRobotsAreDisabled",0) == 1);

  pSimRobotDocument = new CSimRobotDocument();
  char path[512];
  sprintf(path,"%s/Config/Scenes/RobotControl.scn",File::getGTDir());
  pSimRobotDocument->SetFilePath(path);
  pSimRobotDocument->OnSimReset();
}

void CRobotControlSimulatedRobots::onIdle()
{
  if (!doRestart) // do nothing when the robot configuration changed
  {
    pSimRobotDocument->OnIdle();
    
    for (int i=0;i<8;i++)
    {
      if (state[i] == activeWithImages || state[i] == activeWithoutImages)
      {
        bool newMessage = false;

        if (!getQueues().toSimulated.allRobots.isEmpty()) 
          newMessage = true;
        else if (selectedRobot == i && !getQueues().toSimulated.selectedRobot.isEmpty()) 
          newMessage = true;
        else if (!getQueues().toSimulated.robot[i].isEmpty()) 
          newMessage = true;

        if (newMessage)
        {
          SYNC_WITH(*robots[i]);
          getQueues().toSimulated.allRobots.copyAllMessages(robots[i]->theDebugSender);
          getQueues().toSimulated.selectedRobot.copyAllMessages(robots[selectedRobot]->theDebugSender);
          getQueues().toSimulated.robot[i].copyAllMessages(robots[i]->theDebugSender);
          robots[i]->theDebugSender.send();
        }
      }
      getQueues().toSimulated.robot[i].clear();
    }
  }
  getQueues().toSimulated.allRobots.clear();
  getQueues().toSimulated.selectedRobot.clear();
}

void CRobotControlSimulatedRobots::OnSimReset()
{
  pSimRobotDocument->OnSimReset();
  doRestart = false;
  pSimRobotDocument->pSimulatorObjectViewerDlgBar->Repaint();
  getDebugKeyTables().sendForSimulatedRobots();
}

void CRobotControlSimulatedRobots::OnSimStart()
{
  pSimRobotDocument->OnSimStart();
}

void CRobotControlSimulatedRobots::OnSimStep()
{
  pSimRobotDocument->OnSimStep();
}

void CRobotControlSimulatedRobots::setSelectedRobot(int robot)
{
  if (selectedRobot != robot)
  {
    if (robot != -1)
    {
      ASSERT(state[robot] == activeWithImages || state[robot] == activeWithoutImages);
    }
    selectedRobot = robot;
  }
}

int CRobotControlSimulatedRobots::getSelectedRobot() const
{
  return selectedRobot;
}

void CRobotControlSimulatedRobots::setStateAndDesign(int robot,State newState, 
                                                     RobotDesign::Design newDesign)
{
  switch (newState)
  {
  case disabled:
    if (state[robot]!=disabled)
    {
      stop();
      state[robot] = disabled;
      if (selectedRobot == robot) 
      {
        selectFirstActiveRobot();
      }
    }
    break;
  case passive:
    if (state[robot]!=passive || design[robot] != newDesign)
    {
      stop();
      state[robot] = passive;
      design[robot] = newDesign;
      if (selectedRobot == robot) 
      {
        selectFirstActiveRobot();
      }
    }
    break;
  case activeWithImages:
  case activeWithoutImages:
    if (state[robot]==disabled || state[robot]==passive || design[robot] != newDesign)
    {
      stop();
    }
    state[robot] = newState;
    design[robot] = newDesign;

    if (selectedRobot == -1) 
    {
      selectedRobot = robot;
    }
    break;
  }
}

CRobotControlSimulatedRobots::State CRobotControlSimulatedRobots::getState(int robot) const
{
  return state[robot];
}

void CRobotControlSimulatedRobots::selectFirstActiveRobot()
{
  selectedRobot = -1;
  for (int i=0;i<8;i++)
  {
    if (state[i]==activeWithImages || state[i]==activeWithoutImages)
    {
      selectedRobot = i;
      break;
    }
  }
}

void CRobotControlSimulatedRobots::stop()
{
  doRestart = true;
  if (pSimRobotDocument->IsRunning())
  {
    pSimRobotDocument->OnSimStart();
  }
}

bool CRobotControlSimulatedRobots::getDoRestart() const
{
  return doRestart;
}

void CRobotControlSimulatedRobots::setSendOracle(bool doSend)
{
  sendOracle = doSend;
}

bool CRobotControlSimulatedRobots::getSendOracle() const
{
  return sendOracle;
}

CSimRobotDocument* CRobotControlSimulatedRobots::getSimRobotDocument() const
{
  return pSimRobotDocument;
}

bool CRobotControlSimulatedRobots::getSimulatedRobotsAreDisabled() const
{
  return simulatedRobotsAreDisabled;
}

void CRobotControlSimulatedRobots::setSimulatedRobotsAreDisabled(bool disabled)
{
  simulatedRobotsAreDisabled = disabled;
}

RobotDesign::Design CRobotControlSimulatedRobots::getDesign(int robot) const
{
  return design[robot];
}

CRobotControlSimulatedRobots& getSimulatedRobots()
{
  return getRobotControlApp().simulatedRobots;
}

/*
* Change Log:
*
* $Log: RobotControlSimulatedRobots.cpp,v $
* Revision 1.7  2004/01/02 16:03:34  loetzsch
* added support for different robot designs to CRobotControlSimulatedRobots
* and to the tool bar
*
* Revision 1.6  2003/12/13 11:32:06  loetzsch
* renamed WLANConfiguration to CRobotControlWLANConfiguration
* and moved to RobotControlPhysicalRobots.h
*
* Revision 1.5  2003/12/12 17:12:23  loetzsch
* With the button "Disable simulated robots" all simulated robots can be
* disabled. But this does not deactivate any simulated robots.
*
* If the simulated robots are disabled, messages from physical robots or a
* log file are directly sent to the queue "toGUI". (One does not need to
* deactivate all simulated robots anymore if one wants to view images from a log file
* without sending them to the selected simulated robot.)
*
* Revision 1.4  2003/12/09 20:27:52  loetzsch
* sending of messages to specific robots improved
*
* Revision 1.3  2003/12/09 19:49:24  loetzsch
* Renamed some of the main queues of RobotControl.
*
* Added possibility to send messages to specific simulated or physical robots.
*
* Revision 1.2  2003/12/06 06:31:19  loetzsch
* no message
*
* Revision 1.1  2003/10/07 10:09:37  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.8  2003/05/11 23:51:03  dueffert
* Depend now works with RobotControl too
*
* Revision 1.7  2003/03/23 19:14:21  loetzsch
* finished GUI for 8 simulated robots
* - added the passive state
* - added the state[8] variable to CRobotControlSimulatedRobots
* - Repaint of object viewer after reset
* - Resend debug key table after reset
* - don't stop simulation when selected robot changes
*
* Revision 1.6  2003/03/23 14:36:07  roefer
* GenerateSceneFile added
*
* Revision 1.5  2003/03/22 22:37:52  loetzsch
* finished GUI and message routing for 8 simulated robots
* almost all functionallity for simulated robots now is encapsulated in class
*   CRobotControlSimulatedRobots
*
* Revision 1.4  2003/03/22 16:39:33  loetzsch
* continued GUI for simulated robots
*
* Revision 1.3  2003/03/21 14:25:38  loetzsch
* continued support for 8 simulated robots
*
* Revision 1.2  2003/03/20 23:42:17  loetzsch
* changed cast of AfxGetApp() to CRobotControlApp*
* to getRobotControlApp()
*
* Revision 1.1  2003/03/20 20:36:22  loetzsch
* renamed SimulatorToolbar to Simulated Robots toolbar
* added class CRobotControlSimulatedRobots
* started implementing the GUI for 8 simulated robots
*
*/
