/**
* @file RobotControlToolBar.h
*
* Definition of class CRobotControlToolBar
*
* @author Martin Ltzsch
*/

#ifndef __RobotControlToolBar_h_
#define __RobotControlToolBar_h_

#include "resource.h"
#include "StdAfx.h"
#include "MfcTools/IEStyleToolBars/ToolBarEx.h"
#include "Tools/MessageQueue/InMessage.h"

/**
* @class CRobotControlToolBar
*
* The base class of all tool bars in RobotControl.
*
* @author Martin Ltzsch.
*/
class CRobotControlToolBar : public CToolBarEx
{
public:
  /** Initializes the toolbar's controls and members */
  virtual void Init() = 0;

  /** Determines for every control id if the button has a text or not */
  virtual bool HasButtonText( UINT nID ) { return false; }

  /**
  * Is called for incoming debug messages.
  * @param message The message to handle.
  * @return If the message was handled.
  */
  virtual bool handleMessage(InMessage& message)
  { return false; }

  /**
  * Enables the controls in the dialog bar.
  * This function is called from the main window for all controls.
  * @param pCmdUI An interface to the control that allows enabling/disabling, checking etc.
  */
  virtual void updateUI(CCmdUI* pCmdUI) {};

  /**
  * Handles control notifications which arrived in the main frame
  * @param command The id of the control, menu, accelerator etc.
  * @return If the command was handled.
  */
  virtual bool handleCommand(UINT command) {return false;}

  /**
  * Handles selection change events for combo boxes. That function must be called from the
  * main frame window for all combo boxes of the toolbar.
  * @param nID the command id of the combo box
  * @return if the message was handled
  */
  virtual bool handleSelChange(UINT nID) {return false;}

  /**
  * Handles change events for edit controls. That function must be called from the main frame
  * window for all edit controls of the toolbar.
  * @param nID the command id of the edit control
  * @return if the message was handled
  */
  virtual bool handleEditChange(UINT nID) {return false;}

  /**
  * Handles Dropdown events in for dropdown buttons
  * @return if the message was handled
  */
  virtual bool handleDropDown(NMHDR* pNotifyStruct) { return false;}

  /** Is called when the application is in idle state  */
  virtual void OnIdle() {};
};


/** Creates a toolbar inside the main frame */
#define CREATE_TOOLBAR(classname,idd,title,menuentry) \
  splashScreen.setMessage(CString("Creating the ") + CString(title) + CString(" tool bar ...")); \
  CRobotControlToolBar* p##classname = new classname(); \
  p##classname->Create(this, idd); \
  m_wndReBar.AddBar(p##classname,title,0,0,_T(menuentry),false); \
  p##classname->Init();\
  toolBarMap[idd]=p##classname;\
  toolBarArray.Add(p##classname);


#endif //__RobotControlToolBar_h_

/*
* Change Log:
*
* $Log: RobotControlToolBar.h,v $
* Revision 1.5  2004/02/28 17:40:25  dueffert
* resourceh inclusion tests
*
* Revision 1.4  2004/01/17 20:56:22  loetzsch
* changed the CREATE_TOOLBAR macro
*
* Revision 1.3  2003/12/15 13:05:04  loetzsch
* RobotControl now displays a splash screen at the beginning
*
* Revision 1.2  2003/12/06 06:31:19  loetzsch
* no message
*
* Revision 1.1  2003/10/07 10:09:37  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
* created new repository for the competitions in Padova from the
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.2  2003/05/11 23:51:03  dueffert
* Depend now works with RobotControl too
*
* Revision 1.1  2002/09/10 15:49:04  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
*
* Revision 1.3  2002/09/01 17:20:34  loetzsch
* tidied up the MfcTools/ directory. Created directories
* MfcTools/DockingControlBars and MfcTools/IEStyleToolbars.
*
* Revision 1.2  2002/08/29 15:03:47  dueffert
* documentation bug fixed
*
* Revision 1.1  2002/08/13 12:57:41  loetzsch
* - Added base class CRobotControlToolBar for tool bars
* - Moved command handling from CRobotControlMainFrame to the tool bars
* - Renamed CLayoutToolBar to CConfigurationToolBar
* - Added CRobotControlConfigurationManager
* - Removed CToolsToolBar
*
*/
