/**
* @file NonPortable.cpp
*
* Zweck: Anpassung an MS Windows der auf Basis der
*        Mircosoft Foundation Classes
* Datum: 18.4.96
*/

#include "StdAfx.h"
#include <limits.h>
#include "SimRob95/SimRobot/sim3DErrors.h"
#include "NonPortable.h"

//////////////////////////////////////////////////////////////////////////
// class MFCGRAPH: Windows Implementation von Hiddengraph

MFCGRAPH::MFCGRAPH (CDC* pdc)
{
  m_pDC = pdc;
  m_lastColor = 0;
  m_rgb = 0L;
  m_pPen = new CPen(PS_SOLID,0,m_rgb);
  m_pOld = m_pDC->SelectObject(m_pPen);
  m_pColorList = 0;
}

MFCGRAPH::~MFCGRAPH ()
{
  m_pDC->SelectObject(m_pOld);
  delete m_pPen; 
}

void MFCGRAPH::RegisterColors(COLORLIST& ColorList)
{
  m_pColorList = &ColorList;
}

void MFCGRAPH::SetColor ()
{
  m_pColorList->NthColor(m_lastColor);
  if (m_pColorList && m_pColorList->RGBDefined())
  {
    VECTOR v = m_pColorList->ActualRGB();
    m_rgb = RGB(v.x * 255,v.y*255,v.z*255);
  }
  else
    GetSystemPaletteEntries (m_pDC->m_hDC,m_lastColor,1,
                             (PALETTEENTRY*) &m_rgb);
  CPen* pPen = new CPen(PS_SOLID,0,m_rgb);
  m_pDC->SelectObject(pPen);
  delete m_pPen;
  m_pPen = pPen;
}

void MFCGRAPH::SetLine (const COORD& c1,const COORD& c2, COLOR color)
{
  if (color != m_lastColor)
  {
    m_lastColor = color;
    SetColor ();
  }
  if(c1.x >= SHRT_MIN && c1.x <= SHRT_MAX &&
     c1.y >= SHRT_MIN && c1.y <= SHRT_MAX &&
     c2.x >= SHRT_MIN && c2.x <= SHRT_MAX &&
     c2.y >= SHRT_MIN && c2.y <= SHRT_MAX)
  {
    m_pDC->MoveTo ((int) (c1.x+0.5),(int) (c1.y+0.5));
    m_pDC->LineTo ((int) (c2.x+0.5),(int) (c2.y+0.5));
  }
}

//////////////////////////////////////////////////////////////////////////
// Teile aus dem DIBAPI. Wird von BITMAPTEXTURE bentigt.

DECLARE_HANDLE(HDIB);
#define IS_WIN30_DIB(lpbi)  ((*(LPDWORD)(lpbi)) == sizeof(BITMAPINFOHEADER))
#define RECTWIDTH(lpRect)     ((lpRect)->right - (lpRect)->left)
#define RECTHEIGHT(lpRect)    ((lpRect)->bottom - (lpRect)->top)
#define WIDTHBYTES(bits)    (((bits) + 31) / 32 * 4)

WORD WINAPI DIBNumColors(LPSTR lpbi)
{
  WORD wBitCount;

  if (IS_WIN30_DIB(lpbi))
  {
    DWORD dwClrUsed;

    dwClrUsed = ((LPBITMAPINFOHEADER)lpbi)->biClrUsed;
    if (dwClrUsed != 0)
      return (WORD)dwClrUsed;
  }
  if (IS_WIN30_DIB(lpbi))
    wBitCount = ((LPBITMAPINFOHEADER)lpbi)->biBitCount;
  else
    wBitCount = ((LPBITMAPCOREHEADER)lpbi)->bcBitCount;
  switch (wBitCount)
  {
    case 1:
      return 2;
    case 4:
      return 16;
    case 8:
      return 256;
    default:
      return 0;
  }
}

WORD WINAPI PaletteSize(LPSTR lpbi)
{
  if (IS_WIN30_DIB (lpbi))
   return (WORD)(::DIBNumColors(lpbi) * sizeof(RGBQUAD));
  else
   return (WORD)(::DIBNumColors(lpbi) * sizeof(RGBTRIPLE));
}

LPSTR WINAPI FindDIBBits(LPSTR lpbi)
{
  return (lpbi + *(LPDWORD)lpbi + ::PaletteSize(lpbi));
}

DWORD WINAPI DIBWidth(LPSTR lpDIB)
{
  LPBITMAPINFOHEADER lpbmi;  // pointer to a Win 3.0-style DIB
  LPBITMAPCOREHEADER lpbmc;  // pointer to an other-style DIB
  lpbmi = (LPBITMAPINFOHEADER)lpDIB;
  lpbmc = (LPBITMAPCOREHEADER)lpDIB;
  if (IS_WIN30_DIB(lpDIB))
    return lpbmi->biWidth;
  else
    return (DWORD)lpbmc->bcWidth;
}

DWORD WINAPI DIBHeight(LPSTR lpDIB)
{
  LPBITMAPINFOHEADER lpbmi;  // pointer to a Win 3.0-style DIB
  LPBITMAPCOREHEADER lpbmc;  // pointer to an other-style DIB
  lpbmi = (LPBITMAPINFOHEADER)lpDIB;
  lpbmc = (LPBITMAPCOREHEADER)lpDIB;
  if (IS_WIN30_DIB(lpDIB))
    return lpbmi->biHeight;
  else
    return (DWORD)lpbmc->bcHeight;
}

#define DIB_HEADER_MARKER   ((WORD) ('M' << 8) | 'B')

HDIB WINAPI ReadDIBFile(CFile& file)
{
  BITMAPFILEHEADER bmfHeader;
  DWORD dwBitsSize;
  HDIB hDIB;
  LPSTR pDIB;

  dwBitsSize = (DWORD) file.GetLength();
  if ((file.Read((LPSTR)&bmfHeader, sizeof(bmfHeader)) !=
    sizeof(bmfHeader)) || (bmfHeader.bfType != DIB_HEADER_MARKER))
  {
    return NULL;
  }
  hDIB = (HDIB) ::GlobalAlloc(GMEM_MOVEABLE | GMEM_ZEROINIT, dwBitsSize);
  if (hDIB == 0)
  {
    return NULL;
  }
  pDIB = (LPSTR) ::GlobalLock((HGLOBAL) hDIB);
  if (file.Read(pDIB, dwBitsSize - sizeof(BITMAPFILEHEADER)) !=
    dwBitsSize - sizeof(BITMAPFILEHEADER) || 
    ((LPBITMAPINFOHEADER) pDIB)->biCompression != BI_RGB)
  {
    ::GlobalUnlock((HGLOBAL) hDIB);
    ::GlobalFree((HGLOBAL) hDIB);
    return NULL;
  }
  ::GlobalUnlock((HGLOBAL) hDIB);
  return hDIB;
}

RGBQUAD GetDIBPixel(HDIB hDIB,int x,int y)
{
  LPBITMAPINFO pInfo = (LPBITMAPINFO) ::GlobalLock((HGLOBAL) hDIB);
  LPBITMAPINFOHEADER pHdr = (LPBITMAPINFOHEADER) &(pInfo->bmiHeader);
  unsigned char *p;
  unsigned char *pBits = (unsigned char*)
                                ::FindDIBBits((LPSTR) pHdr);
  RGBQUAD rgb,
         *pColors = (RGBQUAD*) ((char*) pInfo + pHdr->biSize);
  long nSize = WIDTHBYTES(pHdr->biWidth*pHdr->biBitCount);
  int i;
  switch (pHdr->biBitCount)
  {
  case 1:
    i = (*(pBits + nSize*y+x/8) >> (7 - (x & 7))) & 1;
    break;
  case 4:
    i = (*(pBits + nSize*y+x/2) >> (1 - (x & 1))*4) & 15;
    break;
  case 8:
    i = *(pBits + nSize*y+x);
    break;
  case 24:
    p = pBits + nSize*y+x*3;
    rgb.rgbRed = *p;
    rgb.rgbGreen = *(p+1);
    rgb.rgbBlue = *(p+2);
    ::GlobalUnlock((HGLOBAL) hDIB);
    return rgb;
  }
  rgb = pColors[i];
  ::GlobalUnlock((HGLOBAL) hDIB);
  return rgb;
}
//////////////////////////////////////////////////////////////////////////
// class BITMAPTEXTURE: Windows Implementation von Bitmap-Texturen

void BITMAPTEXTURE::ReadDIB()
{
  m_hDIB = 0;
  CFile file;
  CFileException fe;
  if (!file.Open((const char *) m_sFileName, 
                 CFile::modeRead | CFile::shareDenyWrite, &fe))
  {
    sim3DOtherError (sim3DInvalidTextureFile);
  }
  TRY
  {
    m_hDIB = (unsigned) ::ReadDIBFile(file);
  }
  CATCH (CFileException, eLoad)
  {
    file.Abort();
    sim3DOtherError (sim3DInvalidTextureFile);
  }
  END_CATCH
  file.Close();
  if (!m_hDIB)
    sim3DOtherError (sim3DInvalidTextureFile);
}

BOOLEAN BITMAPTEXTURE::CalcSize()
{
  LPSTR lpDIB = (LPSTR) ::GlobalLock((HGLOBAL) m_hDIB);
  if (::DIBWidth(lpDIB) > INT_MAX ||::DIBHeight(lpDIB) > INT_MAX)
  {
    ::GlobalUnlock((HGLOBAL) m_hDIB);
    sim3DOtherError (sim3DInvalidTextureFile);
    return FALSE;
  }
  else
  {
    m_xSize = (int) ::DIBWidth(lpDIB);
    m_ySize = (int) ::DIBHeight(lpDIB);
    ::GlobalUnlock((HGLOBAL) m_hDIB);
    return TRUE;
  }
}

BITMAPTEXTURE::BITMAPTEXTURE (STRING name, STRING filename) :
  TEXTUREDESCR (name)
{
  m_sFileName = filename;
  ReadDIB();
  if (m_hDIB)
  {
    if (!CalcSize())
    {
      ::GlobalFree((HGLOBAL) m_hDIB);
      m_hDIB = 0;
    }
  }
}

BITMAPTEXTURE::~BITMAPTEXTURE ()
{
  if (m_hDIB)
    ::GlobalFree((HGLOBAL) m_hDIB);
}

void BITMAPTEXTURE::WriteProperties (BACKUP& backup)
{
  TEXTUREDESCR::WriteProperties (backup);
  backup.WriteIdent ("LOAD");
  backup.WriteString (m_sFileName);
}

void BITMAPTEXTURE::GetSize (INTEGER& planes, REAL& width, REAL& height)
{
  planes = 4;
  width = m_xSize;
  height = m_ySize;
}

SHORTREAL BITMAPTEXTURE::GetValueAt (INTEGER plane, REAL x, REAL y)
{
  RGBQUAD rgb = GetDIBPixel((HDIB) m_hDIB,(INTEGER) (x * (m_xSize-1)),
                                          (INTEGER) ((1-y) * (m_ySize-1)));
  switch (plane)
  {
    case 1: return (SHORTREAL) (rgb.rgbRed / 256.0);
    case 2: return (SHORTREAL) (rgb.rgbGreen / 256.0);
    case 3: return (SHORTREAL) (rgb.rgbBlue / 256.0);
    default: return (SHORTREAL) ((rgb.rgbRed * 77.0 + 
                                  rgb.rgbGreen * 151.0 + 
                                  rgb.rgbBlue * 28.0) / 65536.0);
  }
}

/*
 * Change log:
 *
 * $Log: NonPortable.cpp,v $
 * Revision 1.3  2003/12/04 22:56:07  roefer
 * Compatibility with VC2003.NET
 *
 * Revision 1.2  2003/11/30 01:53:21  loetzsch
 * prepared RobotControl port to Visual C++ .Net
 *
 * Revision 1.1  2003/10/07 10:11:08  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1.1.1  2003/07/02 09:40:26  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.4  2003/05/11 23:36:18  dueffert
 * Depend now works with RobotControl too
 *
 * Revision 1.3  2003/03/28 11:37:31  timrie
 * Added changelog
 *
 */
