/** 
 * @file RobotControl/Visualization/DebugDrawing.cpp
 * Implementation of class DebugDrawing.
 *
 * @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Jngel</A>
 */
#include "DebugDrawing.h"
#include "Tools/FieldDimensions.h"

DebugDrawing::DebugDrawing(Drawings::FieldDrawing drawing)
{
  this->fieldDrawingID = drawing;
  this->typeOfDrawing = Drawings::drawingOnField;
}


DebugDrawing::DebugDrawing(Drawings::ImageDrawing drawing)
{
  this->imageDrawingID = drawing;
  this->typeOfDrawing = Drawings::drawingOnImage;
}

const DebugDrawing& DebugDrawing::operator=(const DebugDrawing& other)
{
  reset();

  fieldDrawingID = other.fieldDrawingID;
  imageDrawingID = other.imageDrawingID;
  typeOfDrawing = other.typeOfDrawing;

  *this += other;

  return *this;
}

const DebugDrawing& DebugDrawing::operator+=(const DebugDrawing& other)
{
  for(List<Element*>::Pos i = other.elements.getFirst(); i; ++i)
    switch(other.elements[i]->type)
    {
      case Element::LINE:
      {
        Line* element = new Line;
        *element = *(Line*) other.elements[i];
        elements.insert(element);
        break;
      }
      case Element::POLYGON:
      {
        Polygon* element = new Polygon;
        *element = *(Polygon*) other.elements[i];
        elements.insert(element);
        break;
      }
      case Element::ELLIPSE:
      {
        Ellipse* element = new Ellipse;
        *element = *(Ellipse*) other.elements[i];
        elements.insert(element);
        break;
      }
    }

  return *this;
}

DebugDrawing::DebugDrawing(const DebugDrawing& other)
{
  *this = other;
}

DebugDrawing::DebugDrawing(const DebugDrawing* pDebugDrawing)
{
  *this = *pDebugDrawing;
}

void DebugDrawing::reset()
{
  for(List<Element*>::Pos i = elements.getFirst(); i; ++i)
    delete elements[i];
  elements.clear();
}

void DebugDrawing::arrow
( 
  Vector2<double> start,
  Vector2<double> end,
  Color penColor
)
{
  Vector2<double> startToEnd((end.x - start.x)/4, (end.y - start.y)/4);
  Vector2<double> perpendicular(startToEnd.y, -1*startToEnd.x);

	// start to endpoint
  line((int)start.x, (int)start.y, (int)(end.x), (int)(end.y), Drawings::ps_solid, 0, penColor);
	// endpoint to left and right
  line((int)(end.x), (int)(end.y), (int)(end.x - startToEnd.x + perpendicular.x), (int)(end.y - startToEnd.y + perpendicular.y), Drawings::ps_solid, 0, penColor);
  line((int)(end.x), (int)(end.y), (int)(end.x - startToEnd.x - perpendicular.x), (int)(end.y - startToEnd.y - perpendicular.y), Drawings::ps_solid, 0, penColor);
}

void DebugDrawing::arrow
( 
 int xStart, 
 int yStart, 
 int xEnd,
 int yEnd,
 Drawings::PenStyle penStyle,
 int width,
 Color penColor
)
{
	arrow(Vector2<double>(xStart, yStart), Vector2<double>(xEnd, yEnd), penColor);
}


void DebugDrawing::line
(
 int xStart, 
 int yStart, 
 int xEnd,
 int yEnd,
 Drawings::PenStyle penStyle,
 int width,
 Color penColor
 )
{
  Line* element = new Line;
  element->xStart = xStart;
  element->yStart = yStart;
  element->xEnd = xEnd;
  element->yEnd = yEnd;
  element->penStyle = penStyle;
  element->width = width;
  element->penColor = penColor;
  elements.insert(element);
}

void DebugDrawing::line
(
 int xStart, 
 int yStart, 
 int xEnd,
 int yEnd
 )
{
  line(xStart, yStart, xEnd, yEnd, Drawings::ps_solid, 1, Color(0,0,0));
}

void DebugDrawing::polygon
(
 const Vector2<int>* points,
 int nCount,
 int width,
 Drawings::PenStyle penStyle,
 Color penColor,
 Drawings::FillStyle fillStyle,
 Color fillColor
 )
{
  Polygon* element = new Polygon;
  memcpy(element->points, points, sizeof(element->points[0]) * nCount);
  element->nCount = nCount;
  element->width = width;
  element->penStyle = penStyle;
  element->penColor = penColor;
  element->fillStyle = fillStyle;
  element->fillColor = fillColor;
  elements.insert(element);
}

void DebugDrawing::dot
(
 int x,
 int y,
 Color penColor,
 Color fillColor
 )
{
  Vector2<int> points[4];
  points[0].x = x - 1;
  points[0].y = y - 1;
  points[1].x = x + 1;
  points[1].y = y - 1;
  points[2].x = x + 1;
  points[2].y = y + 1;
  points[3].x = x - 1;
  points[3].y = y + 1;
  polygon(
    points,
    4,
    0,
    Drawings::ps_solid,
    penColor,
    Drawings::bs_solid,
    fillColor);
}

void DebugDrawing::largeDot
(
 int x,
 int y,
 Color penColor,
 Color fillColor
 )
{
  Vector2<int> points[4];
  points[0].x = x - 25;
  points[0].y = y - 25;
  points[1].x = x + 25;
  points[1].y = y - 25;
  points[2].x = x + 25;
  points[2].y = y + 25;
  points[3].x = x - 25;
  points[3].y = y + 25;
  polygon(
    points,
    4,
    3,
    Drawings::ps_solid,
    penColor,
    Drawings::bs_solid,
    fillColor);
}


void DebugDrawing::rectangle
(
 int left, 
 int right, 
 int top,
 int bottom,
 int width,
 Drawings::PenStyle penStyle,
 Color penColor,
 Drawings::FillStyle fillStyle,
 Color fillColor
 )
{
  Vector2<int> points[4];
  points[0].x = left;
  points[0].y = top;
  points[1].x = right;
  points[1].y = top;
  points[2].x = right;
  points[2].y = bottom;
  points[3].x = left;
  points[3].y = bottom;
  polygon(
    points,
    4,
    width,
    penStyle,
    penColor,
    fillStyle,
    fillColor);
}

void DebugDrawing::ellipse
(
 int left, 
 int top,
 int right, 
 int bottom,
 int width,
 Drawings::PenStyle penStyle,
 Color penColor,
 Drawings::FillStyle fillStyle,
 Color fillColor
 )
{
  Ellipse* element = new Ellipse;
  element->left = left;
  element->right= right;
  element->top = top;
  element->bottom = bottom;
  element->width = width;
  element->penStyle = penStyle;
  element->penColor = penColor;
  element->fillStyle = fillStyle;
  element->fillColor = fillColor;
  elements.insert(element);
}

void DebugDrawing::ellipse
(
 int left, int top, int right, int bottom, Color color
 )
{
  ellipse(left, top, right, bottom, 0, Drawings::ps_solid, Color(0,0,0), Drawings::bs_solid, color);
}

void DebugDrawing::circle
(
 int xCenter,
 int yCenter,
 int radius, 
 int penWidth,
 Drawings::PenStyle penStyle,
 Color penColor,
 Drawings::FillStyle fillStyle,
 Color fillColor
 )
{
  ellipse(
    xCenter - radius, yCenter - radius,
    xCenter + radius + 1, yCenter + radius + 1, 
    penWidth, penStyle, penColor,
    fillStyle, fillColor);
}


// added Feb., 6th 2002 by Denny N. Kone
// draws a Pose2D as an Arrow on the field

void DebugDrawing::pose2DSample
(
 Pose2D pose
 )
{
  Pose2D current = pose;
  current += Pose2D(-100,0);
  line(int(current.translation.x),int(current.translation.y),int(pose.translation.x),int(pose.translation.y));
  current = pose;
  current += Pose2D(-40,-40);
  line(int(current.translation.x),int(current.translation.y),int(pose.translation.x),int(pose.translation.y));
  current = pose;
  current += Pose2D(-40,40);
  line(int(current.translation.x),int(current.translation.y),int(pose.translation.x),int(pose.translation.y));
}

void DebugDrawing::pose2DSample
(
 Pose2D pose, Color color
 )
{
  Pose2D current = pose;
  current += Pose2D(-100,0);
  line(int(current.translation.x),int(current.translation.y),int(pose.translation.x),int(pose.translation.y),
    Drawings::ps_solid, 1, color);
  current = pose;
  current += Pose2D(-40,-40);
  line(int(current.translation.x),int(current.translation.y),int(pose.translation.x),int(pose.translation.y),
    Drawings::ps_solid, 1, color);
  current = pose;
  current += Pose2D(-40,40);
  line(int(current.translation.x),int(current.translation.y),int(pose.translation.x),int(pose.translation.y),
    Drawings::ps_solid, 1, color);
}

void DebugDrawing::potentialfieldRectangles(int potentialColor[])
{
  int fieldWidth = ((2*xPosFrontFlags) / DebugDrawing::POTENTIALFIELD_DRAWING_X_RESOLUTION);
  int fieldHeight = ((2*yPosLeftFlags) / DebugDrawing::POTENTIALFIELD_DRAWING_Y_RESOLUTION);

  int top = yPosRightFlags,
      bottom = top + fieldHeight,
      left = xPosFrontFlags,
      right = left - fieldWidth;

  int index = 0;
  for(int y = 0; y < DebugDrawing::POTENTIALFIELD_DRAWING_Y_RESOLUTION; y++)
  {
    for(int x = 0; x < DebugDrawing::POTENTIALFIELD_DRAWING_X_RESOLUTION; x++)
    {
/*
      this->rectangle(left, right, top, bottom,
                      0, PenStyle::ps_null, DebugDrawing::Color(0,0,0), FillStyle::bs_solid,
                      DebugDrawing::Color(potentialColor[index], potentialColor[index], potentialColor[index]));
*/
      this->rectangle(left, right, top, bottom, DebugDrawing::Color(potentialColor[index], potentialColor[index], potentialColor[index]));
      left = right - 1;
      right = left - fieldWidth;
      
      index++;
    }
    top = bottom + 1;
    bottom = top + fieldHeight;
    left = xPosFrontFlags;
    right = left - fieldWidth;

  }

}

bool DebugDrawing::addShapeFromQueue
(
 InMessage& message,
 Drawings::ShapeType shapeType
 )
{
  switch ((Drawings::ShapeType)shapeType)
  {
  case Drawings::circle:
    {
      int x, y, radius;
      char penWidth, penStyle, penColor;
      message.bin >> x;
      message.bin >> y;
      message.bin >> radius;
      message.bin >> penWidth;
      message.bin >> penStyle;
      message.bin >> penColor;

      this->circle(x, y, radius, 
        penWidth, 
        (Drawings::PenStyle)penStyle, DebugDrawing::Color((Drawings::Color)penColor), 
        Drawings::bs_null, DebugDrawing::Color(0,0,0) 
        );
    }
    break;
  case Drawings::filledCircle:
    break;
  case Drawings::line:
    {
      int x1, y1, x2, y2;
      char penWidth, penStyle, penColor;
      message.bin >> x1;
      message.bin >> y1;
      message.bin >> x2;
      message.bin >> y2;
      message.bin >> penWidth;
      message.bin >> penStyle;
      message.bin >> penColor;

      this->line(x1, y1, x2, y2, 
        (Drawings::PenStyle)penStyle, penWidth, DebugDrawing::Color((Drawings::Color)penColor) );
    }
    break;
  case Drawings::arrow:
    {
      int x1, y1, x2, y2;
      char penWidth, penStyle, penColor;
      message.bin >> x1;
      message.bin >> y1;
      message.bin >> x2;
      message.bin >> y2;
      message.bin >> penWidth;
      message.bin >> penStyle;
      message.bin >> penColor;

      this->arrow(x1, y1, x2, y2, 
        (Drawings::PenStyle)penStyle, penWidth, DebugDrawing::Color((Drawings::Color)penColor) );
    }
    break;
  case Drawings::dot:
    {
      int x, y;
      char penColor, fillColor;
      message.bin >> x;
      message.bin >> y;
      message.bin >> penColor;
      message.bin >> fillColor;

      this->dot(x, y, 
        DebugDrawing::Color((Drawings::Color)penColor), 
        DebugDrawing::Color((Drawings::Color)fillColor) );
    }
    break;
  case Drawings::largeDot:
    {
      int x, y;
      char penColor, fillColor;
      message.bin >> x;
      message.bin >> y;
      message.bin >> penColor;
      message.bin >> fillColor;

      this->largeDot(x, y, 
        DebugDrawing::Color((Drawings::Color)penColor), 
        DebugDrawing::Color((Drawings::Color)fillColor) );
    }
    break;
  case Drawings::octangle:
    {
      char color, fill;
      
      Vector2<int> points[8];

      message.bin >> points[0].x;
      message.bin >> points[0].y;
      message.bin >> points[1].x;
      message.bin >> points[1].y;
      message.bin >> points[2].x;
      message.bin >> points[2].y;
      message.bin >> points[3].x;
      message.bin >> points[3].y;
      message.bin >> points[4].x;
      message.bin >> points[4].y;
      message.bin >> points[5].x;
      message.bin >> points[5].y;
      message.bin >> points[6].x;
      message.bin >> points[6].y;
      message.bin >> points[7].x;
      message.bin >> points[7].y;
      message.bin >> color;
      message.bin >> fill;
  
      if(fill)
      {
        this->polygon(
          &points[0],
          8,
          0,
          Drawings::ps_solid,
          DebugDrawing::Color(127,127,127),
          Drawings::bs_solid,
          DebugDrawing::Color((Drawings::Color)color)
          );
      }
      else
      {
        this->polygon(
          &points[0],
          8,
          1,
          Drawings::ps_solid,
          DebugDrawing::Color((Drawings::Color)color),
          Drawings::bs_null,
          DebugDrawing::Color((Drawings::Color)color)
          );
      }

    }
    break;
  }
  return true;
}

In& operator>>(In& stream,DebugDrawing& debugDrawing)
{
  debugDrawing.reset();

  int num, count;
  stream >> num;
  debugDrawing.fieldDrawingID = (Drawings::FieldDrawing)num;

  stream >> num;
  debugDrawing.imageDrawingID = (Drawings::ImageDrawing)num;

  stream >> num;
  debugDrawing.typeOfDrawing = (enum Drawings::TypeOfDrawing)num;

  stream >> count;

  while(count > 0)
  {
    stream >> num;
    switch(num)
    {
      case DebugDrawing::Element::LINE:
      {
        DebugDrawing::Line* element = new DebugDrawing::Line;
        stream >> element->xStart
               >> element->yStart
               >> element->xEnd
               >> element->yEnd
               >> element->penColor.red
               >> element->penColor.blue
               >> element->penColor.green
               >> num
               >> element->width;
        element->penStyle = (Drawings::PenStyle)num;
        debugDrawing.elements.insert(element);
        break;
      }
      case DebugDrawing::Element::ELLIPSE:
      {
        DebugDrawing::Ellipse* element = new DebugDrawing::Ellipse;
        stream >> element->left
               >> element->right
               >> element->top
               >> element->bottom
               >> num;
        element->penStyle = (Drawings::PenStyle)num;
        stream >> element->width
               >> element->penColor.red
               >> element->penColor.green
               >> element->penColor.blue
               >> num
               >> element->fillColor.red
               >> element->fillColor.green
               >> element->fillColor.blue;
        element->fillStyle = (Drawings::FillStyle)num;
        debugDrawing.elements.insert(element);
        break;
      }
      case DebugDrawing::Element::POLYGON:
      {
        DebugDrawing::Polygon* element = new DebugDrawing::Polygon;
        stream >> num;
        element->penStyle = (Drawings::PenStyle)num;
        stream >> element->width
               >> element->penColor.red
               >> element->penColor.green
               >> element->penColor.blue
               >> num
               >> element->fillColor.red
               >> element->fillColor.green
               >> element->fillColor.blue
               >> element->nCount;
        element->fillStyle = (Drawings::FillStyle)num;
        for(int n = 0; n < element->nCount; n++)
          stream >> element->points[n].x
                 >> element->points[n].y;
        debugDrawing.elements.insert(element);
      }
    }
    --count;
  }
  return stream;
}

Out& operator<<(Out& stream, const DebugDrawing& debugDrawing)
{
  stream << debugDrawing.fieldDrawingID
         << debugDrawing.imageDrawingID
         << debugDrawing.typeOfDrawing
         << debugDrawing.elements.getSize();

  for(List<DebugDrawing::Element*>::Pos i = debugDrawing.elements.getFirst(); i; ++i)
  {
    stream << debugDrawing.elements[i]->type;
    switch(debugDrawing.elements[i]->type)
    {
      case DebugDrawing::Element::LINE:
      {
        const DebugDrawing::Line* element = (const DebugDrawing::Line*) debugDrawing.elements[i];
        stream << element->xStart
               << element->yStart
               << element->xEnd
               << element->yEnd
               << element->penColor.red
               << element->penColor.blue
               << element->penColor.green
               << element->penStyle
               << element->width;
        break;
      }
      case DebugDrawing::Element::ELLIPSE:
      {
        const DebugDrawing::Ellipse* element = (const DebugDrawing::Ellipse*) debugDrawing.elements[i];
        stream << element->left
               << element->right
               << element->top
               << element->bottom
               << element->penStyle
               << element->width
               << element->penColor.red
               << element->penColor.green
               << element->penColor.blue
               << element->fillStyle
               << element->fillColor.red
               << element->fillColor.green
               << element->fillColor.blue;
        break;
      }
      case DebugDrawing::Element::POLYGON:
      {
        const DebugDrawing::Polygon* element = (const DebugDrawing::Polygon*) debugDrawing.elements[i];
        stream << element->penStyle
               << element->width
               << element->penColor.red
               << element->penColor.green
               << element->penColor.blue
               << element->fillStyle
               << element->fillColor.red
               << element->fillColor.green
               << element->fillColor.blue
               << element->nCount;
        for(int n = 0; n < element->nCount; n++)
          stream << element->points[n].x
                 << element->points[n].y;
      }
    }
  }
  return stream;
}


/*
 * Change log :
 * 
 * $Log: DebugDrawing.cpp,v $
 * Revision 1.3  2003/12/07 18:58:50  loetzsch
 * added += operator
 *
 * Revision 1.2  2003/10/30 22:46:28  roefer
 * DebugDrawings now preserve sequence of drawing instructions
 *
 * Revision 1.1  2003/10/07 10:11:08  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1.1.1  2003/07/02 09:40:26  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.7  2003/06/24 10:10:05  jhoffman
 * added arrow so it can be sent
 *
 * Revision 1.6  2003/06/13 14:49:11  juengel
 * Changed largeDot.
 *
 * Revision 1.5  2003/06/10 18:01:53  juengel
 * Added LARGE_DOT.
 *
 * Revision 1.4  2003/03/29 13:44:01  roefer
 * Parameter of polygon changed
 *
 * Revision 1.3  2002/12/18 16:22:56  dueffert
 * doxygen docu corrected
 *
 * Revision 1.2  2002/11/12 16:08:11  loetzsch
 * Added (char) and (int) casts to DebugDrawing Macros
 *
 * Revision 1.1  2002/11/12 10:52:52  juengel
 * New debug drawing macros - second step
 * -moved /Tools/Debugging/PaintMethods.h and . cpp
 *  to /Visualization/DrawingMethods.h and .cpp
 * -moved DebugDrawing.h and .cpp from /Tools/Debugging/
 *  to /Visualization
 *
 * Revision 1.3  2002/10/14 13:14:24  dueffert
 * doxygen comments corrected
 *
 * Revision 1.2  2002/09/22 18:40:52  risler
 * added new math functions, removed GTMath library
 *
 * Revision 1.1  2002/09/10 15:53:58  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.6  2002/06/17 18:34:43  kspiess
 * new drawing method for pose2d with color
 *
 * Revision 1.5  2002/06/12 11:35:36  roefer
 * Deleting lists is now less stack-intensive
 *
 * Revision 1.4  2002/06/06 13:59:35  jhoffman
 * no message
 *
 * Revision 1.3  2002/06/03 13:12:57  kspiess
 * debug drawing for potentialfield added
 *
 * Revision 1.2  2002/05/27 10:40:20  dueffert
 * warnings removed
 *
 * Revision 1.1.1.1  2002/05/10 12:40:32  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.12  2002/04/23 14:16:47  jhoffman
 * added gradient image processor
 *
 * Revision 1.1.1.1  2002/04/10 11:03:22  juengel
 * no message
 *
 * Revision 1.11  2002/04/07 13:02:13  roefer
 * Improved goal drawing
 *
 * Revision 1.10  2002/02/10 12:45:15  loetzsch
 * radar viewer and percept visualization continued
 *
 * Revision 1.9  2002/02/08 13:28:08  loetzsch
 * Visualization of PerceptCollection and RadarViewer started
 * (not implemented yet)
 *
 * Revision 1.8  2002/02/06 15:34:44  Denny
 * added method for drawing of Pose2D as Arrows on the field
 *
 * Revision 1.7  2002/01/22 14:55:15  juengel
 * ImageToPerceptCollection eingefhrt
 *
 * Revision 1.6  2002/01/17 15:13:18  roefer
 * Views and debug output in SimGT2002
 *
 * Revision 1.5  2002/01/06 12:15:01  juengel
 * Streaming Operatoren iplementiert
 *
 * Revision 1.4  2002/01/04 14:35:19  juengel
 * no message
 *
 * Revision 1.3  2001/12/14 17:35:49  juengel
 * struct Color, enum penStyle und enum fillStyle in die Klasse DebugDrawing hereingenommen.
 *
 * Revision 1.2  2001/12/12 17:34:26  risler
 * changed Point to Vector2
 *
 * Revision 1.1  2001/12/12 15:21:09  juengel
 * CDebugKeyToolBar in eigenes .cpp und .h File verlagert.
 *
 * Revision 1.7  2001/12/10 17:47:10  risler
 * change log added
 *
 */
