/** 
 * @file ImageMethods.cpp
 * Implementation of class ImageMethods.
 *
 * @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Jngel</a>
 */
#include "StdAfx.h"
#include "ImageMethods.h"

#include "Tools/Streams/OutStreams.h"

#include <gl/gl.h>
#include <gl/glu.h>

void ImageMethods::paintColorClassImage2CDC
(
 CDC& dc, 
 const ColorClassImage& colorClassImage,
 CRect destinationRect
)
{
  int xSize = colorClassImage.width;
  int ySize = colorClassImage.height;

  BITMAPINFOHEADER* pBuffer = (BITMAPINFOHEADER*) new
    char[sizeof(BITMAPINFOHEADER) + ((xSize * 3 + 3) & 0xfffffffc) * ySize];
  pBuffer->biSize          = sizeof(BITMAPINFOHEADER);
  pBuffer->biWidth         = xSize;
  pBuffer->biHeight        = ySize;
  pBuffer->biPlanes        = 1;
  pBuffer->biBitCount      = 24;
  pBuffer->biCompression   = BI_RGB;
  pBuffer->biSizeImage     = 0;
  pBuffer->biXPelsPerMeter = 0;
  pBuffer->biYPelsPerMeter = 0;
  pBuffer->biClrUsed       = 0;
  pBuffer->biClrImportant  = 0;
  char *p = (char*) (pBuffer+1); // p points to beginning of buffer

  for(int y = ySize-1; y >= 0; --y) // DIBs are upside down
  {
    for(int x = 0; x < xSize; ++x)
    {
      int r,g,b;

      int rgb = ColorClasses::colorClassToRGB((colorClass)colorClassImage.image[y][x]);

      r = rgb % 0x100;
      rgb /= 0x100;
      g = rgb % 0x100;
      rgb /= 0x100;
      b = rgb % 0x100;
      
      *p++ = b;
      *p++ = g;
      *p++ = r;
    }
    p += ((xSize * 3 + 3) & 0xfffffffc) - xSize * 3; // pad to 4 byte boundaries

  }
  /*if(destinationRect.right == xSize)
  {
    SetDIBitsToDevice(
      dc.m_hDC, 
      destinationRect.left,
      destinationRect.top,
      xSize,
      ySize,
      0, 0, 0, ySize, 
      pBuffer+1,(BITMAPINFO*) pBuffer,DIB_RGB_COLORS);
  }
  else*/
  {
    StretchDIBits(dc.m_hDC,
      destinationRect.left, 
      destinationRect.top,
      destinationRect.right - destinationRect.left,
      destinationRect.bottom - destinationRect.top,
      0, 0, xSize, ySize,
      pBuffer+1,(BITMAPINFO*) pBuffer,DIB_RGB_COLORS, SRCCOPY);
  }
  delete [] (char*) pBuffer;
}

void ImageMethods::paintImage2CDCAsYUV
(
 CDC& dc,
 const Image& image,
 CRect destinationRect
)	
{
	ImageMethods::paintImage2CDCAsYUV(dc,image,destinationRect,false);
}

void ImageMethods::paintImage2CDCAsYUV
(
 CDC& dc,
 const Image& image,
 CRect destinationRect,
 bool bwImage	
 )
{
	int xSize,ySize;
	if (bwImage)
	{
		// double the resolution, because the BW Image has double size than color image
		xSize = 2 * image.cameraInfo.resolutionWidth;
		ySize = 2 * image.cameraInfo.resolutionHeight;
	}
	else
	{
		xSize = image.cameraInfo.resolutionWidth;
		ySize = image.cameraInfo.resolutionHeight;
	}


  BITMAPINFOHEADER* pBuffer = (BITMAPINFOHEADER*) new
    char[sizeof(BITMAPINFOHEADER) + ((xSize * 3 + 3) & 0xfffffffc) * ySize];
  pBuffer->biSize          = sizeof(BITMAPINFOHEADER);
  pBuffer->biWidth         = xSize;
  pBuffer->biHeight        = ySize;
  pBuffer->biPlanes        = 1;
  pBuffer->biBitCount      = 24;
  pBuffer->biCompression   = BI_RGB;
  pBuffer->biSizeImage     = 0;
  pBuffer->biXPelsPerMeter = 0;
  pBuffer->biYPelsPerMeter = 0;
  pBuffer->biClrUsed       = 0;
  pBuffer->biClrImportant  = 0;
  char *p = (char*) (pBuffer+1); // p points to beginning of buffer
  char *pStart=p;
	
  int scale = 16384;
	// needed for transformation to rgb
	int r,g,b;
  // old factors for transformation yuv->rgb
	//int factor1 = (int)(1.140 * scale);
	//int factor2 = (int)(0.394 * scale);
	//int factor3 = (int)(2.028 * scale);
	//int factor4 = (int)(0.581 * scale);
  // new factors for transformation ycbcr->rgb
	int factor1 = (int)(1.77100 * scale);
	int factor2 = (int)(0.34560 * scale);
	int factor3 = (int)(1.40210 * scale);
	int factor4 = (int)(0.71448 * scale);

	int yImage,uImage,vImage;
	if (!bwImage)
	{
		for(int y = ySize-1; y >= 0; --y) // DIBs are upside down
		{
			for(int x = 0; x < xSize; ++x)
			{
				yImage = scale * image.image[y][0][x];
				uImage = image.image[y][1][x] - 128;
				vImage = image.image[y][2][x] - 128;

				// transformation to rgb
				r = yImage + factor3 * uImage;
				g = yImage - factor2 * uImage - factor4 * vImage;
				b = yImage + factor1 * vImage;

				if(r<0){r=0;} if(r>255 * scale){r=255 * scale;}
				if(g<0){g=0;} if(g>255 * scale){g=255 * scale;}
				if(b<0){b=0;} if(b>255 * scale){b=255 * scale;}

				*p++ = b / scale;
				*p++ = g / scale;
				*p++ = r / scale;
			}
			p += ((xSize * 3 + 3) & 0xfffffffc) - xSize * 3; // pad to 4 byte boundaries
		}
	}
	else
	{
		for(int y = ySize - 1; y >= 0; --y) // DIBs are upside down
		{
			for(int x = 0; x < xSize; ++x)
			{
				yImage = scale * ((int)image.getHighResY(x, y));
				uImage = image.image[y / 2][1][x / 2] - 128;
				vImage = image.image[y / 2][2][x / 2] - 128;

				// transformation to rgb
				r = yImage + factor3 * uImage;
				g = yImage - factor2 * uImage - factor4 * vImage;
				b = yImage + factor1 * vImage;

				if(r<0){r=0;} if(r>255 * scale){r=255 * scale;}
				if(g<0){g=0;} if(g>255 * scale){g=255 * scale;}
				if(b<0){b=0;} if(b>255 * scale){b=255 * scale;}

				*p++ = b / scale;
				*p++ = g / scale;
				*p++ = r / scale;
			}
			p += ((xSize / 2 * 3 + 3) & 0xfffffffc) - xSize / 2 * 3; // pad to 4 byte boundaries
		}
    /*
    // black and white image conversion
    register unsigned char Val;
		int nextLine = ((xSize * 3 + 3) & 0xfffffffc);
		for(int y = image.cameraInfo.resolutionHeight-1; y>=0; --y)
		{
  		p=pStart;

			for(int x = 0;x < image.cameraInfo.resolutionWidth; ++x)
			{
				signed short LL,HH,HL,LH;
				LL = image.image[y][0][x];					// Average Color Value
				HH = image.image[y][5][x] - 128;		// diagonal Value
				HL = image.image[y][4][x] - 128;		// vertical Value
				LH = image.image[y][3][x] - 128;		// horizontal Value

				// Pixel Top Left				= LL + HH - HL - LH
				// Pixel Top Right			= LL + HL - LH - HH
				// Pixel Bottom Left		= LL + LH - HL - HH
				// Pixel Bottom Right		= LL + HH + HL + LH
				*p++ = Val = LL + (LH - HL - HH); // Pixel Bottom Left
				*p++ = Val;
				*p++ = Val;
        
				*p++ = Val = LL + (HH + HL + LH); // Pixel Bottom Right	
				*p++ = Val;
				*p++ = Val;

				p+=nextLine-6;


				*p++ = Val = LL + (HH - HL - LH); // Pixel Top Left
				*p++ = Val;
				*p++ = Val;
        
				*p++ = Val = LL + (HL - LH - HH); // Pixel Top Right
				*p++ = Val;
				*p++ = Val;


				p-=nextLine;


			}

			pStart+=2*nextLine;
		}
    */
	}
  {
    StretchDIBits(dc.m_hDC,
      destinationRect.left, 
      destinationRect.top,
      destinationRect.right - destinationRect.left,
      destinationRect.bottom - destinationRect.top,
      0, 0, xSize, ySize,
      pBuffer+1,(BITMAPINFO*) pBuffer,DIB_RGB_COLORS, SRCCOPY);
  }
  delete [] (char*) pBuffer;
}

void ImageMethods::paint_U_V_ColorSpaceUsage2CDC
(
 BITMAPINFOHEADER* pBuffer,
 const Image& image
 )
{
  int xSize = 3*64;
  int ySize = 3*64;

  int scale = 16384;
	// needed for transformation to rgb
	int r,g,b;
  // old factors for transformation yuv->rgb
	//int factor1 = (int)(1.140 * scale);
	//int factor2 = (int)(0.394 * scale);
	//int factor3 = (int)(2.028 * scale);
	//int factor4 = (int)(0.581 * scale);
  // new factors for transformation ycbcr->rgb
	int factor1 = (int)(1.77100 * scale);
	int factor2 = (int)(0.34560 * scale);
	int factor3 = (int)(1.40210 * scale);
	int factor4 = (int)(0.71448 * scale);

  int yImage,uImage,vImage;

  char *p = (char*) (pBuffer+1); // p points to beginning of buffer

  for (int y = 0; y < image.cameraInfo.resolutionHeight; y++)
  {
    for (int x = 0; x < image.cameraInfo.resolutionWidth; x++)
    {
      yImage = scale * image.image[y][0][x];
      uImage = image.image[y][1][x] - 128;
      vImage = image.image[y][2][x] - 128;

      //transformation to rgb
      r = yImage + factor1 * uImage;
      g = yImage - factor2 * vImage - factor4 * uImage;
      b = yImage + factor3 * vImage;

      if(r<0){r=0;} if(r>255 * scale){r=255 * scale;}
      if(g<0){g=0;} if(g>255 * scale){g=255 * scale;}
      if(b<0){b=0;} if(b>255 * scale){b=255 * scale;}

      p = (char*)(pBuffer + 1) 
        + (int)(3 * xSize * (int)(image.image[y][2][x] * 0.75) 
        + 3 * (int)(image.image[y][1][x] * 0.75));
      *p++ = b / scale;
      *p++ = g / scale;
      *p++ = r / scale;
    }
 }
}

void ImageMethods::paint_U_V_ColorSpaceUsage2CDC
(
 CDC& dcMem,
 const Image& image
 )
{
  int xSize = 3*64;
  int ySize = 3*64;

  BITMAPINFOHEADER* pBuffer = (BITMAPINFOHEADER*) new
    char[sizeof(BITMAPINFOHEADER) + ((xSize * 3 + 3) & 0xfffffffc) * ySize];
  pBuffer->biSize          = sizeof(BITMAPINFOHEADER);
  pBuffer->biWidth         = xSize;
  pBuffer->biHeight        = ySize;
  pBuffer->biPlanes        = 1;
  pBuffer->biBitCount      = 24;
  pBuffer->biCompression   = BI_RGB;
  pBuffer->biSizeImage     = 0;
  pBuffer->biXPelsPerMeter = 0;
  pBuffer->biYPelsPerMeter = 0;
  pBuffer->biClrUsed       = 0;
  pBuffer->biClrImportant  = 0;

  paint_U_V_ColorSpaceUsage2CDC(pBuffer, image);
  
  SetDIBitsToDevice(dcMem.m_hDC, 0, 0, xSize, ySize,0,0,0,ySize, pBuffer+1,(BITMAPINFO*) pBuffer,DIB_RGB_COLORS);
  delete [] (char*) pBuffer;
}

void ImageMethods::paint_V_Y_ColorSpaceUsage2CDC
(
 BITMAPINFOHEADER* pBuffer,
 const Image& image
 )
{
  int xSize = 3*64;
  int ySize = 3*64;

  int scale = 16384;
	// needed for transformation to rgb
	int r,g,b;
  // old factors for transformation yuv->rgb
	//int factor1 = (int)(1.140 * scale);
	//int factor2 = (int)(0.394 * scale);
	//int factor3 = (int)(2.028 * scale);
	//int factor4 = (int)(0.581 * scale);
  // new factors for transformation ycbcr->rgb
	int factor1 = (int)(1.77100 * scale);
	int factor2 = (int)(0.34560 * scale);
	int factor3 = (int)(1.40210 * scale);
	int factor4 = (int)(0.71448 * scale);

  int yImage,uImage,vImage;

  char *p = (char*) (pBuffer+1); // p points to beginning of buffer

  for (int y = 0; y < image.cameraInfo.resolutionHeight; y++)
  {
    for (int x = 0; x < image.cameraInfo.resolutionWidth; x++)
    {
      yImage = scale * image.image[y][0][x];
      uImage = image.image[y][1][x] - 128;
      vImage = image.image[y][2][x] - 128;

      //transformation to rgb
      r = yImage + factor1 * uImage;
      g = yImage - factor2 * vImage - factor4 * uImage;
      b = yImage + factor3 * vImage;

      if(r<0){r=0;} if(r>255 * scale){r=255 * scale;}
      if(g<0){g=0;} if(g>255 * scale){g=255 * scale;}
      if(b<0){b=0;} if(b>255 * scale){b=255 * scale;}

      p = (char*)(pBuffer + 1) 
        + (int)(3 * xSize * (int)(image.image[y][0][x] * 0.75) 
        + 3 * (int)(image.image[y][2][x] * 0.75));

      *p++ = b / scale;
      *p++ = g / scale;
      *p++ = r / scale;
    }
 }
}

void ImageMethods::paint_V_Y_ColorSpaceUsage2CDC
(
 CDC& dcMem,
 const Image& image
 )
{
  int xSize = 3*64;
  int ySize = 3*64;

  BITMAPINFOHEADER* pBuffer = (BITMAPINFOHEADER*) new
    char[sizeof(BITMAPINFOHEADER) + ((xSize * 3 + 3) & 0xfffffffc) * ySize];
  pBuffer->biSize          = sizeof(BITMAPINFOHEADER);
  pBuffer->biWidth         = xSize;
  pBuffer->biHeight        = ySize;
  pBuffer->biPlanes        = 1;
  pBuffer->biBitCount      = 24;
  pBuffer->biCompression   = BI_RGB;
  pBuffer->biSizeImage     = 0;
  pBuffer->biXPelsPerMeter = 0;
  pBuffer->biYPelsPerMeter = 0;
  pBuffer->biClrUsed       = 0;
  pBuffer->biClrImportant  = 0;

  paint_V_Y_ColorSpaceUsage2CDC(pBuffer, image);
  
  SetDIBitsToDevice(dcMem.m_hDC, 0, 0, xSize, ySize,0,0,0,ySize, pBuffer+1,(BITMAPINFO*) pBuffer,DIB_RGB_COLORS);
  delete [] (char*) pBuffer;
}

void ImageMethods::paint_U_Y_ColorSpaceUsage2CDC
(
 BITMAPINFOHEADER* pBuffer,
 const Image& image
 )
{
  int xSize = 3*64;
  int ySize = 3*64;

  int scale = 16384;
	// needed for transformation to rgb
	int r,g,b;
  // old factors for transformation yuv->rgb
	//int factor1 = (int)(1.140 * scale);
	//int factor2 = (int)(0.394 * scale);
	//int factor3 = (int)(2.028 * scale);
	//int factor4 = (int)(0.581 * scale);
  // new factors for transformation ycbcr->rgb
	int factor1 = (int)(1.77100 * scale);
	int factor2 = (int)(0.34560 * scale);
	int factor3 = (int)(1.40210 * scale);
	int factor4 = (int)(0.71448 * scale);

  int yImage,uImage,vImage;

  char *p; // pointer for navigation in the bitmap

  for (int y = 0; y < image.cameraInfo.resolutionHeight; y++)
  {
    for (int x = 0; x < image.cameraInfo.resolutionWidth; x++)
    {
      yImage = scale * image.image[y][0][x];
      uImage = image.image[y][1][x] - 128;
      vImage = image.image[y][2][x] - 128;

      //transformation to rgb
      r = yImage + factor1 * uImage;
      g = yImage - factor2 * vImage - factor4 * uImage;
      b = yImage + factor3 * vImage;

      if(r<0){r=0;} if(r>255 * scale){r=255 * scale;}
      if(g<0){g=0;} if(g>255 * scale){g=255 * scale;}
      if(b<0){b=0;} if(b>255 * scale){b=255 * scale;}

      p = (char*)(pBuffer + 1) 
        + (int)(3 * xSize * (int)((image.image[y][0][x]) * 0.75) 
        + 3 * (int)(image.image[y][1][x] * 0.75));

      *p++ = b / scale;
      *p++ = g / scale;
      *p++ = r / scale;
    }
 }
}

void ImageMethods::paint_U_Y_ColorSpaceUsage2CDC
(
 CDC& dcMem,
 const Image& image
 )
{
  int xSize = 3*64;
  int ySize = 3*64;

  BITMAPINFOHEADER* pBuffer = (BITMAPINFOHEADER*) new
    char[sizeof(BITMAPINFOHEADER) + ((xSize * 3 + 3) & 0xfffffffc) * ySize];
  pBuffer->biSize          = sizeof(BITMAPINFOHEADER);
  pBuffer->biWidth         = xSize;
  pBuffer->biHeight        = ySize;
  pBuffer->biPlanes        = 1;
  pBuffer->biBitCount      = 24;
  pBuffer->biCompression   = BI_RGB;
  pBuffer->biSizeImage     = 0;
  pBuffer->biXPelsPerMeter = 0;
  pBuffer->biYPelsPerMeter = 0;
  pBuffer->biClrUsed       = 0;
  pBuffer->biClrImportant  = 0;

  paint_U_Y_ColorSpaceUsage2CDC(pBuffer, image);
    
  SetDIBitsToDevice(dcMem.m_hDC, 0, 0, xSize, ySize,0,0,0,ySize, pBuffer+1,(BITMAPINFO*) pBuffer,DIB_RGB_COLORS);
  delete [] (char*) pBuffer;
}

void ImageMethods::writeRobotControlLogoToImage(Image& image)
{
//  int ttt = rand() * 255 / RAND_MAX;
  for (int line = 0;line < image.cameraInfo.resolutionHeight; line++)
  {
    for (int column = 0; column < image.cameraInfo.resolutionWidth; column++)
    {
      image.image[line][0][column] = 0;
      image.image[line][1][column] = 127;
      image.image[line][2][column] = 127;
      if(
        (
        (line == 4 || line == 5)
        && (column >= 22)
        && (column <= 29)
        ) ||
        
        (
        (line == 6 || line == 7)
        && (column >= 2)
        && (column <= 7)
        ) ||
        
        (
        (line == 10 || line == 11)
        && (column >= 2)
        && (column <= 29)
        ) ||

        (
        (line == 18 || line == 19)
        && (column >= 6)
        && (column <= 23)
        ) ||

        (
        (line == 26 || line == 27)
        && (column >= 6)
        && (column <= 11)
        ) ||

        (
        (line == 26 || line == 27)
        && (column >= 18)
        && (column <= 23)
        ) ||

        (
        (column == 2 || column == 3)
        && (line >= 6)
        && (line <= 11)
        ) ||

        (
        (column == 6 || column == 7)
        && (line >= 6)
        && (line <= 27)
        ) ||

        (
        (column == 10 || column == 11)
        && (line >= 18)
        && (line <= 27)
        ) ||

        (
        (column == 18 || column == 19)
        && (line >= 18)
        && (line <= 27)
        ) ||

        (
        (column == 22 || column == 23)
        && (line >= 4)
        && (line <= 27)
        ) ||

        (
        (column == 28 || column == 29)
        && (line >= 4)
        && (line <= 11)
        ) 

        )
      {
        image.image[line][0][column] = 255;
        image.image[line][1][column] = 127;
        image.image[line][2][column] = 127;
      }
    }
  }
}

/*
 * Change log :
 * 
 * $Log: ImageMethods.cpp,v $
 * Revision 1.5  2004/04/28 13:24:25  thomas
 * modified: conversion from yuv now uses ycbcr-transformation
 *
 * Revision 1.4  2004/04/13 18:11:31  risler
 * corrected rgb variable naming (to stop confusing people)
 *
 * Revision 1.3  2004/04/07 13:00:44  risler
 * ddd checkin after go04 - second part
 *
 * Revision 1.4  2004/04/06 13:19:38  risler
 * cleaned up and improved high resolution image support
 *
 * Revision 1.3  2004/03/29 20:45:40  risler
 * idBWImage now draws hi-res image
 *
 * Revision 1.2  2004/03/29 15:19:03  Marc
 * Intruduced the Black and White Image
 * Normal Images (not Jpeg) images were now send as Color Image with BW
 *
 * Revision 1.2  2003/12/15 11:46:59  juengel
 * Introduced CameraInfo
 *
 * Revision 1.1  2003/10/07 10:11:08  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1.1.1  2003/07/02 09:40:26  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.6  2003/05/11 23:32:41  dueffert
 * Depend now works with RobotControl too
 *
 * Revision 1.5  2003/03/23 19:12:10  loetzsch
 * OUTPUT not allowed in the RobotControl thread anymore.
 * Use getQueues().toGUI.out instead.
 *
 * Revision 1.4  2003/03/20 14:06:51  roefer
 * Localization experiment and view for segmented image added
 *
 * Revision 1.3  2002/12/15 23:33:02  dueffert
 * rgb-yuv-convertion moved to Image
 *
 * Revision 1.2  2002/09/20 23:28:34  juengel
 * Moved instance of DebugDrawingManager to RobotControlMainFrame.
 *
 * Revision 1.1  2002/09/10 15:49:12  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 *
 * Revision 1.2  2002/07/23 13:45:44  loetzsch
 * - new streaming classes
 * - removed many #include statements
 * - 5 instead of 3 debug queues in RobotControl
 * - new debug message handling
 * - access to debugkeytables and queues via RobotControlQueues.h and RobotControlDebugKeyTables.h
 * - general clean up
 *
 */
