/** 
* @file OpenGLMethods.cpp
* Implementation of class OpenGLMethods.
*
* @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Jngel</A>
*/
#include "StdAfx.h"
#include "OpenGLMethods.h"
#include "Visualization/ImageMethods.h"
#include "Tools/Math/Geometry.h"

#include "Tools/Actorics/Kinematics.h"

#include <gl/gl.h>
#include <gl/glu.h>

BOOL OpenGLMethods::setWindowPixelFormat(HDC hDC)
{
  PIXELFORMATDESCRIPTOR pixelDesc;
  
  pixelDesc.nSize = sizeof(PIXELFORMATDESCRIPTOR);
  pixelDesc.nVersion = 1;
  
  pixelDesc.dwFlags = PFD_DRAW_TO_WINDOW | 
    PFD_SUPPORT_OPENGL |
    PFD_DOUBLEBUFFER |
    PFD_STEREO_DONTCARE;
  
  pixelDesc.iPixelType = PFD_TYPE_RGBA;
  pixelDesc.cColorBits = 32;
  pixelDesc.cRedBits = 8;
  pixelDesc.cRedShift = 16;
  pixelDesc.cGreenBits = 8;
  pixelDesc.cGreenShift = 8;
  pixelDesc.cBlueBits = 8;
  pixelDesc.cBlueShift = 0;
  pixelDesc.cAlphaBits = 0;
  pixelDesc.cAlphaShift = 0;
  pixelDesc.cAccumBits = 64;
  pixelDesc.cAccumRedBits = 16;
  pixelDesc.cAccumGreenBits = 16;
  pixelDesc.cAccumBlueBits = 16;
  pixelDesc.cAccumAlphaBits = 0;
  pixelDesc.cDepthBits = 32;
  pixelDesc.cStencilBits = 8;
  pixelDesc.cAuxBuffers = 0;
  pixelDesc.iLayerType = PFD_MAIN_PLANE;
  pixelDesc.bReserved = 0;
  pixelDesc.dwLayerMask = 0;
  pixelDesc.dwVisibleMask = 0;
  pixelDesc.dwDamageMask = 0;
  
  int glPixelIndex = ChoosePixelFormat(hDC,&pixelDesc);
  if(glPixelIndex==0) // Choose default
  {
    glPixelIndex = 1;
    if(!DescribePixelFormat(hDC,glPixelIndex,
      sizeof(PIXELFORMATDESCRIPTOR),&pixelDesc))
      return FALSE;
  }
  
  if(!::SetPixelFormat(hDC,glPixelIndex,&pixelDesc))
    return FALSE;
  
  return TRUE;
}


BOOL OpenGLMethods::createViewGLContext(HDC hDC, HGLRC& hGLContext)
{
  hGLContext = wglCreateContext(hDC);
  
  if(hGLContext==NULL)
    return FALSE;
  
  if(wglMakeCurrent(hDC,hGLContext)==FALSE)
    return FALSE;
  return TRUE;
}


void OpenGLMethods::paintSphere(float* center, float radius, int segments)
{ 
  int i,j;
  double theta1,theta2,theta3;
  float e[3], p[3];
  
  if (radius < 0)
    radius = -radius;
  if (segments < 0)
    segments = -segments;
  if (segments < 4 || radius <= 0) {
    glBegin(GL_POINTS);
    glVertex3fv(center);
    glEnd();
    return;
  }
  
  for (j = 0; j < segments/2; j++) 
  {
    theta1 = j * pi2 / segments - pi_2;
    theta2 = (j + 1) * pi2 / segments - pi_2;
    
    glBegin(GL_QUAD_STRIP);
    for (i = 0; i <= segments; i++) 
    {
      theta3 = i * pi2 / segments;
      
      e[0] = (float )(cos(theta2) * cos(theta3));
      e[1] = (float )sin(theta2);
      e[2] = (float )(-cos(theta2) * sin(theta3));
      p[0] = (float )(center[0] + radius * e[0]);
      p[1] = (float )(center[1] + radius * e[1]);
      p[2] = (float )(center[2] + radius * e[2]);
      
      glNormal3fv(e);
      glTexCoord2f(i/(float )segments, 2*(j + 1)/(float )segments);
      glVertex3fv(p);
      
      e[0] = (float )(cos(theta1) * cos(theta3));
      e[1] = (float )(sin(theta1));
      e[2] = (float )(-cos(theta1) * sin(theta3));
      p[0] = (float )(center[0] + radius * e[0]);
      p[1] = (float )(center[1] + radius * e[1]);
      p[2] = (float )(center[2] + radius * e[2]);
      
      glNormal3fv(e);
      glTexCoord2f(i/(float )segments,2*j/(float )segments);
      glVertex3fv(p);
    }
    glEnd();
  }
}

void OpenGLMethods::paintCubeToOpenGLList
(
 int xSize, int ySize, int zSize,
 int listID,
 bool paintFullCube,
 int scale,
 int offsetX,int offsetY,int offsetZ,
 int r, int g, int b
 )
{
  paintCubeToOpenGLList(0,0,0, xSize, ySize, zSize, listID, paintFullCube, scale, offsetX, offsetY, offsetZ, r, g, b);
}

double OpenGLMethods::transformCoordinates
(
 int scale,
 int offset,
 int value
 )
{
  return (value + offset) / (double)scale;
}

void OpenGLMethods::paintCubeToOpenGLList
(
 int xLow, int yLow, int zLow, 
 int xHigh, int yHigh, int zHigh,
 int listID, bool paintFullCube,
 int scale,
 int offsetX,int offsetY,int offsetZ,
 int r, int g, int b,
 bool beginList, bool endList
 )
{
  GLUquadricObj* pQuadric = gluNewQuadric();
  
  if(beginList) glNewList(listID, GL_COMPILE_AND_EXECUTE);
  
  double xL, yL, zL;
  double xH, yH, zH;
  
  xH = OpenGLMethods::transformCoordinates(scale, offsetX, xHigh);
  yH = OpenGLMethods::transformCoordinates(scale, offsetY, yHigh);
  zH = OpenGLMethods::transformCoordinates(scale, offsetZ, zHigh);
  xL = OpenGLMethods::transformCoordinates(scale, offsetX, xLow);
  yL = OpenGLMethods::transformCoordinates(scale, offsetY, yLow);
  zL = OpenGLMethods::transformCoordinates(scale, offsetZ, zLow);
  
  glBegin(GL_LINE_LOOP);
  glColor3ub(r,g,b);
  glVertex3d(xL,yL,zL);
  glVertex3d(xH,yL,zL);
  glVertex3d(xH,yH,zL);
  glVertex3d(xL,yH,zL);
  glEnd();
  
  if(paintFullCube)
  {
    glBegin(GL_LINE_LOOP);
    glVertex3d(xL,yL,zH);
    glVertex3d(xH,yL,zH);
    glVertex3d(xH,yH,zH);
    glVertex3d(xL,yH,zH);
    glEnd();
    
    glBegin(GL_LINES);
    
    glVertex3d(xL,yL,zL);
    glVertex3d(xL,yL,zH);
    glVertex3d(xH,yL,zL);
    glVertex3d(xH,yL,zH);
    glVertex3d(xH,yH,zL);
    glVertex3d(xH,yH,zH);
    glVertex3d(xL,yH,zL);
    glVertex3d(xL,yH,zH);
    
    glEnd();
  }
  if(endList) glEndList();
  
  gluDeleteQuadric(pQuadric);
}

void OpenGLMethods::paintCoordinateSystem(int listID)
{
  GLUquadricObj* pQuadric = gluNewQuadric();
  
  glNewList(listID, GL_COMPILE_AND_EXECUTE);
  
  double maxX =  1600;
  double minX =  -300;
  double maxY =  1600;
  double minY = -1600;
  double maxZ =   400;
  double minZ =  -400;
  
  double scale = 0.0015;
  
  glBegin(GL_LINES);
  
  
  // x-y grid
  for (int x=(int)minX / 100; x<= (int)maxX / 100; x++)
  {
    if (x!=0)
    {
      if(x%5 == 0) glColor3ub(255,255,255);
      else glColor3ub(150,150,150);
      glVertex3d(x*100*scale,minY*scale,0);
      glVertex3d(x*100*scale,maxY*scale,0);
    }
  }
  
  for (int y=(int)minY / 100; y<= (int)maxY / 100; y++)
  {
    if (y!=0)
    {
      if(y%5 == 0) glColor3ub(255,255,255);
      else glColor3ub(150,150,150);
      glVertex3d(minX*scale,y*100*scale,0);
      glVertex3d(maxX*scale,y*100*scale,0);
    }
  }
  
  glColor3ub(250,250,250);
  maxX += 200; minX -= 100;
  maxY += 200; minY -= 100;
  maxZ += 200; minZ -= 100;
  // x axis
  glVertex3d(maxX*scale,0,0);
  glVertex3d(minX*scale,0,0);
  glVertex3d((maxX-50)*scale,0,25*scale);
  glVertex3d(maxX*scale,0,0);
  glVertex3d((maxX-50)*scale,0,-25*scale);
  glVertex3d(maxX*scale,0,0);
  glVertex3d((maxX-50)*scale,25*scale,0);
  glVertex3d(maxX*scale,0,0);
  glVertex3d((maxX-50)*scale,-25*scale,0);
  glVertex3d(maxX*scale,0,0);
  
  // y axis
  glVertex3d(0,maxY*scale,0);
  glVertex3d(0,minY*scale,0);
  glVertex3d(0,(maxY-50)*scale,25*scale);
  glVertex3d(0,maxY*scale,0);
  glVertex3d(0,(maxY-50)*scale,-25*scale);
  glVertex3d(0,maxY*scale,0);
  glVertex3d(25*scale,(maxY-50)*scale,0);
  glVertex3d(0,maxY*scale,0);
  glVertex3d(-25*scale,(maxY-50)*scale,0);
  glVertex3d(0,maxY*scale,0);
  
  // z axis
  glVertex3d(0,0,maxZ*scale);
  glVertex3d(0,0,minZ*scale);
  glVertex3d(0,25*scale,(maxZ-50)*scale);
  glVertex3d(0,0,maxZ*scale);
  glVertex3d(0,-25*scale,(maxZ-50)*scale);
  glVertex3d(0,0,maxZ*scale);
  glVertex3d(25*scale,0,(maxZ-50)*scale);
  glVertex3d(0,0,maxZ*scale);
  glVertex3d(-25*scale,0,(maxZ-50)*scale);
  glVertex3d(0,0,maxZ*scale);
  
  
  glEnd();
  
  glEndList();
  
  gluDeleteQuadric(pQuadric);
}

void OpenGLMethods::paintPoints3DToOpenGLList(const Points3D& points,int listID)
{
  GLUquadricObj* pQuadric = gluNewQuadric();
  
  glNewList(listID, GL_COMPILE_AND_EXECUTE);
  
  double scale = 0.0015;
  
  glBegin(GL_LINES);
  
  
  // x-y grid
  glColor3ub(255,0,0);
  
  for (int i=0; i<points.getNumberOfEntries(); i++)
  {
    glColor3ub(255,255,255);
    glVertex3d(points[i].x*scale, points[i].y*scale, 0);
    if (points[i].z < 0) glColor3ub(0,255,0);
    else glColor3ub(255,0,0);
    glVertex3d(points[i].x*scale, points[i].y*scale, points[i].z*scale);
  }
  
  glEnd();
  
  glEndList();
  
  gluDeleteQuadric(pQuadric);
}

void OpenGLMethods::paintLinesPerceptToOpenGLList(const LinesPercept& linesPercept,int listID)
{
  GLUquadricObj* pQuadric = gluNewQuadric();
  
  glNewList(listID, GL_COMPILE_AND_EXECUTE);
  
  double scale = 0.0015;
  
  unsigned char r,g,b;
  
  for(int i = 0; i < LinesPercept::numberOfLineTypes; ++i)
  {
    switch(LinesPercept::LineType(i))
    {
    case LinesPercept::field: r = 127; g = 127; b = 127; break;
      //    case LinesPercept::sideOfPenaltyArea: r = 0; g = 0; b = 255; break;
    case LinesPercept::border: r = 0; g = 255; b = 0; break;
    case LinesPercept::yellowGoal: r = 255; g = 255; b = 0; break;
    case LinesPercept::skyblueGoal: r = 0; g = 255; b = 255; break;
    }
    glPolygonMode(GL_FRONT,GL_FILL);
    glPolygonMode(GL_BACK,GL_FILL);
    glBegin(GL_QUADS);
    glColor3ub(r,g,b);
    for(int j = 0; j < linesPercept.numberOfPoints[i]; j++)
    {
      glVertex3d(linesPercept.points[i][j].x*scale - 0.02, linesPercept.points[i][j].y*scale - 0.02, 0.03);
      glVertex3d(linesPercept.points[i][j].x*scale - 0.02, linesPercept.points[i][j].y*scale + 0.02, 0.03);
      glVertex3d(linesPercept.points[i][j].x*scale + 0.02, linesPercept.points[i][j].y*scale + 0.02, 0.03);
      glVertex3d(linesPercept.points[i][j].x*scale + 0.02, linesPercept.points[i][j].y*scale - 0.02, 0.03);
    }
    glEnd();
    glPolygonMode(GL_FRONT,GL_LINE);
    glPolygonMode(GL_BACK,GL_LINE);
  }
  
  
  glEndList();
  
  gluDeleteQuadric(pQuadric);
}

void OpenGLMethods::paintObstaclesPerceptToOpenGLList(const ObstaclesPercept& obstaclesPercept, int listID)
{
  GLUquadricObj* pQuadric = gluNewQuadric();
  
  glNewList(listID, GL_COMPILE_AND_EXECUTE);
  fireGLHack();
  
  double scale = 0.0015;
  
  glColor3ub(127,255,127);
  for(int j = 0; j < obstaclesPercept.numberOfPoints; j++)
  {
    glBegin(GL_LINES);
    glVertex3d(obstaclesPercept.nearPoints[j].x*scale, obstaclesPercept.nearPoints[j].y*scale, 0.02);
    glVertex3d(obstaclesPercept.farPoints[j].x*scale, obstaclesPercept.farPoints[j].y*scale, 0.02);
    glEnd();
    
    if(obstaclesPercept.farPointIsOnImageBorder[j])
    {
      glPolygonMode(GL_FRONT,GL_FILL);
      glPolygonMode(GL_BACK,GL_FILL);
      glBegin(GL_QUADS);
      glVertex3d(obstaclesPercept.farPoints[j].x*scale - 0.03, obstaclesPercept.farPoints[j].y*scale - 0.03, 0.02);
      glVertex3d(obstaclesPercept.farPoints[j].x*scale - 0.03, obstaclesPercept.farPoints[j].y*scale + 0.03, 0.02);
      glVertex3d(obstaclesPercept.farPoints[j].x*scale + 0.03, obstaclesPercept.farPoints[j].y*scale + 0.03, 0.02);
      glVertex3d(obstaclesPercept.farPoints[j].x*scale + 0.03, obstaclesPercept.farPoints[j].y*scale - 0.03, 0.02);
      glEnd();
      glPolygonMode(GL_FRONT,GL_LINE);
      glPolygonMode(GL_BACK,GL_LINE);
    }
  }
  
  glEndList();
  gluDeleteQuadric(pQuadric);
}

void OpenGLMethods::paintObstaclesModelToOpenGLList(const ObstaclesModel& obstaclesModel, int listID)
{
  GLUquadricObj* pQuadric = gluNewQuadric();
  
  glNewList(listID, GL_COMPILE_AND_EXECUTE);
  fireGLHack();
  
  double scale = 0.0015;
  
  Vector2<double> point, point2, lastPoint;
  double angle, openingAngle;
  angle = ObstaclesModel::getAngleOfSector(ObstaclesModel::numOfSectors-1);
  openingAngle = tan(pi / ObstaclesModel::numOfSectors);
  
  lastPoint.x = cos(angle) * obstaclesModel.distance[ObstaclesModel::numOfSectors-1]
    - sin(angle) * obstaclesModel.distance[ObstaclesModel::numOfSectors-1]*openingAngle;
  lastPoint.y = sin(angle) * obstaclesModel.distance[ObstaclesModel::numOfSectors-1]
    + cos(angle) * obstaclesModel.distance[ObstaclesModel::numOfSectors-1]*openingAngle;
  
  glColor3ub(255,160,0);
  glBegin(GL_LINES);
  for(int i = 0; i < ObstaclesModel::numOfSectors; i++)
  {
    angle = ObstaclesModel::getAngleOfSector(i);
    
    point.x = cos(angle) * obstaclesModel.distance[i]
      - sin(angle) * obstaclesModel.distance[i]*openingAngle;
    point.y = sin(angle) * obstaclesModel.distance[i]
      + cos(angle) * obstaclesModel.distance[i]*openingAngle;
    
    point2.x = cos(angle) * obstaclesModel.distance[i]
      + sin(angle) * obstaclesModel.distance[i]*openingAngle;
    point2.y = sin(angle) * obstaclesModel.distance[i]
      - cos(angle) * obstaclesModel.distance[i]*openingAngle;
    
    glVertex3d(point2.x * scale, point2.y*scale, 0.05);
    glVertex3d(point.x*scale, point.y*scale, 0.05);
    
    glVertex3d(point2.x * scale, point2.y*scale, 0.05);
    glVertex3d(lastPoint.x*scale, lastPoint.y*scale, 0.05);
    
    lastPoint = point;
  }
  glEnd();
  
  glEndList();
  gluDeleteQuadric(pQuadric);
}

void OpenGLMethods::paintSensorDataToOpenGLList(const SensorData& sensorData, int listID)
{
/*  GLUquadricObj* pQuadric = gluNewQuadric();
  
  glNewList(listID, GL_COMPILE_AND_EXECUTE);
  fireGLHack();*/
  
  double scale = 0.0015;

  int green = 50;
  int greenDiff = 40;
  glColor3ub(0,green,0);
  glBegin(GL_LINES);

  double tilt = fromMicroRad(sensorData.data[SensorData::headTilt]);
  double pan = fromMicroRad(sensorData.data[SensorData::headPan]);
  double roll = fromMicroRad(sensorData.data[SensorData::headRoll]);

  HeadState headState;
  headState.bodyRoll = 0;
  headState.bodyTilt = fromDegrees(9);
  headState.neckHeight = 100;

  RobotDimensionsERS7 r;
  Vector3<double> testVectorCamera(100,0,0);
  Vector3<double> testVectorRobot;

  CameraMatrix cameraMatrix;

  glColor3ub(0,green,0); green += greenDiff;
  testVectorRobot = cameraMatrix.rotation * testVectorCamera;
  glVertex3d(cameraMatrix.translation.x * scale, cameraMatrix.translation.y * scale, cameraMatrix.translation.z * scale);
  glVertex3d((cameraMatrix.translation.x + testVectorRobot.x) * scale, (cameraMatrix.translation.y + testVectorRobot.y) * scale, (cameraMatrix.translation.z + testVectorRobot.z) * scale );

  (Pose3D&)cameraMatrix =
    Pose3D(0,0,headState.neckHeight);
  
  glColor3ub(0,green,0); green += greenDiff;
  testVectorRobot = cameraMatrix.rotation * testVectorCamera;
  glVertex3d(cameraMatrix.translation.x * scale, cameraMatrix.translation.y * scale, cameraMatrix.translation.z * scale);
  glVertex3d((cameraMatrix.translation.x + testVectorRobot.x) * scale, (cameraMatrix.translation.y + testVectorRobot.y) * scale, (cameraMatrix.translation.z + testVectorRobot.z) * scale );

  cameraMatrix = cameraMatrix.rotateX(headState.bodyRoll).rotateY(headState.bodyTilt-tilt).translate(0,0,r.distanceNeckToPanCenter);

  glColor3ub(0,green,0); green += greenDiff;
  testVectorRobot = cameraMatrix.rotation * testVectorCamera;
  glVertex3d(cameraMatrix.translation.x * scale, cameraMatrix.translation.y * scale, cameraMatrix.translation.z * scale);
  glVertex3d((cameraMatrix.translation.x + testVectorRobot.x) * scale, (cameraMatrix.translation.y + testVectorRobot.y) * scale, (cameraMatrix.translation.z + testVectorRobot.z) * scale );
  
  cameraMatrix = cameraMatrix.rotateZ(pan);

  glColor3ub(0,green,0); green += greenDiff;
  testVectorRobot = cameraMatrix.rotation * testVectorCamera;
  glVertex3d(cameraMatrix.translation.x * scale, cameraMatrix.translation.y * scale, cameraMatrix.translation.z * scale);
  glVertex3d((cameraMatrix.translation.x + testVectorRobot.x) * scale, (cameraMatrix.translation.y + testVectorRobot.y) * scale, (cameraMatrix.translation.z + testVectorRobot.z) * scale );
    
  cameraMatrix.rotateY(-roll);

  glColor3ub(0,green,0); green += greenDiff;
  testVectorRobot = cameraMatrix.rotation * testVectorCamera;
  glVertex3d(cameraMatrix.translation.x * scale, cameraMatrix.translation.y * scale, cameraMatrix.translation.z * scale);
  glVertex3d((cameraMatrix.translation.x + testVectorRobot.x) * scale, (cameraMatrix.translation.y + testVectorRobot.y) * scale, (cameraMatrix.translation.z + testVectorRobot.z) * scale );
    
  cameraMatrix.translate(r.distancePanCenterToCameraX,0,-r.distancePanCenterToCameraZ);

  glColor3ub(0,green,0); green += greenDiff;
  testVectorRobot = cameraMatrix.rotation * testVectorCamera;
  glVertex3d(cameraMatrix.translation.x * scale, cameraMatrix.translation.y * scale, cameraMatrix.translation.z * scale);
  glVertex3d((cameraMatrix.translation.x + testVectorRobot.x) * scale, (cameraMatrix.translation.y + testVectorRobot.y) * scale, (cameraMatrix.translation.z + testVectorRobot.z) * scale );
  
  glEnd();
  
/*  glEndList();
  gluDeleteQuadric(pQuadric);*/
}

void OpenGLMethods::paintJointDataToOpenGLList(const RingBuffer<JointData, 200>& jointDataRingBuffer, int listID)
{
  GLUquadricObj* pQuadric = gluNewQuadric();
  
  glNewList(listID, GL_COMPILE_AND_EXECUTE);
  fireGLHack();
  
  double scale = 0.0015;
  
  glColor3ub(255,0,0);
  glBegin(GL_LINES);
  
  RobotVertices robotVertices[200];

  for(int i = 0; i < jointDataRingBuffer.getNumberOfEntries(); i++)
  {
//    Kinematics::getAbsoluteRobotVertices(robotVertices[i],Kinematics::dynamicMode,jointDataRingBuffer[i]);
    Kinematics::getAbsoluteRobotVertices(robotVertices[i],Kinematics::flyMode, jointDataRingBuffer[i],0.266);
  }

  //calculate robot plane
  RobotVertices relativeRobotVertices;
  double tilt, roll;
  Kinematics::getRelativeRobotVertices(relativeRobotVertices, jointDataRingBuffer[0]);
  Kinematics::getRobotTransformation(tilt, roll, relativeRobotVertices, Kinematics::dynamicMode, 0, 0);
  Vector3<double> front( 100.0 * cos(tilt), 0,-100.0 * sin(tilt));
  Vector3<double> back (-100.0 * cos(tilt), 0, 100.0 * sin(tilt));
  Vector3<double> left ( 0, 100.0 * cos(roll), 100.0 * sin(roll));
  Vector3<double> right( 0,-100.0 * cos(roll),-100.0 * sin(roll));

  front += robotVertices[0].neck;
  back += robotVertices[0].neck;
  left += robotVertices[0].neck;
  right += robotVertices[0].neck;


  glColor3ub(127,127,255);
  glVertex3d(front.x * scale, front.y * scale, front.z * scale);
  glVertex3d(back.x * scale, back.y * scale, back.z * scale);
  glVertex3d(left.x * scale, left.y * scale, left.z * scale);
  glVertex3d(right.x * scale, right.y * scale, right.z * scale);

  for(i = 1; i < jointDataRingBuffer.getNumberOfEntries(); i++)
  {
    for(int legIndex = 0; legIndex < 4; legIndex++)
    {
      glVertex3d(robotVertices[i-1].footPosition[legIndex].x * scale, robotVertices[i-1].footPosition[legIndex].y * scale, (robotVertices[i-1].footPosition[legIndex].z) * scale);
      glVertex3d(robotVertices[i].footPosition[legIndex].x * scale,   robotVertices[i].footPosition[legIndex].y * scale,   (robotVertices[i].footPosition[legIndex].z) * scale);
      glVertex3d(robotVertices[i-1].kneePosition[legIndex].x * scale, robotVertices[i-1].kneePosition[legIndex].y * scale, (robotVertices[i-1].kneePosition[legIndex].z) * scale);
      glVertex3d(robotVertices[i].kneePosition[legIndex].x * scale,   robotVertices[i].kneePosition[legIndex].y * scale,   (robotVertices[i].kneePosition[legIndex].z) * scale);
    }
  }
  
  glColor3ub(127,0,0);
  glVertex3d( robotVertices[0].neck.x * scale, robotVertices[0].neck.y * scale, robotVertices[0].neck.z * scale);
  glVertex3d( 100 * scale, 100 * scale, 0 * scale);
  for(int legIndex = 0; legIndex < 4; legIndex++)
  {
    glVertex3d(  robotVertices[0].shoulderPosition[legIndex].x * scale,robotVertices[0].shoulderPosition[legIndex].y * scale, robotVertices[0].shoulderPosition[legIndex].z * scale);
    glVertex3d(  robotVertices[0].kneePosition[legIndex].x * scale,    robotVertices[0].kneePosition[legIndex].y * scale,     robotVertices[0].kneePosition[legIndex].z * scale);
    glVertex3d(  robotVertices[0].kneePosition[legIndex].x * scale,    robotVertices[0].kneePosition[legIndex].y * scale,     robotVertices[0].kneePosition[legIndex].z * scale);
    glVertex3d(  robotVertices[0].footPosition[legIndex].x * scale,    robotVertices[0].footPosition[legIndex].y * scale,     (robotVertices[0].footPosition[legIndex].z) * scale);
  }
  
  glEnd();
  
  glEndList();
  gluDeleteQuadric(pQuadric);
}

void OpenGLMethods::paintSensorDataToOpenGLList(const RingBuffer<SensorData, 200>& sensorDataRingBuffer, int listID)
{
  GLUquadricObj* pQuadric = gluNewQuadric();
  
  glNewList(listID, GL_COMPILE_AND_EXECUTE);
  fireGLHack();
  paintSensorDataToOpenGLList(sensorDataRingBuffer[0], listID);

  double scale = 0.0015;
  
  RobotVertices robotVertices[200];

  for(int i = 0; i < sensorDataRingBuffer.getNumberOfEntries(); i++)
  {
    JointData jointData;
    for(int joint = 0; joint < 3; ++joint)
    {
      const SensorData& s = sensorDataRingBuffer[i];
      jointData.data[JointData::legFL1 + joint] = s.data[SensorData::legFL1 + joint];
      jointData.data[JointData::legFR1 + joint] = s.data[SensorData::legFR1 + joint];
      jointData.data[JointData::legHL1 + joint] = s.data[SensorData::legHL1 + joint];
      jointData.data[JointData::legHR1 + joint] = s.data[SensorData::legHR1 + joint];
    }
    Kinematics::getAbsoluteRobotVertices(robotVertices[i],Kinematics::flyMode, jointData,0.266);
  }

  glColor3ub(0,255,0);
  glBegin(GL_LINES);
  for(i = 1; i < sensorDataRingBuffer.getNumberOfEntries(); i++)
    for(int legIndex = 0; legIndex < 4; legIndex++)
      if(legIndex == 2 && sensorDataRingBuffer[i].data[SensorData::pawHR] && sensorDataRingBuffer[i-1].data[SensorData::pawHR] ||
         legIndex == 3 && sensorDataRingBuffer[i].data[SensorData::pawHL] && sensorDataRingBuffer[i-1].data[SensorData::pawHL])
      {
        glVertex3d(robotVertices[i-1].footPosition[legIndex].x * scale, robotVertices[i-1].footPosition[legIndex].y * scale, (robotVertices[i-1].footPosition[legIndex].z) * scale);
        glVertex3d(robotVertices[i].footPosition[legIndex].x * scale,   robotVertices[i].footPosition[legIndex].y * scale,   (robotVertices[i].footPosition[legIndex].z) * scale);
      }
  glEnd();

  glColor3ub(0,127,0);
  glBegin(GL_LINES);
  for(i = 1; i < sensorDataRingBuffer.getNumberOfEntries(); i++)
    for(int legIndex = 0; legIndex < 4; legIndex++)
    {
      if(!(legIndex == 2 && sensorDataRingBuffer[i].data[SensorData::pawHR] && sensorDataRingBuffer[i-1].data[SensorData::pawHR] ||
          legIndex == 3 && sensorDataRingBuffer[i].data[SensorData::pawHL] && sensorDataRingBuffer[i-1].data[SensorData::pawHL]))
      {
        glVertex3d(robotVertices[i-1].footPosition[legIndex].x * scale, robotVertices[i-1].footPosition[legIndex].y * scale, (robotVertices[i-1].footPosition[legIndex].z) * scale);
        glVertex3d(robotVertices[i].footPosition[legIndex].x * scale,   robotVertices[i].footPosition[legIndex].y * scale,   (robotVertices[i].footPosition[legIndex].z) * scale);
      }
      glVertex3d(robotVertices[i-1].kneePosition[legIndex].x * scale, robotVertices[i-1].kneePosition[legIndex].y * scale, (robotVertices[i-1].kneePosition[legIndex].z) * scale);
      glVertex3d(robotVertices[i].kneePosition[legIndex].x * scale,   robotVertices[i].kneePosition[legIndex].y * scale,   (robotVertices[i].kneePosition[legIndex].z) * scale);
    }
  glEnd();
  
  glEndList();
  gluDeleteQuadric(pQuadric);
}

void OpenGLMethods::paintLandmarksPerceptToOpenGLList
(
 const CameraMatrix& cameraMatrix, 
 const LandmarksPercept& landmarksPercept,
 int listID)
{
  GLUquadricObj* pQuadric = gluNewQuadric();
  
  glNewList(listID, GL_COMPILE_AND_EXECUTE);
  
  double scale = 0.0015;
  
  
  unsigned char rLow,gLow,bLow, rHigh, gHigh, bHigh, r, g, b;
  int rgb;
  double top, bottom;
  
  for (int i=0;i<landmarksPercept.numberOfFlags;i++)
  {
    const Flag& flag = landmarksPercept.flags[i];
    
    Vector2<double> lpright, lpleft;
    lpright.x=cos(flag.x.min)*flag.distance + cameraMatrix.translation.x;
    lpright.y=sin(flag.x.min)*flag.distance + cameraMatrix.translation.y;
    lpleft.x=cos(flag.x.max)*flag.distance + cameraMatrix.translation.x;
    lpleft.y=sin(flag.x.max)*flag.distance + cameraMatrix.translation.y;
    top = sin(flag.y.max)*flag.distance + cameraMatrix.translation.z;
    bottom = sin(flag.y.min)*flag.distance + cameraMatrix.translation.z;
    
    int rgb = ColorClasses::colorClassToRGB(landmarksPercept.flags[i].getLowerColor());
    rLow = rgb % 0x100; rgb /= 0x100; gLow = rgb % 0x100;rgb /= 0x100;bLow = rgb % 0x100;
    rgb = ColorClasses::colorClassToRGB(flag.getUpperColor());
    rHigh = rgb % 0x100; rgb /= 0x100; gHigh = rgb % 0x100;rgb /= 0x100;bHigh = rgb % 0x100;
    
    glPolygonMode(GL_FRONT,GL_FILL);
    glPolygonMode(GL_BACK,GL_FILL);
    glBegin(GL_QUADS);
    glColor3ub(rLow,gLow,bLow);
    glVertex3d(lpright.x*scale, lpright.y*scale, bottom*scale);
    glVertex3d(lpleft.x*scale, lpleft.y*scale, bottom*scale);
    
    glVertex3d(lpleft.x*scale, lpleft.y*scale, (top+bottom)/2*scale);
    glVertex3d(lpright.x*scale, lpright.y*scale, (top+bottom)/2*scale);
    
    glColor3ub(rHigh,gHigh,bHigh);
    glVertex3d(lpleft.x*scale, lpleft.y*scale, top*scale);
    glVertex3d(lpright.x*scale, lpright.y*scale, top*scale);
    glVertex3d(lpright.x*scale, lpright.y*scale, (top+bottom)/2*scale);
    glVertex3d(lpleft.x*scale, lpleft.y*scale, (top+bottom)/2*scale);
    glEnd();
    
    glBegin(GL_LINES);
    
    glColor3ub(rHigh,gHigh,bHigh);
    glVertex3d(cameraMatrix.translation.x*scale, + cameraMatrix.translation.y*scale, + cameraMatrix.translation.z*scale);
    glVertex3d((lpright.x + lpleft.x) / 2 * scale, (lpright.y + lpleft.y ) / 2 *scale, (top) * scale);
    glColor3ub(rLow,gLow,bLow);
    glVertex3d(cameraMatrix.translation.x*scale, + cameraMatrix.translation.y*scale, + cameraMatrix.translation.z*scale);
    glVertex3d((lpright.x + lpleft.x) / 2 * scale, (lpright.y + lpleft.y ) / 2 *scale, (bottom) * scale);
    
    glEnd();
    
    /*    
    glColor3ub(rLow,gLow,bLow);
    glVertex3d(cameraMatrix.translation.x*scale, + cameraMatrix.translation.y*scale, + cameraMatrix.translation.z*scale);
    glVertex3d(lpright.x*scale, lpright.y*scale, bottom*scale);
    
    glVertex3d(cameraMatrix.translation.x*scale, + cameraMatrix.translation.y*scale, + cameraMatrix.translation.z*scale);
    glVertex3d(lpleft.x*scale, lpleft.y*scale, bottom*scale);
    
    
    glColor3ub(rHigh,gHigh,bHigh);
    glVertex3d(cameraMatrix.translation.x*scale, + cameraMatrix.translation.y*scale, + cameraMatrix.translation.z*scale);
    glVertex3d(lpright.x*scale, lpright.y*scale, top*scale);
    
    glVertex3d(cameraMatrix.translation.x*scale, + cameraMatrix.translation.y*scale, + cameraMatrix.translation.z*scale);
    glVertex3d(lpleft.x*scale, lpleft.y*scale, top*scale);
    */
  }
  for (int j=0;j<landmarksPercept.numberOfGoals;j++)
  {
    const Goal& goal = landmarksPercept.goals[j];
    
    Vector2<double> gpleft, gpright;
    gpleft.x=cos(goal.x.max)*goal.distance + cameraMatrix.translation.x;
    gpleft.y=sin(goal.x.max)*goal.distance + cameraMatrix.translation.y;
    gpright.x=cos(goal.x.min)*goal.distance + cameraMatrix.translation.x;
    gpright.y=sin(goal.x.min)*goal.distance + cameraMatrix.translation.y;
    top = sin(goal.y.max)*goal.distance + cameraMatrix.translation.z;
    bottom = sin(goal.y.min)*goal.distance + cameraMatrix.translation.z;
    
    rgb = ColorClasses::colorClassToRGB(goal.color);
    r = rgb % 0x100; rgb /= 0x100; g = rgb % 0x100;rgb /= 0x100;b = rgb % 0x100;
    
    glPolygonMode(GL_FRONT,GL_FILL);
    glPolygonMode(GL_BACK,GL_FILL);
    glBegin(GL_QUADS);
    
//    if(goal.isOnBorder(goal.x.max)) glColor3ub(r / 4, g / 4, b / 4);
//    else 
    glColor3ub(r,g,b);
//    glVertex3d(cameraMatrix.translation.x*scale, + cameraMatrix.translation.y*scale, + cameraMatrix.translation.z*scale);
    glVertex3d(gpright.x*scale, gpright.y*scale, bottom*scale);
    
//    glVertex3d(cameraMatrix.translation.x*scale, + cameraMatrix.translation.y*scale, + cameraMatrix.translation.z*scale);
    glVertex3d(gpright.x*scale, gpright.y*scale, top*scale);
    
//    if(goal.isOnBorder(goal.x.min)) glColor3ub(r / 4, g / 4, b / 4);
//    else 
    glColor3ub(r,g,b);
//    glVertex3d(cameraMatrix.translation.x*scale, + cameraMatrix.translation.y*scale, + cameraMatrix.translation.z*scale);
    glVertex3d(gpleft.x*scale, gpleft.y*scale, top*scale);
    
//    glVertex3d(cameraMatrix.translation.x*scale, + cameraMatrix.translation.y*scale, + cameraMatrix.translation.z*scale);
    glVertex3d(gpleft.x*scale, gpleft.y*scale, bottom*scale);
    
    glEnd();
    
    glBegin(GL_LINES);
    glColor3ub(r/2,g/2,b/2);
    glVertex3d(cameraMatrix.translation.x*scale, + cameraMatrix.translation.y*scale, + cameraMatrix.translation.z*scale);
    glVertex3d((gpright.x + gpleft.x) / 2 * scale, (gpright.y + gpleft.y) / 2 * scale, (bottom + top) / 2 * scale);
    glEnd();
  }
  
  
  glEndList();
  
  gluDeleteQuadric(pQuadric);
}

void OpenGLMethods::paintAngleOfViewToOpenGLList
(
 const CameraMatrix& cameraMatrix,
 const CameraInfo& cameraInfo,
 int listID
 )
{
  double scale = 0.0015;
  double distanceToScreen = 500;
  double x = distanceToScreen * tan(cameraInfo.openingAngleWidth / 2.0);
  double y = distanceToScreen * tan(cameraInfo.openingAngleHeight / 2.0);
  
  Vector3<double> camera[4], world[4];
  
  camera[0].x = distanceToScreen;
  camera[0].y = -x;
  camera[0].z = y;
  
  camera[1].x = distanceToScreen;
  camera[1].y = x;
  camera[1].z = y;
  
  camera[2].x = distanceToScreen;
  camera[2].y = x;
  camera[2].z = -y;
  
  camera[3].x = distanceToScreen;
  camera[3].y = -x;
  camera[3].z = -y;
  
  for(int i = 0; i < 4; i++)
  {
    world[i] = cameraMatrix.rotation * camera[i] + cameraMatrix.translation;
  }
  
  GLUquadricObj* pQuadric = gluNewQuadric();
  glNewList(listID, GL_COMPILE_AND_EXECUTE);
  fireGLHack();
  
  glBegin(GL_LINE_LOOP);
  glColor3ub(0,0,0);
  glVertex3d(world[0].x * scale, world[0].y * scale, world[0].z * scale);
  glVertex3d(world[1].x * scale, world[1].y * scale, world[1].z * scale);
  glVertex3d(world[2].x * scale, world[2].y * scale, world[2].z * scale);
  glVertex3d(world[3].x * scale, world[3].y * scale, world[3].z * scale);
  glEnd();
  
  //camera coordinate system
  world[0] = cameraMatrix.translation;
  camera[1].x = 200;
  camera[1].y = 0;
  camera[1].z = 0;
  
  camera[2].x = 0;
  camera[2].y = 200;
  camera[2].z = 0;
  
  camera[3].x = 0;
  camera[3].y = 0;
  camera[3].z = 200;
  
  world[1] = cameraMatrix.rotation * camera[1] + cameraMatrix.translation;
  world[2] = cameraMatrix.rotation * camera[2] + cameraMatrix.translation;
  world[3] = cameraMatrix.rotation * camera[3] + cameraMatrix.translation;
  
  glBegin(GL_LINES);
  glColor3ub(0,0,0);
  
  glVertex3d(world[0].x * scale, world[0].y * scale, world[0].z * scale);
  glVertex3d(world[1].x * scale, world[1].y * scale, world[1].z * scale);
  
  glVertex3d(world[0].x * scale, world[0].y * scale, world[0].z * scale);
  glVertex3d(world[2].x * scale, world[2].y * scale, world[2].z * scale);
  
  glVertex3d(world[0].x * scale, world[0].y * scale, world[0].z * scale);
  glVertex3d(world[3].x * scale, world[3].y * scale, world[3].z * scale);
  glEnd();
  
  //translated coordinate system
  world[0] = cameraMatrix.translation;
  camera[1].x = 200;
  camera[1].y = 0;
  camera[1].z = 0;
  
  camera[2].x = 0;
  camera[2].y = 200;
  camera[2].z = 0;
  
  camera[3].x = 0;
  camera[3].y = 0;
  camera[3].z = 200;
  
  world[1] = camera[1] + cameraMatrix.translation;
  world[2] = camera[2] + cameraMatrix.translation;
  world[3] = camera[3] + cameraMatrix.translation;
  
  glBegin(GL_LINES);
  glColor3ub(255,255,255);
  
  glVertex3d(world[0].x * scale, world[0].y * scale, world[0].z * scale);
  glVertex3d(world[1].x * scale, world[1].y * scale, world[1].z * scale);
  
  glVertex3d(world[0].x * scale, world[0].y * scale, world[0].z * scale);
  glVertex3d(world[2].x * scale, world[2].y * scale, world[2].z * scale);
  
  glVertex3d(world[0].x * scale, world[0].y * scale, world[0].z * scale);
  glVertex3d(world[3].x * scale, world[3].y * scale, world[3].z * scale);
  glEnd();
  
  glEndList();
  gluDeleteQuadric(pQuadric);
}

void OpenGLMethods::paintRotatedImageToOpenGLList
(
 const CameraMatrix& cameraMatrix, 
 const Image& image, 
 bool clipPointsBelowGround, int listID, 
 bool beginList, bool endList
 )
{
  double scale = 0.0015;
  double distanceToScreen = 500;
  double x = distanceToScreen * tan(image.cameraInfo.openingAngleWidth / 2.0);
  double y = distanceToScreen * tan(image.cameraInfo.openingAngleHeight / 2.0);
  
  Vector3<double> camera[4], world[4];
  
  camera[0].x = distanceToScreen;
  camera[0].y = -x;
  camera[0].z = y;
  
  camera[1].x = distanceToScreen;
  camera[1].y = x;
  camera[1].z = y;
  
  camera[2].x = distanceToScreen;
  camera[2].y = x;
  camera[2].z = -y;
  
  camera[3].x = distanceToScreen;
  camera[3].y = -x;
  camera[3].z = -y;
  
  for(int n = 0; n < 4; n++)
  {
    world[n] = cameraMatrix.rotation * camera[n] + cameraMatrix.translation;
  }
  
  GLUquadricObj* pQuadric = gluNewQuadric();
  if (beginList) glNewList(listID, GL_COMPILE_AND_EXECUTE);
  
/*
  glBegin(GL_LINE_LOOP);
  glColor3ub(255,255,255);
  glVertex3d(world[0].x * scale, world[0].y * scale, world[0].z * scale);
  glVertex3d(world[1].x * scale, world[1].y * scale, world[1].z * scale);
  glVertex3d(world[2].x * scale, world[2].y * scale, world[2].z * scale);
  glVertex3d(world[3].x * scale, world[3].y * scale, world[3].z * scale);
  glEnd();
*/
  Image rgbImage;
  rgbImage.convertFromYUVToRGB(image);
  
  if(clipPointsBelowGround)
  {
    glPolygonMode(GL_FRONT,GL_FILL);
    glPolygonMode(GL_BACK,GL_FILL);
    glBegin(GL_QUADS);
    for(int i = 0; i < image.cameraInfo.resolutionWidth; i++)
    {
      for(int j = 0; j < image.cameraInfo.resolutionHeight; j++)
      {
        Vector2<int> quadCorners[4];
        if(
          Geometry::calculatePointOnField(i,     j, cameraMatrix, image.cameraInfo, quadCorners[0]) &&
          Geometry::calculatePointOnField(i + 1, j, cameraMatrix, image.cameraInfo, quadCorners[1]) &&
          Geometry::calculatePointOnField(i + 1, j + 1, cameraMatrix, image.cameraInfo, quadCorners[2]) &&
          Geometry::calculatePointOnField(i,     j + 1, cameraMatrix, image.cameraInfo, quadCorners[3]) 
          )
        {
          glColor3ub(rgbImage.image[j][0][i], rgbImage.image[j][1][i], rgbImage.image[j][2][i]);
          glVertex3d(quadCorners[0].x * scale, quadCorners[0].y * scale, scale * 5);
          glVertex3d(quadCorners[1].x * scale, quadCorners[1].y * scale, scale * 5);
          glVertex3d(quadCorners[2].x * scale, quadCorners[2].y * scale, scale * 5);
          glVertex3d(quadCorners[3].x * scale, quadCorners[3].y * scale, scale * 5);
        }
      }
    }
    glEnd();
    glPolygonMode(GL_FRONT,GL_LINE);
    glPolygonMode(GL_BACK,GL_LINE);
  }
  else
  {
    fireGLHack();
    glBegin(GL_POINTS);
    for(int i = 0; i < rgbImage.cameraInfo.resolutionWidth; i++)
    {
      for(int j = 0; j < rgbImage.cameraInfo.resolutionHeight; j++)
      {
        double xx, yy, zz;
        xx = world[0].x + i * (world[1].x - world[0].x) / (double)image.cameraInfo.resolutionWidth + j * (world[3].x - world[0].x) / (double)image.cameraInfo.resolutionHeight;
        yy = world[0].y + i * (world[1].y - world[0].y) / (double)image.cameraInfo.resolutionWidth + j * (world[3].y - world[0].y) / (double)image.cameraInfo.resolutionHeight;
        zz = world[0].z + i * (world[1].z - world[0].z) / (double)image.cameraInfo.resolutionWidth + j * (world[3].z - world[0].z) / (double)image.cameraInfo.resolutionHeight;
        
        glColor3ub(rgbImage.image[j][0][(image.cameraInfo.resolutionWidth-1)-i], rgbImage.image[j][1][(image.cameraInfo.resolutionWidth-1)-i], rgbImage.image[j][2][(image.cameraInfo.resolutionWidth-1)-i]);
        glVertex3d(xx * scale, yy * scale, zz * scale);
      }
    }
    glEnd();
  }
  
  if(endList) glEndList();
  gluDeleteQuadric(pQuadric);
}

void OpenGLMethods::paintImagePixelsToOpenGLList
(
 const Image& image,
 int colorModel,
 int zComponent,
 bool polygons,
 int listID
 )
{
  paintImagePixelsToOpenGLList(image, colorModel, zComponent, polygons, listID, 0, image.cameraInfo.resolutionWidth, 0, image.cameraInfo.resolutionHeight);
}

void OpenGLMethods::paintColorTableReferenceColor
(
 const ColorTableReferenceColor& colorTableReferenceColor,
 int listID
 )
{
  // Build a new list
  ::glNewList(listID,GL_COMPILE_AND_EXECUTE);
  
  double tmp = 2.0f/255.0f;
  double x,y;
  double z00;
  
  
  OpenGLMethods::paintCubeToOpenGLList(
    colorTableReferenceColor.getGreenMin(2),
    colorTableReferenceColor.getGreenMin(1),
    colorTableReferenceColor.getGreenMin(0),
    colorTableReferenceColor.getGreenMax(2),
    colorTableReferenceColor.getGreenMax(1),
    colorTableReferenceColor.getGreenMax(0),
    OpenGLMethods::colorSpaceColorTableReferenceColorList, true,
    127, //scale
    -127,-127,-127, // offsets
    0, 255, 0, // green
    false, false// dont begin and dont end the list
    );
  
  glBegin(GL_POINTS);
  
  colorClass color;
  
  for(int colorSpaceY = 0; colorSpaceY < 255; colorSpaceY += 4)
  {
    for(int colorSpaceU = 0; colorSpaceU < 255; colorSpaceU += 4)
    {
      for(int colorSpaceV = 0; colorSpaceV < 255; colorSpaceV += 4)
      {
        color = colorTableReferenceColor.getColorClass(colorSpaceY, colorSpaceU, colorSpaceV);
        
        x   = (double)(-1.0 + colorSpaceV*tmp);
        y   = (double)(-1.0 + colorSpaceU*tmp);
        z00 = (double)(-1.0 + colorSpaceY*tmp);
        
        switch(color)
        {
        case white:
          glColor3ub(255, 255, 255);
          glVertex3d(x, y, z00);
          break;
        case gray:
          glColor3ub(196, 196, 196);
          glVertex3d(x, y, z00);
          break;
        case green:
          glColor3ub(0, 255, 0);
          glVertex3d(x, y, z00);
          break;
        case yellow:
          glColor3ub(255, 255, 0);
          glVertex3d(x, y, z00);
          break;
        case skyblue:
          glColor3ub(0, 255, 255);
          glVertex3d(x, y, z00);
          break;
        case pink:
          //glColor3ub(255, 0, 255);
          //glVertex3d(x, y, z00);
          break;
        case red:
          glColor3ub(255, 0, 0);
          glVertex3d(x, y, z00);
          break;
        case orange:
          glColor3ub(255, 127, 64);
          glVertex3d(x, y, z00);
          break;
        case yellowOrange:
          glColor3ub(255, 192, 32);
          glVertex3d(x, y, z00);
          break;
        } // switch color
      }
    }
  }
  
  glEnd();
  glPolygonMode(GL_FRONT,GL_LINE);
  glPolygonMode(GL_BACK,GL_LINE);
  ::glEndList();
}

void OpenGLMethods::paintColorTableCuboids
(
 const ColorTableCuboids& colorTableCuboids,
 int listID
 )
{
  int rgb, r, g, b;

  for(int color = 0; color < numOfColors; color++)
  {
    colorClass curentColor = colorClass(color);
 
    if(curentColor == green || curentColor == white)
    {
      rgb = ColorClasses::colorClassToRGB(curentColor);
      r = rgb % 0x100; rgb /= 0x100; g = rgb % 0x100;rgb /= 0x100;b = rgb % 0x100;

      OpenGLMethods::paintCubeToOpenGLList(
        colorTableCuboids.getThresholdMin(curentColor,2), colorTableCuboids.getThresholdMin(curentColor,1), colorTableCuboids.getThresholdMin(curentColor,0),
        colorTableCuboids.getThresholdMax(curentColor,2), colorTableCuboids.getThresholdMax(curentColor,1), colorTableCuboids.getThresholdMax(curentColor,0),
        listID, true,  127, //scale
        -127,-127,-127, // offsets
//        r, g, b, color == 0, color == numOfColors - 1
        r, g, b, color == green, color == white
        );
    }
  }
}



void OpenGLMethods::paintColorTable
(
 const ColorTable& colorTable,
 int listID
 )
{
  // Build a new list
  ::glNewList(listID,GL_COMPILE_AND_EXECUTE);
  
  glBegin(GL_POINTS);
  
  double tmp = 2.0f/255.0f;
  double x,y;
  double z00;
  
  for(int colorSpaceY = 0; colorSpaceY < 255; colorSpaceY += 4)
  {
    for(int colorSpaceU = 0; colorSpaceU < 255; colorSpaceU += 4)
    {
      for(int colorSpaceV = 0; colorSpaceV < 255; colorSpaceV += 4)
      {
        colorClass color = 
          colorTable.getColorClass(
          colorSpaceY, colorSpaceU, colorSpaceV);
        
        x   = (double)(-1.0 + colorSpaceV*tmp);
        y   = (double)(-1.0 + colorSpaceU*tmp);
        z00 = (double)(-1.0 + colorSpaceY*tmp);
        
        switch (color)
        {
        case white:
          glColor3ub(255, 255, 255);
          glVertex3d(x, y, z00);
          break;
        case gray:
          glColor3ub(196, 196, 196);
          glVertex3d(x, y, z00);
          break;
        case green:
          glColor3ub(0, 255, 0);
          glVertex3d(x, y, z00);
          break;
        case yellow:
          glColor3ub(255, 255, 0);
          glVertex3d(x, y, z00);
          break;
        case skyblue:
          glColor3ub(0, 255, 255);
          glVertex3d(x, y, z00);
          break;
        case pink:
          glColor3ub(255, 0, 255);
          glVertex3d(x, y, z00);
          break;
        case red:
          glColor3ub(255, 0, 0);
          glVertex3d(x, y, z00);
          break;
        case orange:
          glColor3ub(255, 127, 64);
          glVertex3d(x, y, z00);
          break;
        case yellowOrange:
          glColor3ub(255, 192, 32);
          glVertex3d(x, y, z00);
          break;
        }
      }
    }
  }    
  glEnd();
  glPolygonMode(GL_FRONT,GL_LINE);
  glPolygonMode(GL_BACK,GL_LINE);
  ::glEndList();
}


void OpenGLMethods::paintImagePixelsToOpenGLList
(
 const Image& image,
 int colorModel,
 int zComponent,
 bool polygons,
 int listID,
 int x1,
 int x2,
 int y1,
 int y2
 )
{
  // Build a new list
  ::glNewList(listID,GL_COMPILE_AND_EXECUTE);
  
  if(polygons)
  {
    glPolygonMode(GL_FRONT,GL_FILL);
    glPolygonMode(GL_BACK,GL_FILL);
    glBegin(GL_QUADS);
  }
  else
  {
    glBegin(GL_POINTS);
  }
  
  double tmp = 2.0f/255.0f;
  double x,y;
  double xn,yn;
  double z00, z01, z10, z11;
  
  Image rgbImage;
  rgbImage.convertFromYUVToRGB(image);
  
  Image convertedImage;
  switch(colorModel)
  {
  case 0: //yuv
    convertedImage = image;
    break;
  case 1: //rgb
    convertedImage.convertFromYUVToRGB(image);
    break;
  case 2: //tsl
    convertedImage.convertFromYUVToTSL(image);
    break;
  case 3: //hsi
    convertedImage.convertFromYCbCrToHSI(image);
    break;
  }
  
  for(int j = y1; j < y2 - 2; j++)
  {
    for(int i = x1; i < x2 - 2; i++)
    {
      switch(zComponent)
      {
      case -1: //yuvUsage
        x   = (double)(-1.0 + convertedImage.image[j][2][i]*tmp);
        y   = (double)(-1.0 + convertedImage.image[j][1][i]*tmp);
        z00 = (double)(-1.0 + convertedImage.image[j][0][i]*tmp);
        break;
      case 0://yUsage:
        x = (double)( +i - image.cameraInfo.resolutionWidth / 2) * tmp;
        y = (double)( -j + image.cameraInfo.resolutionHeight / 2) * tmp;
        xn = (double)( +i + 1 - image.cameraInfo.resolutionWidth / 2) * tmp;
        yn = (double)( -j - 1 + image.cameraInfo.resolutionHeight / 2) * tmp;
        z00 = (double)(-0.5 + convertedImage.image[j][0][i] * tmp / 2);
        if(polygons)
        {
          z01 = (double)(-0.5 + convertedImage.image[j+1][0][i] * tmp / 2);
          z10 = (double)(-0.5 + convertedImage.image[j][0][i+1] * tmp / 2);
          z11 = (double)(-0.5 + convertedImage.image[j+1][0][i+1] * tmp / 2);
        }
        break;
      case 1://uUsage:
        x = (double)( +i - image.cameraInfo.resolutionWidth / 2) * tmp;
        y = (double)( -j + image.cameraInfo.resolutionHeight / 2) * tmp;
        xn = (double)( +i + 1 - image.cameraInfo.resolutionWidth / 2) * tmp;
        yn = (double)( -j - 1 + image.cameraInfo.resolutionHeight / 2) * tmp;
        z00 = (double)(-0.5 + convertedImage.image[j][1][i] * tmp / 2);
        if(polygons)
        {
          z01 = (double)(-0.5 + convertedImage.image[j+1][1][i] * tmp / 2);
          z10 = (double)(-0.5 + convertedImage.image[j][1][i+1] * tmp / 2);
          z11 = (double)(-0.5 + convertedImage.image[j+1][1][i+1] * tmp / 2);
        }
        break;
      case 2://vUsage:
        x = (double)( +i - image.cameraInfo.resolutionWidth / 2) * tmp;
        y = (double)( -j + image.cameraInfo.resolutionHeight / 2) * tmp;
        xn = (double)( +i + 1 - image.cameraInfo.resolutionWidth / 2) * tmp;
        yn = (double)( -j - 1 + image.cameraInfo.resolutionHeight / 2) * tmp;
        z00 = (double)(-0.5 + convertedImage.image[j][2][i] * tmp / 2);
        if(polygons)
        {
          z01 = (double)(-0.5 + convertedImage.image[j+1][2][i] * tmp / 2);
          z10 = (double)(-0.5 + convertedImage.image[j][2][i+1] * tmp / 2);
          z11 = (double)(-0.5 + convertedImage.image[j+1][2][i+1] * tmp / 2);
        }
        break;
      }
      
      if (zComponent == -1)//(selection == yuvUsage)
      {
        glColor3ub(rgbImage.image[j][0][i], rgbImage.image[j][1][i], rgbImage.image[j][2][i]);
        glVertex3d(x, y, z00);
      }
      else        
      {
        if(!polygons)
        {
          glColor3ub(rgbImage.image[j][0][i], rgbImage.image[j][1][i], rgbImage.image[j][2][i]);
          glVertex3d(x, y, z00);
        }
        else
        {
          glColor3ub(rgbImage.image[j][0][i], rgbImage.image[j][1][i], rgbImage.image[j][2][i]);
          glVertex3d(x, y, z00);
          
          glColor3ub(rgbImage.image[j][0][i+1], rgbImage.image[j][1][i+1], rgbImage.image[j][2][i+1]);
          glVertex3d(xn, y, z10);
          
          glColor3ub(rgbImage.image[j+1][0][i+1], rgbImage.image[j+1][1][i+1], rgbImage.image[j+1][2][i+1]);
          glVertex3d(xn, yn, z11);
          
          glColor3ub(rgbImage.image[j+1][0][i], rgbImage.image[j+1][1][i], rgbImage.image[j+1][2][i]);
          glVertex3d(x, yn, z01);
        }
      }
    }
  }
  glEnd();
  glPolygonMode(GL_FRONT,GL_LINE);
  glPolygonMode(GL_BACK,GL_LINE);
  ::glEndList();
  
}

void OpenGLMethods::fireGLHack()
{
  glPolygonMode(GL_FRONT,GL_FILL);
  glPolygonMode(GL_BACK,GL_FILL);
  glBegin(GL_QUADS);
  glColor4d(0,0,0,0);
  glVertex3d(0,0,0);
  glVertex3d(0,0.0001,0);
  glVertex3d(0.0001,0.0001,0);
  glVertex3d(0.0001,0,0);
  glEnd();
}


void OpenGLMethods::paintPSDPerceptToOpenGLList
(
 const CameraMatrix& cameraMatrix,
 const PSDPercept& psdPercept,
 int listID
 )
{
  const RobotDimensions& robotDimensions = getRobotConfiguration().getRobotDimensions();

  GLUquadricObj* pQuadric = gluNewQuadric();
  
  glNewList(listID, GL_COMPILE_AND_EXECUTE);
  
  double scale = 0.0015;
  double distanceCameraToPSDSensor = robotDimensions.distanceCameraToPSDSensor;
  
  unsigned char rColor, gColor, bColor;
 

	Vector3<double> vectorToPointWorld, rotatedPSDPercept, translate, translateToCamera;

	for (int i=0;i<psdPercept.numOfPercepts;i++)
   {

	translate.x = 0;
	translate.y = 0;
	translate.z = distanceCameraToPSDSensor;

	translateToCamera = cameraMatrix.rotation*translate*scale;


	vectorToPointWorld.x = scale*(psdPercept[i].x);
    vectorToPointWorld.y = scale*(psdPercept[i].y);
    vectorToPointWorld.z = scale*(psdPercept[i].z);

	//PSDPercept is red if an obstacle detected (near that 90cm) and green else
	 rColor = 255*(!psdPercept[i].tooFarAway);
	 gColor = 255*(psdPercept[i].tooFarAway);
	 bColor = 0;

    glBegin(GL_LINES);
    glColor3ub(rColor,gColor,bColor);
    glVertex3d(cameraMatrix.translation.x*scale + translateToCamera.x, cameraMatrix.translation.y*scale + translateToCamera.y, cameraMatrix.translation.z*scale + translateToCamera.z);
    glVertex3d(vectorToPointWorld.x,vectorToPointWorld.y,vectorToPointWorld.z);
    glEnd();

    glBegin(GL_LINES);
    glColor3ub(0,0,200);
    glVertex3d(cameraMatrix.translation.x*scale + translateToCamera.x, cameraMatrix.translation.y*scale + translateToCamera.y, 1 * scale);
    glVertex3d(vectorToPointWorld.x,vectorToPointWorld.y, scale * 6);

    glVertex3d(vectorToPointWorld.x,vectorToPointWorld.y, scale * 6);
    glVertex3d(vectorToPointWorld.x,vectorToPointWorld.y,vectorToPointWorld.z);
    glEnd();
    
 }//end for

  glEndList();
  
  gluDeleteQuadric(pQuadric);
 
}//end paintPSDPerceptToOpenGLList


void OpenGLMethods::paintBallPerceptToOpenGLList
(
 const BallPercept& ballPercept,
 int listID
 )
{
  double scale = 0.0015;
  float center[3];
  float ballRadius = (float)(scale*43);
  
  Vector2<double> ballVector;
  
  GLUquadricObj* pQuadric = gluNewQuadric();
  glNewList(listID, GL_COMPILE_AND_EXECUTE);
  
  glPolygonMode(GL_FRONT,GL_FILL);
  glPolygonMode(GL_BACK,GL_FILL);
  glBegin(GL_QUADS);

  
  // size based
  ballPercept.getOffsetSizeBased(ballVector);

  center[0] = (float)(ballVector.x*scale);
  center[1] = (float)(ballVector.y*scale);
  center[2] = (float)(ballRadius);
  
  glColor3ub(215,160,100); // orange
  paintSphere(center, ballRadius, 10);

  // bearing based
  ballPercept.getOffsetBearingBased(ballVector);

  center[0] = (float)(ballVector.x*scale);
  center[1] = (float)(ballVector.y*scale);
  center[2] = (float)(ballRadius);
  
  glColor3ub(255,0,0); // red
  paintSphere(center, ballRadius, 10);

  glEnd();
  glPolygonMode(GL_FRONT,GL_LINE);
  glPolygonMode(GL_BACK,GL_LINE);
  
  glEndList();
  gluDeleteQuadric(pQuadric);
  
}//end paintBallPerceptToOpenGLList


/*
* Change log :
* 
* $Log: OpenGLMethods.cpp,v $
* Revision 1.17  2004/05/01 17:09:33  roefer
* Streamlined HSI stuff
*
* Revision 1.16  2004/02/14 13:05:57  roefer
* SensorData drawn for feet and knees
*
* Revision 1.15  2004/02/12 14:14:12  juengel
* Changed visualization of ColorTableReferenceColor.
*
* Revision 1.14  2004/01/25 21:19:01  juengel
* Added indications for bodyRoll and bodyTilt in JointData visualization.
* Different steps of CameraMatrix calculation are shown in SensorData visualization.
*
* Revision 1.13  2004/01/24 11:38:17  juengel
* - bearing-based part of BallPercept is shown, too.
* - added white grid lines (500mm markers).
* - projection of psd percept to ground added.
*
* Revision 1.12  2004/01/21 17:37:29  mellmann
* BallPercept 3DRadar-visualization  exist & works now :)
*
* Revision 1.11  2004/01/14 19:19:02  mellmann
* Visualization of PSDPercept.
*
* Revision 1.10  2004/01/10 12:41:32  juengel
* All cuboids in the ColorSpaceDialog have its center in the rotation center now.
*
* Revision 1.9  2004/01/04 12:08:28  juengel
* adaptation for larger images
*
* Revision 1.8  2003/12/31 23:50:39  roefer
* Removed inclusion of RobotDimensions.h because it is not used
*
* Revision 1.7  2003/12/15 11:49:07  juengel
* Introduced CameraInfo
*
* Revision 1.6  2003/12/09 14:21:47  dueffert
* flyMode added to visualize theoretical footPositions in RadarViewer better
*
* Revision 1.5  2003/11/03 20:19:22  juengel
* Added color class yellowOrange
*
* Revision 1.4  2003/10/30 18:29:15  juengel
* Improved paint of ColorTableCuboids.
*
* Revision 1.3  2003/10/29 13:22:15  juengel
* cuboid for reference color is painted in paint method for ColorTableReferenceColor
*
* Revision 1.2  2003/10/23 07:26:57  juengel
* Renamed ColorTableAuto to ColorTableReferenceColor.
*
* Revision 1.1  2003/10/07 10:11:08  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.8  2003/09/26 11:40:13  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.7  2003/09/16 14:58:07  dueffert
* tilt calculation in forward kinematics added
*
* Revision 1.6  2003/09/15 20:39:43  dueffert
* forward kinematics improved
*
* Revision 1.5  2003/09/11 15:56:47  juengel
* Added paintSensorDataToOpenGLList and paintJointDataToOpenGLList.
*
* Revision 1.4  2003/08/23 14:28:33  roefer
* fireGLHack added
*
* Revision 1.3  2003/08/18 11:53:19  juengel
* Added paintColorTableReferenceColorThresholds.
*
* Revision 1.2  2003/07/21 13:41:53  juengel
* Moved ColorTableReferenceColor and GoalRecognizer to ImageProcessorTools.
*
* Revision 1.1.1.1  2003/07/02 09:40:27  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.28  2003/06/26 12:35:45  juengel
* Visualization of ObstaclesModel.
*
* Revision 1.27  2003/06/25 18:45:33  juengel
* Added nearPoints, farPoints[maxNumberOfPoints] and farPointIsOnImageBorder to ObstaclesPercept.
*
* Revision 1.26  2003/05/11 23:32:41  dueffert
* Depend now works with RobotControl too
*
* Revision 1.25  2003/05/05 12:19:31  juengel
* Changes in ColorTableReferenceColor.
*
* Revision 1.24  2003/04/23 16:17:47  juengel
* ColorTableReferenceColor::getColorClassForFlagsAndGoals returns a color class now.
*
* Revision 1.23  2003/04/02 13:21:57  juengel
* Added white to flag colors
*
* Revision 1.22  2003/04/01 14:00:59  juengel
* Added visualization for color tables.
*
* Revision 1.21  2003/03/11 11:26:56  juengel
* Added obstaclesPercept to drawingMethods.
*
* Revision 1.20  2003/03/10 14:02:01  juengel
* Prepared new type sideOfPenaltyArea
*
* Revision 1.19  2003/02/27 12:55:55  juengel
* Improved cube drawing.
*
* Revision 1.18  2003/02/24 22:33:04  juengel
* ColorSpace Dialog can convert images to RGB, HSI and TSL.
*
* Revision 1.17  2003/02/21 16:34:16  dueffert
* common pi in own code, warnings removed, platform compatibility restored
*
* Revision 1.16  2003/01/09 14:04:01  jhoffman
* stuff needed for 3DFieldView
*
* Revision 1.15  2002/12/15 23:33:02  dueffert
* rgb-yuv-convertion moved to Image
*
* Revision 1.14  2002/12/04 12:25:03  juengel
* Changed parameter "pointOnField" of method Geometry::calculatePointOnField() from Vector2<double> to Vector2<int>.
*
* Revision 1.13  2002/11/28 18:53:52  juengel
* RadarViewer3D shows images projected on ground.
*
* Revision 1.12  2002/11/26 17:29:12  juengel
* Removed "mirrored image bug".
*
* Revision 1.11  2002/11/26 14:17:36  juengel
* RadarViewer3DDlg displays images.
*
* Revision 1.10  2002/11/04 13:41:48  juengel
* no message
*
* Revision 1.9  2002/10/11 13:45:03  juengel
* Methods for visualization of percepts added.
*
* Revision 1.8  2002/10/10 16:18:39  loetzsch
* some minor improvements and better doxygen comments
*
* Revision 1.7  2002/10/10 13:09:50  loetzsch
* First experiments with the PSD Sensor
* - SensorDataProcessor now calculates PSDPercept
* - Added the PerceptBehaviorControl solution PSDTest
* - Added the RadarViewer3D to RobotControl, which can display the Points3D structure
*
* Revision 1.6  2002/09/22 18:40:51  risler
* added new math functions, removed GTMath library
*
* Revision 1.5  2002/09/20 23:28:34  juengel
* Moved instance of DebugDrawingManager to RobotControlMainFrame.
*
* Revision 1.4  2002/09/19 12:05:49  loetzsch
* MathLib -> Mathlib
*
* Revision 1.3  2002/09/17 23:55:23  loetzsch
* - unraveled several datatypes
* - changed the WATCH macro
* - completed the process restructuring
*
* Revision 1.2  2002/09/15 07:52:27  juengel
* Added some sliders in ColorSpaceDlgBar.
*
* Revision 1.1  2002/09/10 15:49:12  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
*
*
*/
