/**
 * @file Cylinder.cpp
 * 
 * Implementation of class Cylinder
 *
 * @author <A href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</A>
 */ 

#include "Platform/OpenGL.h"
#include "Cylinder.h"
#include "Surface.h"


Cylinder::Cylinder()
{
  closed = false;
}

void Cylinder::draw(const Vector3d& pointOfView,
                    const VisualizationParameterSet& visParams)
{
  if(invisible && !drawForCamera)
  {
    return;
  }
  Vector3d cylVec(0.0,0.0,1.0);
  cylVec.rotate(rotation);
  double rotX(cylVec.getXRotation());
  cylVec.rotateX(-rotX);
  double rotY(cylVec.getYRotation());
  rotX *= 180.0 / M_PI;
  rotY *= 180.0 / M_PI;
  Vector3d color(surface->getColor(inverted && !drawForCamera));
  glColor3f((float) color.v[0],(float) color.v[1],(float) color.v[2]);
  glTranslated(position.v[0],position.v[1],position.v[2]);
  glRotatef((float) rotX, 1.0f, 0.0f, 0.0f);
  glRotatef((float) rotY, 0.0f, 1.0f, 0.0f);
  GLUquadricObj* q = gluNewQuadric();
  gluCylinder(q, radius, radius, height, 15, 15);
  if(closed)
  {
    gluDisk(q, 0, radius, 15, 15);
    glTranslated(0,0,height);
    gluDisk(q, 0, radius, 15, 15);
    glTranslated(0,0,-height);
  }
  gluDeleteQuadric(q);
  glRotatef((float) -rotY, 0.0f, 1.0f, 0.0f);
  glRotatef((float) -rotX, 1.0f, 0.0f, 0.0f);
  glTranslated(-position.v[0],-position.v[1],-position.v[2]);
  SimObject::draw(pointOfView, visParams);
}

SimObject* Cylinder::clone() const
{
  Cylinder* newCylinder = new Cylinder();
  newCylinder->setName(name);
  newCylinder->setFullName(fullName);
  newCylinder->setPosition(position);
  newCylinder->rotation = rotation;
  newCylinder->surface = surface;
  newCylinder->radius = radius;
  newCylinder->height = height;
  newCylinder->closed = closed;
  newCylinder->invisible = invisible;
  std::list<SimObject*>::const_iterator pos;
  for(pos = childNodes.begin(); pos != childNodes.end(); ++pos)
  {
    SimObject* childNode = (*pos)->clone();
    newCylinder->addChildNode(childNode, false);
  }
  SimObject* newObject = newCylinder;
  return newObject;
}

double Cylinder::getMaxDistanceTo(Vector3d& base) const
{
  //TODO: Check this!
  double dist=0;
  Vector3d vec = base - position;
  double vecLen = vec.getLength();
  double cylinderRadius = sqrt(height*height + radius*radius);
  dist = vecLen + cylinderRadius;
  std::list<SimObject*>::const_iterator pos;
  for(pos = childNodes.begin(); pos != childNodes.end(); ++pos)
  {
    double childDist = (*pos)->getMaxDistanceTo(base);
    if(dist < childDist)
    {
      dist = childDist;
    }
  }
  return dist;
}

bool Cylinder::intersectWithRay(const Vector3d& pos, const Vector3d& ray, Vector3d& intersectPos) 
{
  return false;
}

/*
 * $Log: Cylinder.cpp,v $
 * Revision 1.4  2003/12/09 13:40:50  roefer
 * href attribute corrected
 *
 * Revision 1.11  2003/12/04 19:57:37  roefer
 * Memory leak (gluNewQuadric) closed
 *
 * Revision 1.10  2003/12/03 18:12:37  roefer
 * Compatibility with VC2003.NET, GUI does still not work completely there
 *
 * Revision 1.9  2003/10/26 14:47:03  tim
 * - fixed cylinder backtransformation problem
 * - fixed wireframe polyeder drawing problem
 *
 * Revision 1.8  2003/10/26 12:09:31  tim
 * - changed polygon rendering to vertex arrays
 * - improved polygon intersection test
 * - removed backtransformation stuff
 *
 * Revision 1.7  2003/10/11 14:53:57  tim
 * - added standard length
 * - added invisibility for objects
 * - changed parser implementation
 *
 * Revision 1.6  2003/10/05 15:24:30  tim
 * - changed drag & drop visualization
 *
 * Revision 1.5  2003/09/18 01:52:06  tim
 * - changed OpenGL surface computation
 * - added stepLength
 *
 * Revision 1.4  2003/09/12 11:34:14  tim
 * - added sensor visualization framework
 * - implemented visualization for whisker
 *
 * Revision 1.3  2003/09/08 22:32:07  tim
 * - removed files
 * - added some doxygen documentation
 * - added some const qualifiers
 * - partial code clean-up
 * - minor code changes
 * - remove __ from guards (__ should only be used by compiler)
 *
 * Revision 1.2  2003/09/04 13:34:21  tim
 * - better parsing of numbers
 * - fixed macro bug
 * - better integration of macros in the object tree
 * - added getObjectReference() to Simulation
 * - faster object look-up in Simulation
 * - added changed log
 *
 */
