/**
 * @file MovableObject.cpp
 * 
 * Implementation of class MovableObject
 *
 * @author <A href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</A>
 */ 

#include "MovableObject.h"
#include "Actuatorport.h"
#include "Sensorport.h"
#include <string>


MovableObject::MovableObject()
{
  rotationSpeedX.min = 0.0;
  rotationSpeedX.max = 0.0;
  rotationSpeedX.value = 0.0;
  rotationSpeedY.min = 0.0;
  rotationSpeedY.max = 0.0;
  rotationSpeedY.value = 0.0;
  rotationSpeedZ.min = 0.0;
  rotationSpeedZ.max = 0.0;
  rotationSpeedZ.value = 0.0;
  speedX.min = 0.0;
  speedX.max = 0.0;
  speedX.value = 0.0;
  speedY.min = 0.0;
  speedY.max = 0.0;
  speedY.value = 0.0;
  speedZ.min = 0.0;
  speedZ.max = 0.0;
  speedZ.value = 0.0;
}

void MovableObject::addToLists(std::vector<Sensorport*>& sensorportList,
                               std::vector<Actuatorport*>& actuatorportList,
                               std::vector<Actuator*>& actuatorList) 
{
  Actuatorport* xSpeedPort = new Actuatorport("xSpeed", this, xSpeed,
                                              speedX.min,speedX.max);
  Actuatorport* ySpeedPort = new Actuatorport("ySpeed", this, ySpeed,
                                              speedY.min,speedY.max);
  Actuatorport* zSpeedPort = new Actuatorport("zSpeed", this, zSpeed,
                                              speedZ.min,speedZ.max);
  Actuatorport* xRotationPort = new Actuatorport("xRotation", this, xRotation,
                                                 rotationSpeedX.min,rotationSpeedX.max);
  Actuatorport* yRotationPort = new Actuatorport("yRotation", this, yRotation,
                                                 rotationSpeedY.min,rotationSpeedY.max);
  Actuatorport* zRotationPort = new Actuatorport("zRotation", this, zRotation,
                                                 rotationSpeedZ.min,rotationSpeedZ.max);
  actuatorportList.push_back(xSpeedPort);
  actuatorportList.push_back(ySpeedPort);
  actuatorportList.push_back(zSpeedPort);
  actuatorportList.push_back(xRotationPort);
  actuatorportList.push_back(yRotationPort);
  actuatorportList.push_back(zRotationPort);
  actuatorList.push_back(this);

  Sensorport* xSpeedSensor = new Sensorport("xSpeedSensor", xSpeed, doubleSensor, this, speedX.min, speedX.max);
  Sensorport* ySpeedSensor = new Sensorport("ySpeedSensor", ySpeed, doubleSensor, this, speedY.min, speedY.max);
  Sensorport* zSpeedSensor = new Sensorport("zSpeedSensor", zSpeed, doubleSensor, this, speedZ.min, speedZ.max);
  Sensorport* xRotationSensor = new Sensorport("xRotationSensor", xRotation, doubleSensor, this, rotationSpeedX.min, rotationSpeedX.max);
  Sensorport* yRotationSensor = new Sensorport("yRotationSensor", yRotation, doubleSensor, this, rotationSpeedY.min, rotationSpeedY.max);
  Sensorport* zRotationSensor = new Sensorport("zRotationSensor", zRotation, doubleSensor, this, rotationSpeedZ.min, rotationSpeedZ.max);
  sensorportList.push_back(xSpeedSensor);
  sensorportList.push_back(ySpeedSensor);
  sensorportList.push_back(zSpeedSensor);
  sensorportList.push_back(xRotationSensor);
  sensorportList.push_back(yRotationSensor);
  sensorportList.push_back(zRotationSensor);
}

void MovableObject::addToDescriptions(std::vector<ObjectDescription>& objectDescriptionTree,
                                      int depth) 
{
  Actuator::addToDescriptions(objectDescriptionTree, depth);  

  //Add Actuatorports:
  ObjectDescription speedXDesc;
  speedXDesc.name = "xSpeed";
  speedXDesc.fullName = fullName + ".xSpeed";
  speedXDesc.depth = depth + 1;
  speedXDesc.type = OBJECT_TYPE_ACTUATORPORT;
  objectDescriptionTree.push_back(speedXDesc);
  ObjectDescription speedYDesc;
  speedYDesc.name = "ySpeed";
  speedYDesc.fullName = fullName + ".ySpeed";
  speedYDesc.depth = depth + 1;
  speedYDesc.type = OBJECT_TYPE_ACTUATORPORT;
  objectDescriptionTree.push_back(speedYDesc);
  ObjectDescription speedZDesc;
  speedZDesc.name = "zSpeed";
  speedZDesc.fullName = fullName + ".zSpeed";
  speedZDesc.depth = depth + 1;
  speedZDesc.type = OBJECT_TYPE_ACTUATORPORT;
  objectDescriptionTree.push_back(speedZDesc);
  ObjectDescription rotationXDesc;
  rotationXDesc.name = "xRotation";
  rotationXDesc.fullName = fullName + ".xRotation";
  rotationXDesc.depth = depth + 1;
  rotationXDesc.type = OBJECT_TYPE_ACTUATORPORT;
  objectDescriptionTree.push_back(rotationXDesc);
  ObjectDescription rotationYDesc;
  rotationYDesc.name = "yRotation";
  rotationYDesc.fullName = fullName + ".yRotation";
  rotationYDesc.depth = depth + 1;
  rotationYDesc.type = OBJECT_TYPE_ACTUATORPORT;
  objectDescriptionTree.push_back(rotationYDesc);
  ObjectDescription rotationZDesc;
  rotationZDesc.name = "zRotation";
  rotationZDesc.fullName = fullName + ".zRotation";
  rotationZDesc.depth = depth + 1;
  rotationZDesc.type = OBJECT_TYPE_ACTUATORPORT;
  objectDescriptionTree.push_back(rotationZDesc);

  //Add Sensorports:
  ObjectDescription sensorPortDesc;
  sensorPortDesc.depth = depth + 1;
  sensorPortDesc.type = OBJECT_TYPE_SENSORPORT;

  sensorPortDesc.name = "xSpeedSensor";
  sensorPortDesc.fullName = fullName + ".xSpeedSensor";
  objectDescriptionTree.push_back(sensorPortDesc);
  sensorPortDesc.name = "ySpeedSensor";
  sensorPortDesc.fullName = fullName + ".ySpeedSensor";
  objectDescriptionTree.push_back(sensorPortDesc);
  sensorPortDesc.name = "zSpeedSensor";
  sensorPortDesc.fullName = fullName + ".zSpeedSensor";
  objectDescriptionTree.push_back(sensorPortDesc);
  sensorPortDesc.name = "xRotationSensor";
  sensorPortDesc.fullName = fullName + ".xRotationSensor";
  objectDescriptionTree.push_back(sensorPortDesc);
  sensorPortDesc.name = "yRotationSensor";
  sensorPortDesc.fullName = fullName + ".yRotationSensor";
  objectDescriptionTree.push_back(sensorPortDesc);
  sensorPortDesc.name = "zRotationSensor";
  sensorPortDesc.fullName = fullName + ".zRotationSensor";
  objectDescriptionTree.push_back(sensorPortDesc);
}

void MovableObject::setValue(double value, int port)
{
  switch(port)
  {
    case xSpeed: 
      speedX.value = value; 
      break;
    case ySpeed: 
      speedY.value = value;
      break;
    case zSpeed: 
      speedZ.value = value;
      break;
    case xRotation: 
      rotationSpeedX.value = value * M_PI/180.0;
      break;
    case yRotation:
      rotationSpeedY.value = value * M_PI/180.0;
      break;
    case zRotation: 
      rotationSpeedZ.value = value * M_PI/180.0;
      break;
    default: break;
  }
}

void MovableObject::act(bool initial)
{
  if(initial) // ignore initial step
    return;
  Vector3d trans;
  trans.v[0] = speedX.value * stepLength;
  trans.v[1] = speedY.value * stepLength;
  trans.v[2] = speedZ.value * stepLength;
  trans.rotate(rotation);
  translate(trans);
  Vector3d dummyPosition = position;
  Vector3d axis;
  
  Matrix3d relRotation;
  if(rotationSpeedX.value != 0.0)
  {
    Matrix3d XRotation;
    axis.init(1.0,0.0,0.0);
    axis.rotate(rotation);
    axis.normalize();
    XRotation.setRotationAroundAxis(axis,rotationSpeedX.value * stepLength);
    relRotation = XRotation * relRotation;
  }
  if(rotationSpeedY.value != 0.0)
  {
    Matrix3d YRotation;
    axis.init(0.0,1.0,0.0);
    axis.rotate(rotation);
    axis.normalize();
    YRotation.setRotationAroundAxis(axis,rotationSpeedY.value * stepLength);
    relRotation = YRotation * relRotation;
  }
  if(rotationSpeedZ.value != 0.0)
  {
    Matrix3d ZRotation;
    axis.init(0.0,0.0,1.0);
    axis.rotate(rotation);
    axis.normalize();
    ZRotation.setRotationAroundAxis(axis,rotationSpeedZ.value * stepLength);
    relRotation = ZRotation * relRotation;
  }
  rotate(relRotation, dummyPosition);
}

void MovableObject::computeValue(double& value, int portId)
{
  switch(portId)
  {
    case xSpeed: 
      value = speedX.value; 
      break;
    case ySpeed: 
      value = speedY.value; 
      break;
    case zSpeed: 
      value = speedZ.value; 
      break;
    case xRotation: 
      value = rotationSpeedX.value * 180/M_PI;
      break;
    case yRotation:
      value = rotationSpeedY.value * 180/M_PI;
      break;
    case zRotation: 
      value = rotationSpeedZ.value * 180/M_PI;
      break;
    default: break;
  }
}

SimObject* MovableObject::clone() const
{
  MovableObject* newObject  = new MovableObject();
  newObject->setName(name);
  newObject->setFullName(fullName);
  newObject->setPosition(position);
  newObject->rotation = rotation;
  newObject->rotationSpeedX = rotationSpeedX;
  newObject->rotationSpeedY = rotationSpeedY;
  newObject->rotationSpeedZ = rotationSpeedZ;
  newObject->speedX = speedX;
  newObject->speedY = speedY;
  newObject->speedZ = speedZ;
  newObject->childNodes.clear();
  std::list<SimObject*>::const_iterator pos;
  for(pos = childNodes.begin(); pos != childNodes.end(); ++pos)
  {
    SimObject* childNode = (*pos)->clone();
    newObject->addChildNode(childNode, false);
  }
  SimObject* returnObject = newObject;
  return returnObject;
}

bool MovableObject::isMovableOrClickable(std::string& nameOfObject,
                                         bool testClickable) const
{
  nameOfObject = fullName;
  return true;
}

/*
 * $Log: MovableObject.cpp,v $
 * Revision 1.2  2003/12/09 13:40:50  roefer
 * href attribute corrected
 *
 * Revision 1.7  2003/12/09 12:38:26  roefer
 * href attribute corrected
 *
 * Revision 1.6  2003/10/18 11:25:44  tim
 * - fixed intersection tests
 * - faster intersection test
 * - reimplementation of SimGeometry
 * - added portId for sensor calls
 * - finished sensor interfaces for joint and movableObject
 *
 * Revision 1.5  2003/10/12 13:19:13  tim
 * - added interactive buttons
 *
 * Revision 1.4  2003/09/28 14:50:04  roefer
 * Planes changed, initialValue for joints added
 *
 * Revision 1.3  2003/09/08 22:32:08  tim
 * - removed files
 * - added some doxygen documentation
 * - added some const qualifiers
 * - partial code clean-up
 * - minor code changes
 * - remove __ from guards (__ should only be used by compiler)
 *
 * Revision 1.2  2003/09/04 13:34:21  tim
 * - better parsing of numbers
 * - fixed macro bug
 * - better integration of macros in the object tree
 * - added getObjectReference() to Simulation
 * - faster object look-up in Simulation
 * - added changed log
 *
 */