/**
 * @file Simulation.cpp
 * 
 * Implementation of class Simulation
 *
 * @author <A href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</A>
 */

#include "Platform/OpenGL.h"
#include <cmath>
#include <cassert>

#include "Simulation.h"
#include "Debug.h"
#include "View.h"
#include "Actuator.h"
#include "Actuatorport.h"
#include "Surface.h"
#include "SAX2Parser.h"
#include "SimGeometry.h"
#include "InteractiveButton.h"


Simulation::Simulation(): stepLength(1.0), objectTree(0)
{
  simulationStatus = NOT_LOADED;
  sceneViewFar = 10.0;
  sceneViewNear = 0.1;
}

void Simulation::setStepLength(double stepLength)
{
  this->stepLength = stepLength;
  if(objectTree)
  {
    objectTree->setStepLength(stepLength);
  }
}

void Simulation::draw(double xAngle, double yAngle, double zAngle, double zoom, 
                      int width, int height, double fovy,
                      const VisualizationParameterSet& visParams,
                      const std::string& objectName)
{
  SimObject* drawingObject;
  if(objectName == "")
  {
    drawingObject = objectTree;
  }
  else
  {
    drawingObject = getObjectReference(objectName);
  }
  Vector3d center(drawingObject->getPosition());

  glViewport(0, 0, width,height);

  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  double angleRatio = double(width) / double(height);
  double sceneRadius = drawingObject->getIntersectionSphereRadius();
  sceneViewFar = zoom + sceneRadius;
  if(zoom > sceneRadius)
  {
    sceneViewNear = zoom - sceneRadius;
  }
  else
  {
    sceneViewNear = 0.1;
  }
  gluPerspective(fovy, angleRatio, sceneViewNear, sceneViewFar);
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  glTranslatef(0.0,0.0,0.0);

  // Set parameters for surfaces
  if(visParams.surfaceStyle == VisualizationParameterSet::SMOOTH_SHADING)
  {
    glShadeModel(GL_SMOOTH);
    glEnable(GL_POLYGON_SMOOTH); 
    glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);
    glEnable(GL_LIGHTING);
    GLfloat LightAmbient[]= 
    { 
      float(ambientColor.v[0]),
      float(ambientColor.v[1]),
      float(ambientColor.v[2]),
      1.0f 
    };  // Ambient Light Values ( NEW )
    GLfloat LightDiffuse[]= 
    { 
      float(1 - ambientColor.v[0]),
      float(1 - ambientColor.v[1]),
      float(1 - ambientColor.v[2]),
      1.0f 
    }; // Diffuse Light Values ( NEW )
    GLfloat LightPosition[]= { 0.0f, 0.0f, 10.0f, 1.0f };// Light Position ( NEW )
    glLightfv(GL_LIGHT1, GL_AMBIENT, LightAmbient);      // Setup The Ambient Light
    glLightfv(GL_LIGHT1, GL_DIFFUSE, LightDiffuse);      // Setup The Diffuse Light
    glLightfv(GL_LIGHT1, GL_POSITION,LightPosition);     // Position The Light
    glEnable(GL_LIGHT1);                                 // Enable Light One
    glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
    glEnable(GL_COLOR_MATERIAL);
    glLightModeli(GL_LIGHT_MODEL_TWO_SIDE,1);
  }
  else if(visParams.surfaceStyle == VisualizationParameterSet::WIREFRAME)
  {
    glLineWidth(1.0);
    glPolygonMode(GL_FRONT_AND_BACK,GL_LINE);
    glShadeModel(GL_FLAT);
    glDisable(GL_LIGHTING);
  }
  else //if(visParams.surfaceStyle == VisualizationParameterSet::FLAT_SHADING)
  {
    glShadeModel(GL_FLAT);
    glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);
    glDisable(GL_LIGHTING);
  }
  glEnable(GL_DEPTH_TEST);

  //set new view
  gluLookAt(center.v[0], center.v[1], center.v[2] + zoom, 
            center.v[0], center.v[1], center.v[2], 0.0, 1.0, 0.0);
  glTranslatef((float) center.v[0],(float) center.v[1],(float) center.v[2]);
  glRotatef((float) xAngle, 1.0f, 0.0f, 0.0f);
  glRotatef((float) yAngle, 0.0f, 1.0f, 0.0f);
  glRotatef((float) zAngle, 0.0f, 0.0f, 1.0f);
  glTranslatef((float) -center.v[0],(float) -center.v[1],(float) -center.v[2]);
  // Clear and draw
  glClearColor((float) backgroundColor.v[0],
               (float) backgroundColor.v[1],
               (float) backgroundColor.v[2],0);
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  Vector3d pointOfView(0.0,0.0,zoom);
  pointOfView.rotateX(xAngle*M_PI/180.0);
  pointOfView.rotateY(yAngle*M_PI/180.0);
  pointOfView.rotateZ(zAngle*M_PI/180.0);
  pointOfView += center;
  if(drawingObject)
  {
    drawingObject->draw(pointOfView, visParams);
  }
}


double Simulation::getOptimalZoomForObject(ZoomFitType zoomType,
                                           int width, int height, double fovy,
                                           const std::string& objectName) 
{
  if((width == 0) || (height == 0))
  {
    return 0.0;
  }
  double alpha(fovy * M_PI / 180.0);
  if(zoomType != ZOOM_FIT_HEIGHT)
  {
    double maxWidth(120.0*M_PI/180.0);
    double widthAngle(alpha * ((double)width/(double)height));
    if(widthAngle > maxWidth)
    {
      widthAngle = maxWidth;
    }
    if(zoomType == ZOOM_FIT_WIDTH)
    {
      alpha = widthAngle;
    }
    else //(zoomType == ZOOM_FIT)
    {
      if(widthAngle < alpha)
      {
        alpha = widthAngle;
      }
    }
  }
  alpha /= 2.0;
  double beta((M_PI/2.0)-alpha);
  double r;
  if(objectName == "")
  {
    r = objectTree->getIntersectionSphereRadius();
  }
  else
  {
    r = getObjectReference(objectName)->getIntersectionSphereRadius();
  }
  double realR (r/sin(beta));
  return (realR/tan(alpha));
}


void Simulation::setVisualizeSensors(bool visualizeSensors)
{
  assert(objectTree);
  objectTree->setVisualizeSensors(visualizeSensors);
}


void Simulation::projectClick(int x, int y, Vector3d& viewerPosition, Vector3d& vecToClick) const
{
  double aspect = double(selection.width) / double(selection.height);
  double rotationX = -selection.xAngle * M_PI / 180.0;
  double rotationY = selection.yAngle * M_PI / 180.0;
  double rotationZ = -selection.zAngle * M_PI / 180.0;
  viewerPosition = Vector3d(0.0,0.0,selection.zoom);
  Vector3d centerPosition(0.0,0.0,selection.zoom - sceneViewNear);
  Vector3d alongXAxis(1.0,0.0,0.0);
  Vector3d alongYAxis(0.0,1.0,0.0);
  alongXAxis *= tan(selection.fovy * M_PI / 360.0) * aspect * sceneViewNear;
  alongYAxis *= tan(selection.fovy * M_PI / 360.0) * sceneViewNear;
  Vector3d endOfXAxis(centerPosition + alongXAxis);
  Vector3d endOfYAxis(centerPosition + alongYAxis);
  viewerPosition.rotateX(rotationX);
  viewerPosition.rotateY(rotationY);
  viewerPosition.rotateZ(rotationZ);
  centerPosition.rotateX(rotationX);
  centerPosition.rotateY(rotationY);
  centerPosition.rotateZ(rotationZ);
  endOfXAxis.rotateX(rotationX);
  endOfXAxis.rotateY(rotationY);
  endOfXAxis.rotateZ(rotationZ);
  endOfYAxis.rotateX(rotationX);
  endOfYAxis.rotateY(rotationY);
  endOfYAxis.rotateZ(rotationZ);
  double scaleX = double(x) / (double(selection.width) / 2.0) - 1.0;
  double scaleY = double(y) / (double(selection.height) / 2.0) - 1.0;

  Vector3d Xcomponent = endOfXAxis - centerPosition;
  Vector3d Ycomponent = endOfYAxis - centerPosition;
  Xcomponent *= scaleX;
  Ycomponent *= -scaleY;
  Vector3d clickPosition = centerPosition;
  clickPosition += Xcomponent + Ycomponent;

  vecToClick = clickPosition - viewerPosition;
  viewerPosition += selection.root->getPosition();
}

InteractiveSelectionType Simulation::selectObject(
                int x, int y, double xAngle, double yAngle, double zAngle, 
                double zoom, int width, int height, double fovy, DragAndDropPlane plane,
                const std::string& objectName, bool selectButtons)
{
  selection.xAngle = xAngle;
  selection.yAngle = yAngle;
  selection.zAngle = zAngle;
  selection.zoom = zoom;
  selection.fovy = fovy;
  selection.width = width;
  selection.height = height;
  selection.root = objectName == "" ? objectTree : getObjectReference(objectName);
  switch(plane)
  {
    case XY_PLANE:
      selection.plane = Vector3d(0,0,1);
      break;
    case XZ_PLANE:
      selection.plane = Vector3d(0,1,0);
      break;
    case YZ_PLANE:
      selection.plane = Vector3d(1,0,0);
      break;
  }

  Vector3d viewerPosition, 
           vecToClick;
  projectClick(x, y, viewerPosition, vecToClick);

  std::vector<Intersection*> intersections;
  bool result = selection.root->intersect(intersections, viewerPosition, vecToClick);

  //Check if any objects have been selected
  if(result)
  {
    //find all intersected movables and clickables
    std::vector<Intersection*> intersectedInteractives;
    for(unsigned int j=0; j<intersections.size(); j++)
    {
      SimObject* intersectedObject;
      objectTree->findObject(intersectedObject, intersections[j]->objectName);
      std::string nameOfInteractive;
      // don't select the root object of a view
      if(intersectedObject->isMovableOrClickable(nameOfInteractive,selectButtons) && selection.root->getFullName() != nameOfInteractive)
      {
        intersectedInteractives.push_back(intersections[j]);
      }
    }
    result = intersectedInteractives.size() != 0;
    if(result)
    {
      //Find nearest intersection:
      double shortestDistance = (intersectedInteractives[0]->position - viewerPosition).getLength();
      double idxShortest = 0;
      unsigned int k;
      for(k=0; k<intersectedInteractives.size(); k++)
      {
        double dist = (intersectedInteractives[k]->position - viewerPosition).getLength();
        if(dist<shortestDistance)
        {
          shortestDistance = dist;
          idxShortest = k;
        }
      }
      std::string name(intersectedInteractives[(int) idxShortest]->objectName);
      SimObject* dummyObject;
      objectTree->findObject(dummyObject, name);
      dummyObject->isMovableOrClickable(name,selectButtons);
      objectTree->findObject(selection.object, name);
      selection.object->invertColors();
      Vector3d planeIntersection;
      SimGeometry::intersectRayAndPlane(viewerPosition, vecToClick, 
                                        selection.object->getPosition(),
                                        selection.plane, planeIntersection);
      selection.clickOffset = planeIntersection - selection.object->getPosition();
    }
    //Delete all intersection objects and clear vectors:
    for(unsigned int k=0; k<intersections.size(); k++)
      delete intersections[k];
    intersections.clear();
    intersectedInteractives.clear();
    if(result)
    {
      std::string kind(selection.object->getKind());
      if(kind == "movable")
      {
        return OBJECT_SELECTION;
      }
      else //(kind == "interactiveButton")
      {
        selection.object->hasBeenClicked();
        return BUTTON_SELECTION;
      }
    }
    else
    {
      return NO_SELECTION;
    }
  }
  else
  {
    return NO_SELECTION;
  }
}

void Simulation::unselectObject()
{
  selection.object->getParentNode()->computeIntersectionSphereRadius();
  selection.object->invertColors();
  selection.object->hasBeenReleased();
  selection.object = 0;
}

void Simulation::translateObject(int x, int y)
{
  assert(selection.object);
  Vector3d viewerPosition, 
           vecToClick;
  projectClick(x, y, viewerPosition, vecToClick);
  Vector3d planeIntersection;
  if(SimGeometry::intersectRayAndPlane(viewerPosition, vecToClick, selection.object->getPosition(),
                                       selection.plane, planeIntersection))
  {
    Vector3d translationVec = planeIntersection - selection.object->getPosition() - selection.clickOffset;
    selection.object->translate(translationVec);
  }
}

void Simulation::rotateObject(int x, int y)
{
  assert(selection.object);
  Vector3d viewerPosition, 
           vecToClick;
  projectClick(x, y, viewerPosition, vecToClick);
  Vector3d planeIntersection;
  SimGeometry::intersectRayAndPlane(viewerPosition, vecToClick, selection.object->getPosition(),
                                    selection.plane, planeIntersection);
  Vector3d clickOffset = planeIntersection - selection.object->getPosition();
  double angle;
  
  if(selection.plane.v[0])
    angle = atan2(clickOffset.v[2], clickOffset.v[1]) -
            atan2(selection.clickOffset.v[2], selection.clickOffset.v[1]);
  else if(selection.plane.v[1])
    angle = atan2(clickOffset.v[0], clickOffset.v[2]) -
            atan2(selection.clickOffset.v[0], selection.clickOffset.v[2]);
  else
    angle = atan2(clickOffset.v[1], clickOffset.v[0]) -
            atan2(selection.clickOffset.v[1], selection.clickOffset.v[0]);
  selection.object->rotateAroundAxis(angle, selection.plane);
  selection.clickOffset = clickOffset;
}

bool Simulation::loadFile(const std::string& filename, const std::string& dtdname)
{
  this->filename = filename;
  this->dtdname = dtdname;
  closeSimulation();
  parser = new SAX2Parser();
  parser->parse(filename, dtdname, this, objectTree, &errorManager,
                &sensorportList, &actuatorportList,
                &actuatorList, &surfaces);
  delete parser;
  if(objectTree)
  {
    simulationStep = 0;
    objectTree->setSimulationStep(simulationStep);
    objectTree->setStepLength(stepLength);
    objectTree->setBackgroundColor(backgroundColor);
    objectTree->setAmbientColor(ambientColor);
    for(unsigned int i=0; i<actuatorList.size(); i++)
    {
      actuatorList[i]->act(true);
    }
    simulationStatus = SCENE_LOADED;
    sceneGraphChanged = true;
    return true;
  }
  else
  {
    sceneGraphChanged = false;
    return false;
  }
}

void Simulation::closeSimulation()
{
  {
    std::list<View*>::iterator iter;
    for(iter = viewList.begin(); iter != viewList.end(); ++iter)
      delete (*iter);
    viewList.clear();
  }
  {
    std::vector<Surface*>::iterator iter;
    for(iter = surfaces.begin(); iter != surfaces.end(); ++iter)
      delete (*iter);
    surfaces.clear();
  }
  {
    std::vector<Actuatorport*>::iterator iter;
    for(iter = actuatorportList.begin(); iter != actuatorportList.end(); ++iter)
      delete (*iter);
    actuatorportList.clear();
  }
  {
    std::vector<Sensorport*>::iterator iter;
    for(iter = sensorportList.begin(); iter != sensorportList.end(); ++iter)
      delete (*iter);
    sensorportList.clear();
  }
  actuatorList.clear();
  if(objectTree)
  {
    delete objectTree;
    objectTree = 0;
  }
  simulationStatus = NOT_LOADED;
  selection.object = 0;
}

void Simulation::resetSimulation()
{
  if(loadFile(filename, dtdname))
  {
    //Update views
    std::list<View*>::const_iterator pos;
    for(pos = viewList.begin(); pos != viewList.end(); ++pos)
    {
      (*pos)->update();
    }
  }
}

void Simulation::doSimulationStep()
{
  //Gain information from actuator windows:

  //Let actuators manipulate the scene
  for(unsigned int i=0; i<actuatorList.size(); i++)
  {
    actuatorList[i]->act();
  }

  //Update views
  std::list<View*>::const_iterator pos;
  for(pos = viewList.begin(); pos != viewList.end(); ++pos)
  {
    (*pos)->update();
  }

  //Set next step
  ++simulationStep;
  objectTree->setSimulationStep(simulationStep);
  sceneGraphChanged = false;
}

SimObject* Simulation::getObjectReference(const std::string& objectName)
{
  NameToPointerMap::const_iterator pos;
  pos = objectMap.find(objectName);
  if(pos == objectMap.end())
  {
    SimObject* newObject;
    objectTree->findObject(newObject, objectName);
    assert(newObject != 0);
    objectMap[objectName] = newObject;
    return newObject;
  }
  else
  {
    return pos->second;
  }
}

int Simulation::getActuatorportId(const std::string& actuatorportName) const
{
  int result = -1;
  for(unsigned int i=0; i<actuatorportList.size(); i++)
  {
    if(actuatorportList[i]->getName() == actuatorportName)
    {
      result = i;
      break;
    }
  }
  assert(result>=0);
  return result;
}

double Simulation::getActuatorportMinValue(int id) const
{
  assert(id >= 0 && id < (int) actuatorportList.size());
  return actuatorportList[id]->getMinValue();
}

double Simulation::getActuatorportMaxValue(int id) const
{
  assert(id >= 0 && id < (int) actuatorportList.size());
  return actuatorportList[id]->getMaxValue();
}

void Simulation::setActuatorport(int id, double value)
{
  assert(id >= 0 && id < (int) actuatorportList.size());
  actuatorportList[id]->setValue(value);
}

void Simulation::setActuatorport(int id, bool value)
{
  assert(id >= 0 && id < (int) actuatorportList.size());
  actuatorportList[id]->setValue(value);
}

int Simulation::getSensorportId(const std::string& sensorportName) const
{
  int result = -1;
  for(unsigned int i=0; i<sensorportList.size(); i++)
  {
    if(sensorportList[i]->getName() == sensorportName)
    {
      result = i;
      break;
    }
  }
  assert(result>=0);
  return result;
}

const std::vector<int>& Simulation::getSensorDimensions(int id) const
{
  assert(id >= 0 && id < (int) sensorportList.size());
  return sensorportList[id]->getDimensions();
}

void Simulation::getSensorportValue(int id, bool& value)
{
  assert(id >= 0 && id < (int) sensorportList.size());
  sensorportList[id]->getValue(value);
}

void Simulation::getSensorportValue(int id, double& value)
{
  assert(id >= 0 && id < (int) sensorportList.size());
  sensorportList[id]->getValue(value);
}

void Simulation::getSensorportValue(int id, int& value)
{
  assert(id >= 0 && id < (int) sensorportList.size());
  sensorportList[id]->getValue(value);
}

void Simulation::getSensorportValue(int id, double*& value)
{
  assert(id >= 0 && id < (int) sensorportList.size());
  sensorportList[id]->getValue(value);
}

void Simulation::getSensorportValue(int id, unsigned char*& value)
{
  assert(id >= 0 && id < (int) sensorportList.size());
  sensorportList[id]->getValue(value);
}

SensorType Simulation::getSensorportType(int id) const
{
  assert(id >= 0 && id < (int) sensorportList.size());
  return sensorportList[id]->getType();
}

double Simulation::getSensorportMinValue(int id) const
{
  assert(id >= 0 && id < (int) sensorportList.size());
  return sensorportList[id]->getMinValue();
}

double Simulation::getSensorportMaxValue(int id) const
{
  assert(id >= 0 && id < (int) sensorportList.size());
  return sensorportList[id]->getMaxValue();
}

Surface* Simulation::getSurface(const std::string& name) const
{
  std::vector<Surface*>::const_iterator iter;
  for(iter = surfaces.begin(); iter != surfaces.end(); ++iter)
    if((*iter)->name == name)
      return *iter;
  assert(false);
  return 0;
}

void Simulation::addView(View* view,const std::string& name)
{
  view->setName(name);
  view->setFullName(objectTree->getFullName() + ".views." + name);
  view->addToLists(sensorportList, actuatorportList, actuatorList);
  viewList.push_back(view);
  sceneGraphChanged = true;
}

View* Simulation::getView(int id)
{
  assert(id >= 0 && id < (int) sensorportList.size());
  assert(sensorportList[id]->getType() == viewSensor);
  return (View*) sensorportList[id]->getSensor();
}

void Simulation::getObjectDescriptionTree(std::vector<ObjectDescription>& objectDescriptionTree) const
{
  objectDescriptionTree.clear();
  if(objectTree)
  {
    objectTree->addToDescriptions(objectDescriptionTree, 0);
  }
  if(!viewList.empty())
  {
    ObjectDescription viewsDesc;
    viewsDesc.name = "views";
    viewsDesc.fullName = objectTree->getFullName() + ".views";
    viewsDesc.depth = 1;
    viewsDesc.type = OBJECT_TYPE_NONE;
    objectDescriptionTree.push_back(viewsDesc);
    std::list<View*>::const_iterator pos;
    std::string path = "";
    int depth = 1;
    for(pos = viewList.begin(); pos != viewList.end(); ++pos)
    {
      // shorten path
      while((*pos)->getName().compare(0, path.length(), path))
      {
        int p = path.find_last_of('.', path.length() - 2);
        path = p == -1 ? "" : path.substr(0, p + 1);
        --depth;
      }

      // lengthen path
      for(;;)
      {
        int p = (*pos)->getName().find_first_of('.', path.length() + 1);
        if(p == -1)
          break;
        ++depth;
        ObjectDescription folderDesc;
        folderDesc.name = (*pos)->getName().substr(path.length(), p - path.length());
        path = (*pos)->getName().substr(0, p + 1);
        folderDesc.fullName = path.substr(0, path.length() - 1);
        folderDesc.depth = depth;
        folderDesc.type = OBJECT_TYPE_NONE;
        objectDescriptionTree.push_back(folderDesc);
      }
      (*pos)->addToDescriptions(objectDescriptionTree, depth + 1);
    }
  }
}

void Simulation::getFirstError(ErrorDescription& error)
{
  errorManager.getFirstError(error);
  errorManager.deleteFirstError();
}

void Simulation::getAllErrors(std::vector<ErrorDescription>& errors)
{
  errorManager.getAllErrors(errors);
  errorManager.deleteAllErrors();
}

void Simulation::deleteAllErrors()
{
  errorManager.deleteAllErrors();
}

/*
 * $Log: Simulation.cpp,v $
 * Revision 1.5  2004/04/20 13:14:53  roefer
 * All console commands now also work outside the start script
 *
 * Revision 1.4  2004/04/09 18:55:16  roefer
 * Better structure for views
 * cp command added
 * Timing view added
 *
 * Revision 1.3  2003/12/09 13:40:53  roefer
 * href attribute corrected
 *
 * Revision 1.40  2003/12/09 12:38:28  roefer
 * href attribute corrected
 *
 * Revision 1.39  2003/12/03 18:12:37  roefer
 * Compatibility with VC2003.NET, GUI does still not work completely there
 *
 * Revision 1.38  2003/10/21 22:32:56  roefer
 * Documented controller
 *
 * Revision 1.37  2003/10/21 12:56:32  roefer
 * onSelected added
 *
 * Revision 1.36  2003/10/20 17:11:18  roefer
 * Ambient light added
 *
 * Revision 1.35  2003/10/18 20:07:31  roefer
 * DirectView added
 *
 * Revision 1.34  2003/10/18 11:25:44  tim
 * - fixed intersection tests
 * - faster intersection test
 * - reimplementation of SimGeometry
 * - added portId for sensor calls
 * - finished sensor interfaces for joint and movableObject
 *
 * Revision 1.33  2003/10/13 05:10:50  roefer
 * Progress with SimGT2004
 *
 * Revision 1.32  2003/10/12 13:19:13  tim
 * - added interactive buttons
 *
 * Revision 1.31  2003/10/12 10:24:51  roefer
 * getSurface added to Simulation
 *
 * Revision 1.30  2003/10/07 08:20:47  roefer
 * Calculation simplified
 *
 * Revision 1.29  2003/10/05 15:24:30  tim
 * - changed drag & drop visualization
 *
 * Revision 1.28  2003/10/01 18:57:27  roefer
 * DTD is automatically selected now
 *
 * Revision 1.27  2003/09/28 14:50:04  roefer
 * Planes changed, initialValue for joints added
 *
 * Revision 1.26  2003/09/25 15:14:40  roefer
 * Selection corrected
 *
 * Revision 1.25  2003/09/25 13:41:19  roefer
 * Drag and drop GUI, corrections for sub-object views
 *
 * Revision 1.24  2003/09/25 07:54:57  roefer
 * Drag and drop/rotate in all three axes
 *
 * Revision 1.23  2003/09/23 07:41:35  roefer
 * Drag and rotate added
 *
 * Revision 1.22  2003/09/22 15:43:36  roefer
 * Memory leak closed in selectObject
 *
 * Revision 1.21  2003/09/21 12:36:09  roefer
 * Made stepLength optional
 *
 * Revision 1.20  2003/09/18 01:52:09  tim
 * - changed OpenGL surface computation
 * - added stepLength
 *
 * Revision 1.19  2003/09/17 02:39:26  roefer
 * GL_FRONT_AND_BACK used everywhere
 *
 * Revision 1.18  2003/09/12 11:34:14  tim
 * - added sensor visualization framework
 * - implemented visualization for whisker
 *
 * Revision 1.17  2003/09/12 05:59:40  roefer
 * SensorportMin/MaxValue bugs fixed
 *
 * Revision 1.16  2003/09/11 22:54:05  tim
 * - background color now in camera images, too
 *
 * Revision 1.15  2003/09/11 22:10:57  tim
 * - added new zoom functions
 * - added changing of opening angle
 *
 * Revision 1.14  2003/09/10 10:09:14  tim
 * - added comments
 * - small interface changes
 *
 * Revision 1.13  2003/09/09 16:03:14  roefer
 * New simulation scene sets SimObject::simulationStep now
 *
 * Revision 1.12  2003/09/08 22:32:08  tim
 * - removed files
 * - added some doxygen documentation
 * - added some const qualifiers
 * - partial code clean-up
 * - minor code changes
 * - remove __ from guards (__ should only be used by compiler)
 *
 * Revision 1.11  2003/09/08 15:37:08  tim
 * - changed shading parameters
 *
 * Revision 1.10  2003/09/08 14:46:59  tim
 * - added surface style selection
 *
 * Revision 1.9  2003/09/08 11:46:54  tim
 * - added Simulation::getSensorportMinValue
 * - added Simulation::getSensorportMaxValue
 *
 * Revision 1.8  2003/09/07 17:51:08  roefer
 * Sensor and actuator view added, interface changes for simulation
 *
 * Revision 1.7  2003/09/06 21:20:21  roefer
 * Memory leaks in error case removed
 *
 * Revision 1.6  2003/09/06 20:53:00  roefer
 * Test case removed
 *
 * Revision 1.5  2003/09/05 21:01:27  tim
 * - drawn objects are now always at the point of origin
 *
 * Revision 1.4  2003/09/05 20:40:52  tim
 * - changed getObjectReference
 *
 * Revision 1.3  2003/09/05 20:20:24  tim
 * - added background color for scene
 *
 * Revision 1.2  2003/09/04 13:34:22  tim
 * - better parsing of numbers
 * - fixed macro bug
 * - better integration of macros in the object tree
 * - added getObjectReference() to Simulation
 * - faster object look-up in Simulation
 * - added changed log
 *
 */