/**
* @file Kinematics.cpp
*
* Class Kinematics provides methods for robots kinematic calculations
*
* @author Max Risler
* @author Uwe Dffert
*/

#include "Kinematics.h"
#include "Tools/RobotConfiguration.h"
#include "Tools/Math/Geometry.h"
#include "Tools/Math/Pose3D.h"
#include "Tools/Debugging/Debugging.h"


void Kinematics::getRelativeRobotVertices(RobotVertices& robotVertices, const JointData& jointData)
{
    RobotVertices rob(getRobotConfiguration().getRobotDimensions()); //get shoulder positions from standard constructor

    kneePositionFromJoints(fr,
        fromMicroRad(jointData.data[JointData::legFR1]),
        fromMicroRad(jointData.data[JointData::legFR2]),
        rob.kneePosition[0].x, rob.kneePosition[0].y, rob.kneePosition[0].z);

    kneePositionFromJoints(fl,
        fromMicroRad(jointData.data[JointData::legFL1]),
        fromMicroRad(jointData.data[JointData::legFL2]),
        rob.kneePosition[1].x, rob.kneePosition[1].y, rob.kneePosition[1].z);

    kneePositionFromJoints(hr,
        fromMicroRad(jointData.data[JointData::legHR1]),
        fromMicroRad(jointData.data[JointData::legHR2]),
        rob.kneePosition[2].x, rob.kneePosition[2].y, rob.kneePosition[2].z);

    kneePositionFromJoints(hl,
        fromMicroRad(jointData.data[JointData::legHL1]),
        fromMicroRad(jointData.data[JointData::legHL2]),
        rob.kneePosition[3].x, rob.kneePosition[3].y, rob.kneePosition[3].z);


    legPositionFromJoints(fr,
        fromMicroRad(jointData.data[JointData::legFR1]),
        fromMicroRad(jointData.data[JointData::legFR2]),
        fromMicroRad(jointData.data[JointData::legFR3]),
        rob.footPosition[0].x, rob.footPosition[0].y, rob.footPosition[0].z);

    legPositionFromJoints(fl,
        fromMicroRad(jointData.data[JointData::legFL1]),
        fromMicroRad(jointData.data[JointData::legFL2]),
        fromMicroRad(jointData.data[JointData::legFL3]),
        rob.footPosition[1].x, rob.footPosition[1].y, rob.footPosition[1].z);

    legPositionFromJoints(hr,
        fromMicroRad(jointData.data[JointData::legHR1]),
        fromMicroRad(jointData.data[JointData::legHR2]),
        fromMicroRad(jointData.data[JointData::legHR3]),
        rob.footPosition[2].x, rob.footPosition[2].y, rob.footPosition[2].z);

    legPositionFromJoints(hl,
        fromMicroRad(jointData.data[JointData::legHL1]),
        fromMicroRad(jointData.data[JointData::legHL2]),
        fromMicroRad(jointData.data[JointData::legHL3]),
        rob.footPosition[3].x, rob.footPosition[3].y, rob.footPosition[3].z);

    for (int i=0;i<4;i++)
    {
        rob.footPosition[i] += rob.shoulderPosition[i];
        rob.kneePosition[i] += rob.shoulderPosition[i];
    }
    robotVertices = rob;
}

void Kinematics::getRobotTransformation(double& tilt,double& roll, const RobotVertices& relativeRobotVertices, const GroundMode mode, double expectedBodyTilt, double expectedBodyRoll)
{
    switch (mode)
    {
    case constantMode:
    case flyMode:
        //assumption: user knows it better, eg from odometry.cfg
        tilt=expectedBodyTilt;
        roll=expectedBodyRoll;
        break;
    case dynamicMode:
        {
            //assumption: bodyRoll=0, feet or knees on ground
            roll=0;
            Vector2<double>front[4];
            Vector2<double>hind[4];
            int i;
            double lowLowTilt;
            //create a list of all relevant knee and foot points
            for (i=0;i<2;i++)
            {
                front[i]=Vector2<double>(relativeRobotVertices.footPosition[i].x,relativeRobotVertices.footPosition[i].z);
                front[i+2]=Vector2<double>(relativeRobotVertices.kneePosition[i].x,relativeRobotVertices.kneePosition[i].z);
                hind[i]=Vector2<double>(relativeRobotVertices.footPosition[i+2].x,relativeRobotVertices.footPosition[i+2].z);
                hind[i+2]=Vector2<double>(relativeRobotVertices.kneePosition[i+2].x,relativeRobotVertices.kneePosition[i+2].z);
            }
            int lowestFront,lowestHind;
            for (lowestFront=0;lowestFront<4;lowestFront++)
            {
                //find the lowest hind point for each front point
                lowestHind=0;
                for (int j=1;j<4;j++)
                {
                    if ((front[lowestFront]-hind[j]).angle()>(front[lowestFront]-hind[lowestHind]).angle())
                    {
                        lowestHind=j;
                    }
                }
                lowLowTilt=(front[lowestFront]-hind[lowestHind]).angle();
                //check whether the used front point is the lowestpoint for the found hind point
                if (((front[0]-hind[lowestHind]).angle()>lowLowTilt-0.0001)&&
                    ((front[1]-hind[lowestHind]).angle()>lowLowTilt-0.0001)&&
                    ((front[2]-hind[lowestHind]).angle()>lowLowTilt-0.0001)&&
                    ((front[3]-hind[lowestHind]).angle()>lowLowTilt-0.0001))
                {
                    //in this case we have found the pair of the lowest front and hind point
                    break;
                }
            }
            tilt=lowLowTilt;
            break;
        }
    case staticMode:
        //assumption: feet on ground
        /** @todo calulate lower feet diagonale calculate lower supporting
        * feet triangle by checking on which side of the diagonale the center
        * of gravity is
        */
        break;
    case superMode:
        //assumption: robots CG falls from expected bodyTilt/bodyRoll (e.g. given
        //by sensor or previous call) to static position (side,front,hind,feet)
        /** @todo thats quite tricky, but we may just check falling to
        * side/front/hind and return staticMode solution otherwise
        */
        break;
    default:
        //OUTPUT(idText,text,"unknown GroundMode in Kinematics::getRobotTransformation()");
        break;
    }
}

void Kinematics::getAbsoluteRobotVertices(RobotVertices& robotVertices, const GroundMode mode, const JointData& jointData, double expectedBodyTilt, double expectedBodyRoll)
{
    const RobotDimensions& r = getRobotConfiguration().getRobotDimensions();
    RobotVertices relativeRobotVertices;
    getRelativeRobotVertices(relativeRobotVertices, jointData);
    double tilt,roll;
    getRobotTransformation(tilt,roll, relativeRobotVertices, mode, expectedBodyTilt, expectedBodyRoll);
    robotVertices.bodyTilt=tilt;
    robotVertices.bodyRoll=roll;
    robotVertices.neck=Vector3<double>(0,0,0);
    int i;
    for (i=0;i<4; i++)
    {
        {
            robotVertices.footPosition[i].y=relativeRobotVertices.footPosition[i].y-relativeRobotVertices.neck.y;
            Vector2<double>dist(relativeRobotVertices.footPosition[i].x-relativeRobotVertices.neck.x,relativeRobotVertices.footPosition[i].z-relativeRobotVertices.neck.z);
            double arc=dist.angle()-(tilt-r.zeroBodyTilt);
            robotVertices.footPosition[i].x=cos(arc)*dist.abs();
            robotVertices.footPosition[i].z=sin(arc)*dist.abs();
        }
        {
            robotVertices.kneePosition[i].y=relativeRobotVertices.kneePosition[i].y-relativeRobotVertices.neck.y;
            Vector2<double>dist(relativeRobotVertices.kneePosition[i].x-relativeRobotVertices.neck.x,relativeRobotVertices.kneePosition[i].z-relativeRobotVertices.neck.z);
            double arc=dist.angle()-(tilt-r.zeroBodyTilt);
            robotVertices.kneePosition[i].x=cos(arc)*dist.abs();
            robotVertices.kneePosition[i].z=sin(arc)*dist.abs();
        }
        {
            robotVertices.shoulderPosition[i].y=relativeRobotVertices.shoulderPosition[i].y-relativeRobotVertices.neck.y;
            Vector2<double>dist(relativeRobotVertices.shoulderPosition[i].x-relativeRobotVertices.neck.x,relativeRobotVertices.shoulderPosition[i].z-relativeRobotVertices.neck.z);
            double arc=dist.angle()-(tilt-r.zeroBodyTilt);
            robotVertices.shoulderPosition[i].x=cos(arc)*dist.abs();
            robotVertices.shoulderPosition[i].z=sin(arc)*dist.abs();
        }
    }

    //move the hovering robot to ground :-)
    double minZ=0;
    if (mode==flyMode)
    {
        minZ=-180;
    }
    else
    {
        for (i=0;i<4;i++)
        {
            if (robotVertices.footPosition[i].z<minZ) minZ=robotVertices.footPosition[i].z;
            if (robotVertices.kneePosition[i].z<minZ) minZ=robotVertices.kneePosition[i].z;
        }
    }
    for (i=0;i<4;i++)
    {
        robotVertices.footPosition[i].z -= minZ;
        robotVertices.kneePosition[i].z -= minZ;
        robotVertices.shoulderPosition[i].z -= minZ;
    }
    robotVertices.neck.z -= minZ;
}

void Kinematics::legPositionFromJoints(LegIndex leg, double j1, double j2, double j3, double &x, double &y, double &z, bool correctCalculation)
{
    const RobotDimensions& r = getRobotConfiguration().getRobotDimensions();
    double lowerleg;

    if (correctCalculation) j1 += r.zeroShoulderArc; //this is because knee is not in the x-middle of a leg

    if (FORELEG(leg))
    {
        if (correctCalculation) j3 -= r.zeroFrontKneeArc; //this is because knee is not in the x-middle of a leg
        lowerleg=r.lowerForeLegLength;
    }
    else
    {
        if (correctCalculation) j3 -= r.zeroHindKneeArc; //this is because knee is not in the x-middle of a leg
        lowerleg=r.lowerHindLegLength;
    }

    double s1=sin(j1),s2=sin(j2),s3=sin(j3);
    double c1=cos(j1),c2=cos(j2),c3=cos(j3);


    //calculate x,y relative to leg base
    x=lowerleg*s1*c2*c3+lowerleg*c1*s3+r.upperLegLength*s1*c2;
    y=lowerleg*s2*c3+r.upperLegLength*s2;

    //calculate z
    z=-lowerleg*c1*c2*c3+lowerleg*s1*s3-r.upperLegLength*c1*c2;

    if (LEGBASE(leg)) return;

    if (!FORELEG(leg))
        x=-x;


    if (!(LEFTLEG(leg)))
    {
        y=-y;
    }
}

void Kinematics::kneePositionFromJoints(LegIndex leg, double j1, double j2, double &x, double &y, double &z, bool correctCalculation)
{
    const RobotDimensions& r = getRobotConfiguration().getRobotDimensions();

    if (correctCalculation) j1 += r.zeroShoulderArc; //this is because knee is not in the x-middle of a leg

    double s1=sin(j1),s2=sin(j2);
    double c1=cos(j1),c2=cos(j2);

    //calculate x,y relative to leg base
    x=r.upperLegLength*s1*c2;
    y=r.upperLegLength*s2;

    //calculate z
    z=-r.upperLegLength*c1*c2;

    if (LEGBASE(leg)) return;

    if (!FORELEG(leg))
        x=-x;

    if (!(LEFTLEG(leg)))
    {
        y=-y;
    }
}

void Kinematics::transformToLegBase(LegIndex leg,double &x, double &y, double &z, double &lowerleg) {
    const RobotDimensions& r = getRobotConfiguration().getRobotDimensions();
    if (FORELEG(leg))
        lowerleg=r.lowerForeLegLength;
    else
        lowerleg=r.lowerHindLegLength;

    if (LEGBASE(leg)) return;

    //position relative to leg base
    if (!FORELEG(leg))
        x=-x;

    if (LEFTLEG(leg))
        y=y-r.bodyWidth/2;
    else
        y=-y-r.bodyWidth/2;
}

bool Kinematics::jointsFromLegPosition(LegIndex leg, double x, double y, double z, double &j1, double &j2, double &j3, double bodyTilt, bool correctCalculation)
{
    const RobotDimensions& r = getRobotConfiguration().getRobotDimensions();
    double a,b,c,d,c2;
    double lowerleg;
    double beta;

    //double ox=x,oy=y,oz=z;

    bool calcError=false;

    transformToLegBase(leg,x,y,z,lowerleg);

    //Beintransformation :
    //(x,y,z,1) = Rot(y,-q1)*Rot(x,q2)*Trans(0,0,-u)*Rot(y,-q3)*Trans(0,0,-l)*(0,0,0,1)

    // Winkel j3 ist der von einem Dreieck mit laengster Seite 
    // gleich der Laenge von Vektor (x,y,z)
    // j3 mit Kosinussatz

    c=x*x+y*y+z*z;

    if (c>(lowerleg+r.upperLegLength)*(lowerleg+r.upperLegLength)) {
        //OUTPUT(idText,"JointsFromLegPosition: Error "<<stringFromLeg(leg) <<" too far");
        c=(lowerleg+r.upperLegLength)*(lowerleg+r.upperLegLength);
        calcError=true;
    }
    if (c<(lowerleg-r.upperLegLength)*(lowerleg-r.upperLegLength)) {
        //OUTPUT(idText,"JointsFromLegPosition: Error "<<stringFromLeg(leg) <<" too close");
        c=(lowerleg-r.upperLegLength)*(lowerleg-r.upperLegLength);
        calcError=true;
    }

    d=(c-lowerleg*lowerleg-r.upperLegLength*r.upperLegLength)/(2*lowerleg*r.upperLegLength);
    j3=acos(d);

    if (j3<r.jointLimitLeg3N) {
        //OUTPUT(idText,"JointsFromLegPosition: Error "<<stringFromLeg(leg) <<" joint3");
        j3=r.jointLimitLeg3N;
        calcError=true;
    } else if (j3>r.jointLimitLeg3P) {
        //OUTPUT(idText,"JointsFromLegPosition: Error "<<stringFromLeg(leg) <<" joint3");
        j3=r.jointLimitLeg3P;
        calcError=true;
    }

    //mit j3 kann Teil der Transformation berechnet werden:
    // Trans(0,0,-u)*Rot(y,-q3)*Trans(0,0,-l)*(0,0,0,1)=(a,0,b,1)

    a=lowerleg*sin(j3);
    b=-lowerleg*d-r.upperLegLength;

    //Damit gilt (x,y,z,1) = Rot(y,-q1)*Rot(x,q2)*(a,0,b,1)
    //    y = -b*sin(j2)

    if (fabs(y)>fabs(b)) {
        //OUTPUT(idText,"JointsFromLegPosition: Error "<<stringFromLeg(leg) <<" unreachable");
        j2=pi_2;
        calcError=true;
    } else
        j2=asin(-y/b);

    if (j2<r.jointLimitLeg2N) {
        //OUTPUT(idText,"JointsFromLegPosition: Error "<<stringFromLeg(leg) <<" joint2");
        j2=r.jointLimitLeg2N;
        calcError=true;
    } else if (j2>r.jointLimitLeg2P) {
        //OUTPUT(idText,"JointsFromLegPosition: Error "<<stringFromLeg(leg) <<" joint2");
        j2=r.jointLimitLeg2P;
        calcError=true;
    }
    c2=cos(j2);


    //und x = a*cos(j1) - b*sin(j1)*cos(j2)

    //mit a=d*cos(beta), b*c2=d*sin(beta),
    //gilt x=d*cos(j2+beta)

    if (fabs(c2)<0.0001) {
        if (fabs(a)<0.0001)
            j1=0; //Sonderfall Bein ausgestreckt j1 irrelevant
        else
            j1=acos(x/a);
        if (z<0) j1=-j1;
    } else {
        beta=atan2(b*c2,a);
        d=(b*c2)/sin(beta);
        j1=acos(x/d);
        if (z<0) j1=-j1;
        j1-=beta;
    } 

    if (correctCalculation) j1 -= r.zeroShoulderArc; //this is because knee is not in the x-middle of a leg

    // reflect body tilt angle
    if (FORELEG(leg))
    {
        j1 += bodyTilt;
        if (correctCalculation) j3 += r.zeroFrontKneeArc; //this is because knee is not in the x-middle of a leg
    }
    else
    {
        j1 -= bodyTilt;
        if (correctCalculation) j3 += r.zeroHindKneeArc; //this is because knee is not in the x-middle of a leg
    }

    if (j1<(FORELEG(leg)?r.jointLimitLeg1FN:r.jointLimitLeg1HN)) {
        //OUTPUT(idText,"JointsFromLegPosition: Error "<<stringFromLeg(leg) <<" joint1");
        j1=FORELEG(leg)?r.jointLimitLeg1FN:r.jointLimitLeg1HN;
        calcError=true;
    } else if (j1>(FORELEG(leg)?r.jointLimitLeg1FP:r.jointLimitLeg1HP)) {
        //OUTPUT(idText,"JointsFromLegPosition: Error "<<stringFromLeg(leg) <<" joint1");
        j1=FORELEG(leg)?r.jointLimitLeg1FP:r.jointLimitLeg1HP;
        calcError=true;
    }

    /*
    //test results
    if (FORELEG(leg)) { j1 -= bodyTilt; } else { j1 += bodyTilt; }
    double tx,ty,tz;
    legPositionFromJoints(leg,j1,j2,j3,tx,ty,tz);
    if (!(FORELEG(leg))) { tx = -tx; }
    if (!(LEFTLEG(leg))) { ty = -ty; }
    //OUTPUT(idText,text,"x: "<<tx<<" y: "<<ty<<" z: "<<tz);
    if ((fabs(tx-x)>0.01 || fabs(ty-y)>0.01 || fabs(tz-z)>0.01) && !calcError)
    OUTPUT(idText,text,"JointsFromLegPosition incorrect for "<<stringFromLeg(leg)<<" x: "<<x<<" y: "<<y<<" z: "<<z);
    */
    return !calcError;
}

char *Kinematics::stringFromLeg(LegIndex leg) {
    switch(leg) {
    case fr:
        return "fore right leg";
    case fl:
        return "fore left leg";
    case hr:
        return "hind right leg";
    case hl:
        return "hind left leg";
    case basefront:
        return "front leg";
    case basehind:
        return "hind leg";
    default:
        return "wrong leg";
    }
}

void Kinematics::calcRelativeRobotVertices(RobotVertices& rob, const SensorData& sensorData)
{
    const RobotDimensions& r = getRobotConfiguration().getRobotDimensions();
    long specsKneeArc = long(r.specsKneeArc * 1e6);
    rob.shoulderPosition[fr] = Vector3<double>(-r.neckToLegsLengthX, -r.shoulderWidth / 2, -r.neckToLegsLengthZ);
    Vector3<double> lower(r.lowerForeLegLengthX, 0, -r.lowerForeLegLengthZ);
    lower *= (r.lowerForeLegLength - r.footRadius) / r.lowerForeLegLength;
    calcLegPosition(rob.kneePosition[fr], rob.footPosition[fr], 
        -sensorData.data[SensorData::legFR1],
        -sensorData.data[SensorData::legFR2], -sensorData.data[SensorData::legFR3] + specsKneeArc,
        rob.shoulderPosition[fr], Vector3<double>(r.upperLegLengthX, -r.upperLegLengthY, -r.upperLegLengthZ),
        lower);

    rob.shoulderPosition[fl] = Vector3<double>(-r.neckToLegsLengthX, r.shoulderWidth / 2, -r.neckToLegsLengthZ);
    calcLegPosition(rob.kneePosition[fl], rob.footPosition[fl], 
        -sensorData.data[SensorData::legFL1],
        sensorData.data[SensorData::legFL2], -sensorData.data[SensorData::legFL3] + specsKneeArc,
        rob.shoulderPosition[fl], Vector3<double>(r.upperLegLengthX, r.upperLegLengthY, -r.upperLegLengthZ),
        lower);

    lower = Vector3<double>(-r.lowerHindLegLengthX, 0, -r.lowerHindLegLengthZ);
    lower *= (r.lowerHindLegLength - r.footRadius) / r.lowerHindLegLength;
    rob.shoulderPosition[hr] = Vector3<double>(-r.neckToLegsLengthX - r.lengthBetweenLegs, -r.shoulderWidth / 2, -r.neckToLegsLengthZ);
    calcLegPosition(rob.kneePosition[hr], rob.footPosition[hr], 
        sensorData.data[SensorData::legHR1],
        -sensorData.data[SensorData::legHR2], sensorData.data[SensorData::legHR3] - specsKneeArc,
        rob.shoulderPosition[hr], Vector3<double>(-r.upperLegLengthX, -r.upperLegLengthY, -r.upperLegLengthZ),
        lower);

    rob.shoulderPosition[hl] = Vector3<double>(-r.neckToLegsLengthX - r.lengthBetweenLegs, r.shoulderWidth / 2, -r.neckToLegsLengthZ);
    calcLegPosition(rob.kneePosition[hl], rob.footPosition[hl], 
        sensorData.data[SensorData::legHL1], 
        sensorData.data[SensorData::legHL2], sensorData.data[SensorData::legHL3] - specsKneeArc,
        rob.shoulderPosition[hl],
        Vector3<double>(-r.upperLegLengthX, r.upperLegLengthY, -r.upperLegLengthZ),
        lower);

    for(int i = 0; i < 4; ++i)
    {
        rob.kneePosition[i].z -= r.kneeRadius;
        rob.footPosition[i].z -= r.footRadius;
    }
    rob.frameNumber = sensorData.frameNumber;
}

void Kinematics::calcNeckAndLegPositions(const SensorData& sensorData, RobotVertices& rob)
{
    const RobotDimensions& r = getRobotConfiguration().getRobotDimensions();

    calcRelativeRobotVertices(rob, sensorData);

    int i;
    for(i = 0; i < 4; ++i)
    {
        if(rob.footPosition[i].z - rob.shoulderPosition[i].z > -r.shoulderRadius)
            rob.footPosition[i] = rob.shoulderPosition[i] + Vector3<double>(0, 0, -r.shoulderRadius);
        if(rob.kneePosition[i].z - rob.shoulderPosition[i].z > -r.shoulderRadius)
            rob.kneePosition[i] = rob.shoulderPosition[i] + Vector3<double>(0, 0, -r.shoulderRadius);
    }

    getRobotTransformation(rob.bodyTilt, rob.bodyRoll, rob, Kinematics::dynamicMode);

    Pose3D pose;
    pose.rotateY(rob.bodyTilt);
    double minZ = 0;
    for(i = 0; i < 4; ++i)
    {
        rob.footPosition[i] = (Pose3D(pose).translate(rob.footPosition[i])).translation;
        rob.kneePosition[i] = (Pose3D(pose).translate(rob.kneePosition[i])).translation;
        if(minZ > rob.footPosition[i].z)
            minZ = rob.footPosition[i].z;
        if(minZ > rob.kneePosition[i].z)
            minZ = rob.kneePosition[i].z;
    }
    rob.neckHeight = -minZ;
}

void Kinematics::calcLegPosition(Vector3<double>& kneePosition, Vector3<double>& footPosition,
                                 long joint1, long joint2, long joint3,
                                 const Vector3<double> shoulder,
                                 const Vector3<double>& upper, const Vector3<double>& lower)
{
    Pose3D pose = Pose3D(shoulder).rotateY(joint1 * 1e-6).rotateX(joint2 * 1e-6).translate(upper);
    kneePosition = pose.translation;
    footPosition = pose.rotateY(joint3 * 1e-6).translate(lower).translation;
}

void Kinematics::calculateCameraMatrix(const SensorData& sensorData, const HeadState& headState, CameraMatrix& cameraMatrix)
{
    const RobotConfiguration& rc = getRobotConfiguration();
    const RobotDimensions& r = rc.getRobotDimensions();

    double tilt = fromMicroRad(sensorData.data[SensorData::headTilt]);
    double pan = fromMicroRad(sensorData.data[SensorData::headPan]);
    double roll = fromMicroRad(sensorData.data[SensorData::headRoll]);

    if(rc.getRobotDesign() == RobotDesign::ERS210)
        (Pose3D&)cameraMatrix =
        Pose3D(0, 0, headState.neckHeight).
        rotateX(rc.getRobotCalibration().bodyRollOffset).
        rotateY(headState.bodyTilt + rc.getRobotCalibration().bodyTiltOffset - tilt * rc.getRobotCalibration().tiltFactor).
        translate(0, 0, r.distanceNeckToPanCenter).
        rotateZ(pan * rc.getRobotCalibration().panFactor).
        rotateY(rc.getRobotCalibration().headTiltOffset).
        rotateX(roll + rc.getRobotCalibration().headRollOffset).
        translate(r.distancePanCenterToCameraX, 0, 0);
    else
        (Pose3D&)cameraMatrix =
        Pose3D(0, 0, headState.neckHeight).
        rotateX(rc.getRobotCalibration().bodyRollOffset).
        rotateY(headState.bodyTilt + rc.getRobotCalibration().bodyTiltOffset - tilt * rc.getRobotCalibration().tiltFactor).
        translate(0, 0, r.distanceNeckToPanCenter).
        rotateZ(pan * rc.getRobotCalibration().panFactor).
        rotateY(-roll * rc.getRobotCalibration().tilt2Factor + rc.getRobotCalibration().headTiltOffset). // tilt2 joint
        translate(r.distancePanCenterToCameraX, 0, -r.distancePanCenterToCameraZ).
        rotateX(rc.getRobotCalibration().headRollOffset);
}

void Kinematics::calcHeadHeight(double& headHeight, const double bodyTilt, const double neckHeight, const SensorData& sensorData)
{
    const RobotDimensions& rob = getRobotConfiguration().getRobotDimensions();

    if (bodyTilt - fromMicroRad(sensorData.data[SensorData::headTilt]) > pi/2)
    {
        headHeight = neckHeight - (cos(bodyTilt - fromMicroRad(sensorData.data[SensorData::headTilt])) * rob.distanceNeckToPanCenter);
    }else
    {
        headHeight = neckHeight + (cos(bodyTilt - fromMicroRad(sensorData.data[SensorData::headTilt])) * rob.distanceNeckToPanCenter);
    }
}

void Kinematics::calcNoseHeight(double& noseHeight, const double absHeadTilt, const double headHeight, const double bodyTilt, const double headRoll)
{
    // @todo isn't noseHeight == cameraMatrix.translation.z ?
    // Problem/difference is that this is able to calculate in advance - not measure
    const RobotDimensions& rob = getRobotConfiguration().getRobotDimensions();

    noseHeight = headHeight + (rob.distancePanCenterToCameraX * sin(absHeadTilt)) * sin(fabs(headRoll));
}


void Kinematics::calcAbsRoll(double& absRoll, const double bodyTilt, const SensorData& sensorData)
{
    absRoll = - bodyTilt + fromMicroRad(sensorData.data[SensorData::headTilt] + sensorData.data[SensorData::headRoll]);
}

double Kinematics::calcHighestPossibleTilt(const double absRoll, const double bodyTilt)
{
    const RobotDimensions& rob = getRobotConfiguration().getRobotDimensions();

    if (absRoll < rob.jointLimitHeadRollN - bodyTilt)
    {
        return absRoll + bodyTilt - rob.jointLimitHeadRollN;
    }
    else
    {
        return rob.jointLimitHeadTiltP;
    }
}

double Kinematics::calcLowestPossibleTilt(const double absRoll, const double bodyTilt)
{
    const RobotDimensions& rob = getRobotConfiguration().getRobotDimensions();

    if (absRoll > rob.jointLimitHeadRollP - bodyTilt + rob.jointLimitHeadTiltN)
    {
        return absRoll + bodyTilt - rob.jointLimitHeadRollP;
    }
    else
    {
        return rob.jointLimitHeadTiltN;
    }
}


void Kinematics::calcAbsRollJoints(double& neckTilt, double& absRoll, const HeadState& headState)
{
    const RobotDimensions& rob = getRobotConfiguration().getRobotDimensions();
    const double lowestTilt = Kinematics::calcLowestPossibleTilt(absRoll, headState.bodyTilt);
    const double highestTilt = Kinematics::calcHighestPossibleTilt(absRoll, headState.bodyTilt);

    if (neckTilt < lowestTilt)
    {
        neckTilt = lowestTilt;
        absRoll = rob.jointLimitHeadRollP;
    }
    else if (neckTilt > highestTilt)
    {
        neckTilt = highestTilt;
        absRoll = rob.jointLimitHeadRollN;
    }
    else
    {
        neckTilt = neckTilt;
        absRoll += headState.bodyTilt - neckTilt;
    }
}


/*
* Change log :
* 
* $Log: Kinematics.cpp,v $
* Revision 1.24  2004/03/22 15:56:13  hodie
* Fixed calcAbsRollJoints
*
* Revision 1.23  2004/03/20 09:55:27  roefer
* Preparation for improved odometry
*
* Revision 1.22  2004/03/18 18:07:21  hodie
* Some more absolute HeadControlModes
* Absolute HeadPathPlanner
*
* Revision 1.21  2004/03/17 16:21:12  hodie
* Some more absolute HeadControlModes
* Absolute HeadPathPlanner
*
* Revision 1.20  2004/03/15 15:09:46  hodie
* new absRoll and absStraightRoll
* some new functions and tools
* modified lookArounds
*
* Revision 1.19  2004/03/11 17:02:27  risler
* different limits for front and hind leg joint 1 for ERS-7
*
* Revision 1.18  2004/03/08 02:32:18  roefer
* Calibration parameters changed
*
* Revision 1.17  2004/03/04 23:01:56  roefer
* Warnings and errors removed
*
* Revision 1.16  2004/03/04 16:33:59  hodie
* without the unstable functions..sorry for that
*
* Revision 1.15  2004/03/04 16:32:22  hodie
* fixed calcNoseHeight for rotated head (pan)
*
* Revision 1.14  2004/03/04 15:56:20  hodie
* added calc -AbsTilt, -HeadHeight, and -NoseHeight
*
* Revision 1.13  2004/02/29 14:56:03  roefer
* Additional calibration parameters
*
* Revision 1.12  2004/02/26 23:03:30  roefer
* CameraMatrix improved
*
* Revision 1.11  2004/02/24 19:01:06  roefer
* Additional calibration parameters added
*
* Revision 1.10  2004/02/24 00:01:21  roefer
* CameraMatrix improved
*
* Revision 1.9  2004/02/04 13:41:33  roefer
* Some place holders for BB2004 modules added
*
* Revision 1.8  2004/01/28 21:55:50  roefer
* RobotDimensions revised
*
* Revision 1.7  2004/01/17 19:19:18  roefer
* Simulator calculates robot pose based on class Kinematics now
*
* Revision 1.6  2004/01/01 10:58:51  roefer
* RobotDimensions are in a class now
*
* Revision 1.5  2003/12/19 15:56:05  dueffert
* now old and new (more correct) kinematics is supported. new one is default
*
* Revision 1.4  2003/12/09 14:21:47  dueffert
* flyMode added to visualize theoretical footPositions in RadarViewer better
*
* Revision 1.3  2003/12/05 16:32:09  dueffert
* old version restored, calculation is wrong but walking looks better
*
* Revision 1.2  2003/12/02 18:13:32  dueffert
* correction values named
*
* Revision 1.1  2003/10/07 10:13:21  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.3  2003/09/30 11:04:10  dueffert
* no inverse kinematics correction before code release
*
* Revision 1.1  2003/09/26 11:40:40  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.6  2003/09/23 12:18:07  risler
* slight optimizations
*
* Revision 1.5  2003/09/18 10:32:11  juengel
* error fixed
*
* Revision 1.4  2003/09/16 14:57:41  dueffert
* tilt calculation in forward kinematics added
*
* Revision 1.3  2003/09/16 11:08:52  dueffert
* doxygen comments improved
*
* Revision 1.2  2003/09/15 20:38:08  dueffert
* forward kinematics improved
*
* Revision 1.1.1.1  2003/07/02 09:40:28  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.9  2003/03/03 17:34:57  risler
* legPositionFromJoints adjusted to changed leg coordinate system
*
* Revision 1.8  2003/01/18 17:45:45  risler
* changed walking engine parameter coordinate system
* jointsFromLegPosition handles positive z values correctly
*
* Revision 1.7  2003/01/17 16:23:06  dueffert
* started to add ground based calculations
*
* Revision 1.6  2002/11/28 14:05:15  dueffert
* doxygen docu improved
*
* Revision 1.5  2002/11/25 12:32:56  dueffert
* bodyTilt should be used to calculate leg positions from joints and vice versa
*
* Revision 1.4  2002/11/19 15:43:03  dueffert
* doxygen comments corrected
*
* Revision 1.3  2002/09/23 13:54:48  risler
* abs replaced by fabs/labs
*
* Revision 1.2  2002/09/22 18:40:53  risler
* added new math functions, removed GTMath library
*
* Revision 1.1  2002/09/10 15:53:58  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.2  2002/08/30 14:36:22  dueffert
* removed unused includes
*
* Revision 1.1.1.1  2002/05/10 12:40:32  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.2  2002/04/06 12:44:57  risler
* no message
*
* Revision 1.1  2002/04/04 11:02:17  risler
* added class Kinematics
*
*
*/
