/**
 * @file RobotDimensions.h
 *
 * Description of the Dimensions of the Robot
 *
 * @author Matthias Jngel, Max Risler
 */

#ifndef __RobotDimensions_H__
#define __RobotDimensions_H__

#include "Tools/Math/Vector2.h"

class RobotDimensions
{
public:
  const double distancePanCenterToCameraX;//!<x-distance camera to center of head (specs)
  const double distancePanCenterToCameraZ;//!<z-distance camera to center of head (specs)
  const double distanceCameraToPSDSensor; //!<z-distanse camera to PSDsensor (estimated?)
  const double distanceNeckToPanCenter;   //!<distance center of head to neck (specs)

  const double shoulderWidth;             //!<width between shoulder joints (specs)
  const double bodyWidth;                 //!<width between knee joints (specs)
  const double overallBodyWidth;          //!<overall width of the body including legs (specs)

  const double upperLegLengthX;           //!<length upper leg joint to joint in x-direction (specs)
  const double upperLegLengthY;           //!<length upper leg joint to joint in y-direction (calculated)
  const double upperLegLengthZ;           //!<length upper leg joint to joint in z-direction (specs)
  const double upperLegLength;            //!<length upper leg joint to joint (calculated)
  const double lowerForeLegLengthX;       //!<length lower front leg joint to footbase in x-direction (specs)
  const double lowerForeLegLengthZ;       //!<length lower front leg joint to footbase in z-direction (specs)
  const double lowerForeLegLength;        //!<length lower front leg joint to footbase (calculated)
  const double lowerHindLegLengthX;       //!<length lower rear leg joint to footbase in x-direction (specs)
  const double lowerHindLegLengthZ;       //!<length lower rear leg joint to footbase in z-direction (specs)
  const double lowerHindLegLength;        //!<length lower rear leg joint to footbase (calculated)
  const double lengthBetweenLegs;         //!<length from front to rear legs (specs)
  const double neckToLegsLengthZ;         //!<height from neck to front legs (specs)
  const double neckToLegsLengthX;         //!<length from front legs to neck projected on x-axis (specs) 
  const double lengthNeckToBodyCenter;    //!<length from center of Robot to neck projected on x-axis (calculated)

  const double zeroShoulderArc;           //!<arc resulting from upperLegLengthX and -Z (calculated)
  const double specsKneeArc;              //!<knee arc lower leg lengths are based on (specs)
  const double zeroFrontKneeArc;          //!<arc resulting from lowerForeLegLengthX  and -Z (calculated)
  const double zeroHindKneeArc;           //!<arc resulting from lowerHindLegLengthX  and -Z (calculated)
  const double zeroBodyTilt;              //!<body tilt when all joints are 0 (calculated)

  const double bodyLength;                //!<length of robot body (estimated)
  const double lowerBodyWidth;            //!<width of body core at bottom (estimated), 
  const double shoulderRadius;            //!< thickness of the body around the shoulder (estimated)
  const double kneeRadius;                //!< thickness of the leg at the knee (estimated)
  const double footRadius;                //!< thickness of the leg at the foot (estimated)

  //!@name joint limits in rad (estimated)
  //!@{
  const double jointLimitLeg1FN;
  const double jointLimitLeg1FP;
  const double jointLimitLeg1HN;
  const double jointLimitLeg1HP;
  const double jointLimitLeg2N;
  const double jointLimitLeg2P;
  const double jointLimitLeg3N;
  const double jointLimitLeg3P;

  const double jointLimitHeadTiltN;
  const double jointLimitHeadTiltP;
  const double jointLimitHeadPanN;
  const double jointLimitHeadPanP;
  const double jointLimitHeadRollN;
  const double jointLimitHeadRollP;

  const double jointLimitTailTiltN;
  const double jointLimitTailTiltP;
  const double jointLimitTailPanN;
  const double jointLimitTailPanP;
  //!@}

  const double motionCycleTime;    //! Length of one motion cycle in seconds.

  RobotDimensions(
    const double distancePanCenterToCameraX,
    const double distancePanCenterToCameraZ,
	const double distanceCameraToPSDSensor,
    const double distanceNeckToPanCenter,
    const double shoulderWidth,
    const double bodyWidth,
    const double overallBodyWidth,
    const double upperLegLengthX,
    const double upperLegLengthZ,
    const double lowerForeLegLengthX,
    const double lowerForeLegLengthZ,
    const double lowerHindLegLengthX,
    const double lowerHindLegLengthZ,
    const double lengthBetweenLegs,
    const double neckToLegsLengthX,
    const double neckToLegsLengthZ,
    const double specsKneeArc,
    const double bodyLength,
    const double lowerBodyWidth,
    const double shoulderRadius,
    const double kneeRadius,
    const double footRadius,
    const double jointLimitLeg1FN,
    const double jointLimitLeg1FP,
    const double jointLimitLeg1HN,
    const double jointLimitLeg1HP,
    const double jointLimitLeg2N,
    const double jointLimitLeg2P,
    const double jointLimitLeg3N,
    const double jointLimitLeg3P,
    const double jointLimitHeadTiltN,
    const double jointLimitHeadTiltP,
    const double jointLimitHeadPanN,
    const double jointLimitHeadPanP,
    const double jointLimitHeadRollN,
    const double jointLimitHeadRollP,
    const double jointLimitTailTiltN,
    const double jointLimitTailTiltP,
    const double jointLimitTailPanN,
    const double jointLimitTailPanP,
    const double motionCycleTime
  )
  : distancePanCenterToCameraX(distancePanCenterToCameraX),
    distancePanCenterToCameraZ(distancePanCenterToCameraZ),
	distanceCameraToPSDSensor(distanceCameraToPSDSensor),
    distanceNeckToPanCenter(distanceNeckToPanCenter),
    shoulderWidth(shoulderWidth),
    bodyWidth(bodyWidth),
    overallBodyWidth(overallBodyWidth),
    upperLegLengthX(upperLegLengthX),
    upperLegLengthY((bodyWidth - shoulderWidth) / 2),
    upperLegLengthZ(upperLegLengthZ),
    upperLegLength(Vector2<double>(upperLegLengthX, upperLegLengthZ).abs()),
    lowerForeLegLengthX(lowerForeLegLengthX),
    lowerForeLegLengthZ(lowerForeLegLengthZ),
    lowerForeLegLength(Vector2<double>(lowerForeLegLengthX, lowerForeLegLengthZ).abs()),
    lowerHindLegLengthX(lowerHindLegLengthX),
    lowerHindLegLengthZ(lowerHindLegLengthZ),
    lowerHindLegLength(Vector2<double>(lowerHindLegLengthX, lowerHindLegLengthZ).abs()),
    lengthBetweenLegs(lengthBetweenLegs),
    neckToLegsLengthX(neckToLegsLengthX),
    neckToLegsLengthZ(neckToLegsLengthZ),
    lengthNeckToBodyCenter(neckToLegsLengthX + lengthBetweenLegs / 2),
    zeroShoulderArc(atan2(upperLegLengthX, upperLegLengthZ)),
    specsKneeArc(specsKneeArc),
    zeroFrontKneeArc(zeroShoulderArc + specsKneeArc - atan2(lowerForeLegLengthX, lowerForeLegLengthZ)),
    zeroHindKneeArc(zeroShoulderArc + specsKneeArc - atan2(lowerHindLegLengthX, lowerHindLegLengthZ)),
    zeroBodyTilt(atan2(lowerHindLegLength * cos(zeroFrontKneeArc - zeroShoulderArc) - 
                       lowerForeLegLength * cos(zeroHindKneeArc - zeroShoulderArc), lengthBetweenLegs)),
    bodyLength(bodyLength),
    lowerBodyWidth(lowerBodyWidth),
    shoulderRadius(shoulderRadius),
    kneeRadius(kneeRadius),
    footRadius(footRadius),
    jointLimitLeg1FN(jointLimitLeg1FN),
    jointLimitLeg1FP(jointLimitLeg1FP),
    jointLimitLeg1HN(jointLimitLeg1HN),
    jointLimitLeg1HP(jointLimitLeg1HP),
    jointLimitLeg2N(jointLimitLeg2N),
    jointLimitLeg2P(jointLimitLeg2P),
    jointLimitLeg3N(jointLimitLeg3N),
    jointLimitLeg3P(jointLimitLeg3P),
    jointLimitHeadTiltN(jointLimitHeadTiltN),
    jointLimitHeadTiltP(jointLimitHeadTiltP),
    jointLimitHeadPanN(jointLimitHeadPanN),
    jointLimitHeadPanP(jointLimitHeadPanP),
    jointLimitHeadRollN(jointLimitHeadRollN),
    jointLimitHeadRollP(jointLimitHeadRollP),
    jointLimitTailTiltN(jointLimitTailTiltN),
    jointLimitTailTiltP(jointLimitTailTiltP),
    jointLimitTailPanN(jointLimitTailPanN),
    jointLimitTailPanP(jointLimitTailPanP),
    motionCycleTime(motionCycleTime)
  {}
};

class RobotDimensionsERS210 : public RobotDimensions
{
public:
  RobotDimensionsERS210()
    : RobotDimensions(
        66.6,          // x-distance camera to center of head (specs)
        0.0,           // z-distance camera to center of head (specs)
        10.0,          // z-distanse camera to PSDsensor (?)
        48.0,          // distance center of head to neck (specs)

        118.4,         // width between shoulder joints (specs)
        119.4,         // width between knee joints (specs)
        152.0,         // overall width of the body including legs (specs)
        12.8,          // length upper leg joint to joint in x-direction (specs)
        64.0,          // length upper leg joint to joint in z-direction (specs)
        22.4,          // length lower front leg joint to footbase in x-direction (specs)
        64.9,          // length lower front leg joint to footbase in z-direction (specs)
        16.6,          // length lower rear leg joint to footbase in x-direction (specs)
        74.9,          // length lower rear leg joint to footbase in z-direction (specs)
        119.0,         // length from front to rear legs (specs)
        15.5,          // length from front legs to neck projected on x-axis (specs) 
        50.0,          // height from neck to front legs (specs)
        0.611,         // knee arc lower leg lengths are based on (35, specs)
        200.0,         // length of robot body (estimated)
        61.0,          // width of body core at bottom (used to avoid ball inside the robot body, estimated)
        58.0,          // thickness of the body around the shoulder (estimated)
        25.0,          // thickness of the leg at the knee (estimated)
        10.0,          // thickness of the leg at the foot (estimated)

        -2.04, 2.04,   // front leg joint 1 limits
        -2.04, 2.04,   // rear leg joint 1 limits
        -0.2, 1.571,   // leg joint 2 limits
        -0.46, 2.56,   // leg joint 3 limits
        -1.571, 0.75,  // head tilt joint limits
        -1.571, 1.571, // head pan joint limits
        -0.46, 0.46,   // head roll joint limits
        -0.38, 0.38,   // tail tilt joint limits
        -0.38, 0.38,   // tail pan joint limits
        0.008)         // motion cycle
  {}
};

class RobotDimensionsERS7 : public RobotDimensions
{
public:
  RobotDimensionsERS7()
    : RobotDimensions(
        81.06,         // x-distance camera to center of head (specs)
        14.6,          // z-distance camera to center of head (specs)
		20.0,          // z-distanse camera to PSDsensor (?)
        80.0,          // distance center of head to neck (specs)

        125.0,         // width between shoulder joints (specs)
        134.4,         // width between knee joints (specs)
        180.15,        // overall width of the body including legs (specs)
        9.0,           // length upper leg joint to joint in x-direction (specs)
        69.5,          // length upper leg joint to joint in z-direction (specs)
        28.3,          // length lower front leg joint to footbase in x-direction (specs)
        71.5,          // length lower front leg joint to footbase in z-direction (specs)
        21.3,          // length lower rear leg joint to footbase in x-direction (specs)
        76.5,          // length lower rear leg joint to footbase in z-direction (specs)
        130.0,         // length from front to rear legs (specs)
        2.5,           // length from front legs to neck projected on x-axis (specs)
        19.5,          // height from neck to front legs (specs)
        0.524,         // knee arc lower leg lengths are based on (30, specs)
        230.0,         // length of robot body (estimated)
        80.0,          // width of body core at bottom (used to avoid ball inside the robot body, estimated)
        52.0,          // thickness of the body around the shoulder (estimated)
        13.0,          // thickness of the leg at the knee (estimated)
        22.5,          // thickness of the leg at the foot (seems to be suitable to express the curve of the lower leg)

        -2.0944, 2.3562,  // front leg joint 1 limits       -120, 135
        -2.3562, 2.0944,  // rear leg joint 1 limits       -135, 120
        -0.2618, 1.6232,  // leg joint 2 limits		  -15,  93 (specs)
        -0.5236, 2.2166,  // leg joint 3 limits       -30,  127 (specs)
        -1.3963, 0.0524,  // head tilt joint limits   -80,  3 (specs)
        -1.6232, 1.6232,  // head pan joint limits    -93,  93 (specs)
        -0.3491, 0.8727,  // head tilt2 joint limits  -20,  50 (specs)
        +0.0000, 1.1345,  // tail tilt joint limits     0,  65 (specs)
        -1.0472, 1.0472,  // tail pan joint limits    -60,  60 (specs)
        0.008)            // motion cycle
  {}
}; 
#endif

/*
 * Change log :
 * 
 * $Log: RobotDimensions.h,v $
 * Revision 1.21  2004/03/11 17:02:27  risler
 * different limits for front and hind leg joint 1 for ERS-7
 *
 * Revision 1.20  2004/03/10 08:02:28  roefer
 * Limits of ERS-7 head joints corrected
 *
 * Revision 1.19  2004/02/26 23:03:31  roefer
 * CameraMatrix improved
 *
 * Revision 1.18  2004/02/24 19:01:06  roefer
 * Additional calibration parameters added
 *
 * Revision 1.17  2004/02/24 00:01:21  roefer
 * CameraMatrix improved
 *
 * Revision 1.16  2004/02/06 20:25:36  roefer
 * ERS-7 knee radius changed
 *
 * Revision 1.15  2004/01/28 21:55:50  roefer
 * RobotDimensions revised
 *
 * Revision 1.14  2004/01/23 00:13:24  roefer
 * ERS-7 simulation, first draft
 *
 * Revision 1.13  2004/01/14 19:18:01  mellmann
 * Added distanceCameraToPSDSensor.
 *
 * Revision 1.12  2004/01/08 18:19:28  mkunz
 * distancePanCenterToCamera -> distancePanCenterToCameraX
 * new: distancePanCenterToCameraZ
 *
 * Revision 1.11  2004/01/06 14:14:52  mellmann
 * added ERS7 dimensions
 *
 * Revision 1.10  2004/01/05 17:54:05  juengel
 * Added joint limits for head and tail for ERS7.
 *
 * Revision 1.9  2004/01/05 10:02:15  juengel
 * Prepared RobotDimensionsERS7.
 *
 * Revision 1.8  2004/01/03 16:36:13  roefer
 * motionCycleTime 8 ms -> 0.008 s
 *
 * Revision 1.7  2004/01/01 10:58:51  roefer
 * RobotDimensions are in a class now
 *
 * Revision 1.6  2003/12/31 23:51:40  roefer
 * Removed unused constants
 *
 * Revision 1.5  2003/12/30 20:12:04  roefer
 * Image size is now 208 x 160. Smaller images are placed in the upper left corner
 *
 * Revision 1.4  2003/12/15 11:49:09  juengel
 * Introduced CameraInfo
 *
 * Revision 1.3  2003/12/02 18:10:11  dueffert
 * missing correction values added
 *
 * Revision 1.2  2003/10/23 15:41:40  roefer
 * Oracled obstacle model added
 *
 * Revision 1.1  2003/10/07 10:13:21  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1  2003/09/26 11:40:40  juengel
 * - sorted tools
 * - clean-up in DataTypes
 *
 * Revision 1.3  2003/09/12 12:11:57  dueffert
 * knee position added
 *
 * Revision 1.2  2003/07/03 10:26:45  roefer
 * lower body width
 *
 * Revision 1.1.1.1  2003/07/02 09:40:28  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.8  2003/06/17 13:47:02  dueffert
 * constants added
 *
 * Revision 1.7  2003/04/15 15:52:07  risler
 * DDD GO 2003 code integrated
 *
 * Revision 1.7  2003/03/26 17:45:27  max
 * bodyLength added
 *
 * Revision 1.6  2003/03/15 13:30:26  juengel
 * Added joint limits for tail.
 *
 * Revision 1.5  2003/02/21 16:34:16  dueffert
 * common pi in own code, warnings removed, platform compatibility restored
 *
 * Revision 1.4  2002/11/19 15:43:03  dueffert
 * doxygen comments corrected
 *
 * Revision 1.3  2002/10/10 13:31:05  jhoffman
 * added camera parameters
 *
 * Revision 1.2  2002/09/22 18:40:53  risler
 * added new math functions, removed GTMath library
 *
 * Revision 1.1  2002/09/10 15:53:58  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.2  2002/06/04 11:01:41  juengel
 * jointLimitHeadTiltP modified.
 *
 * Revision 1.1.1.1  2002/05/10 12:40:32  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.10  2002/04/05 16:46:20  kosen
 * BremenGOWalkingEngine added
 *
 * Revision 1.9  2002/04/03 13:25:41  risler
 * changed doubles to Value/Angle, tan to tan
 *
 * Revision 1.8  2002/04/02 14:01:39  dueffert
 * minor odometry enhancements
 *
 * Revision 1.7  2002/04/02 13:10:22  dueffert
 * big change: odometryData and cameraMatrix in image now, old logfiles may be obsolete
 *
 * Revision 1.6  2002/02/06 10:30:11  AndySHB
 * MonteCarloLocalization First Draft
 *
 * Revision 1.5  2002/01/30 01:08:56  dueffert
 * fixed endless-compile-bug caused by up/downcase difference MathLib vs. Mathlib, makefile should be caseUNsensitive (in such cases) now
 *
 * Revision 1.4  2002/01/19 21:36:24  risler
 * added HeadMotionTester, HeadControlSelector
 *
 * Revision 1.3  2001/12/13 19:13:58  risler
 * added odometry updates in GT2001MotionNetSpecialActions
 *
 * Revision 1.2  2001/12/10 17:47:10  risler
 * change log added
 *
 */
