/**
* @file WalkAccelerationRestrictor.cpp
*
* Implementation of class WalkAccelerationRestrictor
*
* @author Martin Ltzsch
*/

#include "WalkAccelerationRestrictor.h"
#include "Platform/SystemCall.h"

WalkAccelerationRestrictor::WalkAccelerationRestrictor(MotionRequest& motionRequest)
: mr(motionRequest)
{
}

void WalkAccelerationRestrictor::saveLastWalkParameters()
{
  if (mr.motionType != MotionRequest::walk)
  {
    lastTranslationX = 0;
    lastTranslationY = 0;
    lastRotation = 0;
  }
  else
  {
    lastTranslationX = mr.walkParams.translation.x;
    lastTranslationY = mr.walkParams.translation.y;
    lastRotation = mr.walkParams.rotation;
  }
}

void WalkAccelerationRestrictor::restrictAccelerations(double maxTranslationXAcceleration,
                                                       double maxTranslationYAcceleration,
                                                       double maxRotationAcceleration)
{
  unsigned long currentTime = SystemCall::getCurrentSystemTime();
  if (timeOfLastExecution == 0 || timeOfLastExecution > currentTime + 250)
    timeOfLastExecution = currentTime - 50;
  unsigned long timeSinceLastExecution = currentTime - timeOfLastExecution;
  if (timeSinceLastExecution > 1000) timeSinceLastExecution = 1000;

  double maximumTranslationXChange = 
    maxTranslationXAcceleration * timeSinceLastExecution / 1000;

  double maximumTranslationYChange = 
    maxTranslationYAcceleration * timeSinceLastExecution / 1000;

  double vMaximumRotationChange =
    maxRotationAcceleration * timeSinceLastExecution / 1000;

  double maximumRotationChange;
  maximumRotationChange = fromDegrees(vMaximumRotationChange);

  if (mr.walkParams.translation.x - lastTranslationX > maximumTranslationXChange)
    mr.walkParams.translation.x = lastTranslationX + maximumTranslationXChange;

  if (mr.walkParams.translation.x - lastTranslationX < -1 * maximumTranslationXChange)
    mr.walkParams.translation.x = lastTranslationX - maximumTranslationXChange;

  if (mr.walkParams.translation.y - lastTranslationY > maximumTranslationYChange)
    mr.walkParams.translation.y = lastTranslationY + maximumTranslationYChange;

  if (mr.walkParams.translation.y - lastTranslationY < -1 * maximumTranslationYChange)
    mr.walkParams.translation.y = lastTranslationY - maximumTranslationYChange;

  if (mr.walkParams.rotation - lastRotation > maximumRotationChange)
    mr.walkParams.rotation = lastRotation + maximumRotationChange;

  if (mr.walkParams.rotation - lastRotation < -1 * maximumRotationChange)
    mr.walkParams.rotation = lastRotation - maximumRotationChange;

  timeOfLastExecution = SystemCall::getCurrentSystemTime();
}


/*
* Change Log:
* 
* $Log: WalkAccelerationRestrictor.cpp,v $
* Revision 1.1  2003/10/07 10:13:21  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1  2003/09/26 11:40:40  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.1.1.1  2003/07/02 09:40:28  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.1  2003/06/04 00:43:45  loetzsch
* put the GT2003WalkAccelerationRestrictor into two extra files in /Src/Tools
*
*/

