/**
* @file ContinuousBasicBehavior.h
*
* Definiton of a ContinuousbasicBehavior that is used by Xabsl2 based BehaviorControls.
* 
* @author Martin Kallnik
*/

#ifndef __ContinuousBasicBehavior_h_
#define __ContinuousBasicBehavior_h_


#include "ContinuousRule.h"

#include "Modules/BehaviorControl/BehaviorControl.h"
#include "Tools/Xabsl2/Xabsl2Engine/Xabsl2BasicBehavior.h"
#include "Tools/Actorics/WalkAccelerationRestrictor.h"

/**
* @class ContinuousBasicBehavior
* The base class for continuousBasicBehaviors that are used by the Xabsl2Engine
* @author Martin Kallnik
*/
class ContinuousBasicBehavior
  :public Xabsl2BasicBehavior,
   public BehaviorControlInterfaces{

public:


 /** constructor
  * @param interfaces The BehaviorInterfaces worldmodel and output)
	* @param errorHandler deals with outputmessages 
  * @param basicBehaviorName The name of this basicBehavior
  */
  ContinuousBasicBehavior(const BehaviorControlInterfaces& interfaces,
                          Xabsl2ErrorHandler& errorHandler,
                          char* basicBehaviorName
                         );

  /** 
   * destructor.
   */
  ~ContinuousBasicBehavior();

  /**
   * set the walk type
   */
  void setWalkType(MotionRequest::WalkType walkType);

  /**
   * set maximum speed
   */
  void setMaxSpeed(double maxSpeed);
  
  /**
   * set maximum rotation speed
   */
  void setMaxRotationSpeed(double maxRotationSpeed);

  /**
   * set accelerations
   */
  void setAccelerations(double maxAccX, double maxAccY, double maxAccRot);

  /**
   * add a continuous rule
   */
  void addRule(ContinuousRule *pRule);

  /**
   * enables/disabled walking backwards
   */
  void enableBackwards(bool backwards);

  /** 
   * executes the basic behavior
   */
  void execute();

  /**
   * EvaluationMode 
   * discrete: Use only the RobotPose for the evaluation
   * sample:   Use all the selfLocatorSamples for the evaluation and deliver a merged result
   */
  enum EvaluationMode { discrete, sample, numOfModes } evaluationMode;

  /**
   * set the evaluation mode
   */
  void setEvaluationMode(EvaluationMode evaluationMode);

  /**
   * set smoothing
   */
  void setSmoothing(int smooth);

private:

  /** smmothing (1= no smmothing, >1 number of evaluated steps)*/
  int smooth;

  /** the rules which are used by this basicBehavior*/
  Xabsl2Array<ContinuousRule*> pRules;

  /** the additional robot relative rules which are used by this basicBehavior*/
  Xabsl2Array<ContinuousRule*> pRulesRelative;

	/** the used walkType*/
	MotionRequest::WalkType walkType;

  /** maximum speed */
  double maxSpeed;

  /** maximum rotation speed */
  double maxRotationSpeed;

  /** whether walking backwards is allowed or not */
  bool backwards;

  /** maximum acceleration in x direction */
  double maxAccX;

  /** maximum acceleration in y direction */
  double maxAccY;

  /** maximum rotation acceleration */
  double maxAccRot;

  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;

  /** draw the DebugDrawing CBBVector*/
  void drawVectorField();

  /** draw the DebugDrawing CBBWay*/
  void drawWay();

  /** 
  * executes the basic behavior
  * @param robotPose the position the behavior is executed
  * @param motionRequest reference to resulting motionRequest
  * @param suppressPrintRules true if PrintRules-debug output should be suppressed
  */
  void executeAt(const RobotPose& robotPose, MotionRequest& motionRequest, const bool suppressPrintRules);

};





#endif //__ContinuousBasicBehavior_h_


/*
* Change log:
*
* $Log: ContinuousBasicBehavior.h,v $
* Revision 1.2  2004/03/08 01:07:18  roefer
* Interfaces should be const
*
* Revision 1.1  2003/10/07 10:13:22  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/09/26 11:40:40  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.1.1.1  2003/07/02 09:40:28  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.11  2003/06/05 09:22:25  kallnik
* ContinuousBasicBehavior:
* Smoothing added
*
* Revision 1.10  2003/06/04 11:16:47  risler
* added acceleration restriction, enableBackwards
*
* Revision 1.9  2003/06/03 10:51:24  risler
* ContinuousRules can be in relative coordinates
* no rotation rule active => no rotation
*
* Revision 1.8  2003/05/19 10:29:46  risler
* added max speeds to continuousbasicbehaviors
*
* Revision 1.7  2003/05/13 14:45:16  brunn
* added evaluationMode
* default evaluationMode is discrete which uses only the robotPose
* added sample evaluation Mode which uses all selflocatorSamples
*
* Revision 1.6  2003/05/09 15:25:08  risler
* ContinuousBasicBehavior uses Xabsl2Array,
* ContinuousRules inherit from Xabsl2NamedItem
*
* Revision 1.5  2003/05/09 12:59:36  loetzsch
* removed useless #include statements
*
* Revision 1.4  2003/05/07 11:05:55  risler
* ContinuousBasicBehaviors:
* removed classes Attracting/RepellingFieldRule, replaced with getAttracting/RepellingForce functions
* some clean up
*
* Revision 1.3  2003/05/07 10:46:38  kallnik
* warnings removed
*
* Revision 1.2  2003/05/07 10:02:09  kallnik
* ContinuousBasicBehavior:
* -DebugDrawings fixed
* -copyfiles: do no copy Config/ContBeh(files deleted)
*
* Revision 1.1  2003/05/06 17:03:43  kallnik
* ContinuousBasicBehaviors code review
*
* Revision 1.11  2003/04/28 11:01:02  kallnik
* new ContinuousBasicBehavior version (NORules is taken from number of lines in Rules-section
* and some bugfixes
*
* Revision 1.10  2003/04/25 12:57:14  kallnik
* new version of ContinuousBasicBehavior released
*
* Revision 1.9  2003/04/15 15:52:11  risler
* DDD GO 2003 code integrated
*
* Revision 1.12  2003/04/09 16:44:45  kallnik
* DebugDrawing moved from ContinuousBasicBehaviorTester to ContinuousBasicBehavior
*
* Revision 1.11  2003/04/09 15:51:57  kallnik
* ContinuousBasicBehavior Debug Drawing   for   ContinuousBasicBehaviorTester
*
* Revision 1.10  2003/04/02 19:03:32  max
* bugfix: wrong config files wont crash anymore
*
* Revision 1.9  2003/04/02 12:11:14  max
* some rules added, modified
*
* Revision 1.8  2003/03/12 22:58:41  roefer
* Destructor added, memory leak closed
*
* Revision 1.7  2003/03/04 14:30:06  kallnik
* continuousbasicbehvior: bug fixes
*
* Revision 1.6  2003/02/25 11:43:45  dueffert
* line feed and doxygen bugs fixed
*
* Revision 1.5  2003/02/25 10:11:28  dueffert
* doxygen bugs fixed
*
* Revision 1.4  2003/02/24 14:48:17  kallnik
* some bugs in continuousbasicbehaviors fixed
*
* Revision 1.3  2003/02/20 15:55:17  dueffert
* bugs fixed
*
* Revision 1.2  2003/02/20 12:40:24  kallnik
* added continuousBasicBehaviorTester in behaviorSelector
*
* Revision 1.1  2003/02/20 12:02:29  kallnik
* ContinuousBasicBehavior added
*
*
*/
