/** 
* @file DebugKeyTable.cpp
* Implementation of class DebugKey and DebugKeyTable
*
* @author <a href=mailto:brunn@sim.informatik.tu-darmstadt.de>Ronnie Brunn</a>
* @author <a href=mailto:martin@martin-loetzsch.de>Martin Ltzsch</a>
* @author <a href=mailto:risler@sim.informatik.tu-darmstadt.de>Max Risler</a>
*/

#include "Platform/SystemCall.h"
#include "DebugKeyTable.h"


DebugKey::DebugKey(modes mode, unsigned int n) 
: mode(mode), n(n), last(0), active(false) 
{}

void DebugKey::set(modes mode, unsigned int n)
{
  this->mode = mode;
  this->n = n;
  last = 0;
  active = false;
}

DebugKeyTable::DebugKeyTable()
{
  reset();
}

void DebugKeyTable::set(debugKeyID id, DebugKey::modes mode, int n) 
{
  debugKeys[id].set(mode, n);
}

void DebugKeyTable::setMode(debugKeyID id, DebugKey::modes mode)
{
  debugKeys[id].set(mode,debugKeys[id].n);
}

void DebugKeyTable::setN(debugKeyID id, unsigned int n)
{
  debugKeys[id].set(debugKeys[id].mode,n);
}

DebugKey::modes DebugKeyTable::getMode(debugKeyID id) const
{ 
  return debugKeys[id].mode;
}

unsigned int DebugKeyTable::getN(debugKeyID id) const
{ 
  return debugKeys[id].n;
}

bool DebugKeyTable::isActive(debugKeyID id) const
{
  return debugKeys[id].active;
}


void DebugKeyTable::reset()
{
  for (int key=0;key<DebugKeyTable::numOfDebugKeys;key++)
  {
    set((debugKeyID) key, DebugKey::disabled, 0);
  }
}

void DebugKeyTable::activate()
{
  for (int key=0;key<DebugKeyTable::numOfDebugKeys;key++)
  {
    switch (debugKeys[key].mode)
    {
      case DebugKey::disabled:
        debugKeys[key].active = false;
        break;
      case DebugKey::always:
        debugKeys[key].active = true;
        break;
      case DebugKey::n_times:
        if (debugKeys[key].last < debugKeys[key].n) 
        {
          debugKeys[key].last++;
          debugKeys[key].active = true;
        }
        else
        { 
          debugKeys[key].active = false;
        }    
        break;
      case DebugKey::every_n_times:
        if(debugKeys[key].n == 0) debugKeys[key].n = 1;
        debugKeys[key].last++;
        debugKeys[key].active = (debugKeys[key].last % debugKeys[key].n == 0);
        break;
      case DebugKey::every_n_ms:
        {
          unsigned long currentTime = SystemCall::getCurrentSystemTime();
          if (currentTime >= debugKeys[key].last + debugKeys[key].n) 
          {
            debugKeys[key].last = currentTime;
            debugKeys[key].active = true;
          }
          else
          {
            debugKeys[key].active = false;
          }
        }
        break;
      default:
        debugKeys[key].active = false;
    }
  }
}

bool DebugKeyTable::operator == (const DebugKeyTable& other)
{
  for (int key=0;key<DebugKeyTable::numOfDebugKeys;key++)
  {
    if (debugKeys[key].mode != other.debugKeys[key].mode
      || debugKeys[key].n != other.debugKeys[key].n)
      return false;
  }    
  return true;
}

In& operator>>(In& stream,DebugKeyTable& debugKeyTable)
{
  //old versions of a debugKeyTable might have less entries than numOfDebugKeys.
  char numberOfDebugKeysInStream;
  stream >> numberOfDebugKeysInStream;

  for (int i = 0;i < numberOfDebugKeysInStream; i++)
  {
    if(i < DebugKeyTable::numOfDebugKeys)
    {
      int mode;
      unsigned int n;
      stream >> mode >> n;
      debugKeyTable.set((DebugKeyTable::debugKeyID)i, (DebugKey::modes)mode, n);
    }
  }
  return stream;
}

Out& operator<<(Out& stream, DebugKeyTable& debugKeyTable)
{
  stream << (char)DebugKeyTable::numOfDebugKeys;

  for (int i=0;i<DebugKeyTable::numOfDebugKeys;i++)
  {
    stream << (int)debugKeyTable.getMode((DebugKeyTable::debugKeyID)i)
      << debugKeyTable.getN((DebugKeyTable::debugKeyID)i);
  }
  return stream;
}

/*
 * Change log :
 * 
 * $Log: DebugKeyTable.cpp,v $
 * Revision 1.1  2003/10/07 10:13:22  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1.1.1  2003/07/02 09:40:28  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.7  2002/12/18 16:22:56  dueffert
 * doxygen docu corrected
 *
 * Revision 1.6  2002/11/19 17:38:31  dueffert
 * doxygen bugs corrected
 *
 * Revision 1.5  2002/11/19 15:43:03  dueffert
 * doxygen comments corrected
 *
 * Revision 1.4  2002/10/01 11:14:34  loetzsch
 * Redesigned DebugKey and DebugKeyTable
 *
 * Revision 1.3  2002/09/29 18:03:29  loetzsch
 * renamed calculate... to activateDebugKeys
 *
 * Revision 1.2  2002/09/29 12:32:37  juengel
 * Changed semantics of "debug key is active".
 * If a debug key is active changes only before the execution of a process.
 *
 * Revision 1.1  2002/09/10 15:53:58  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.2  2002/06/04 00:11:52  loetzsch
 * added == and != operator
 *
 * Revision 1.1.1.1  2002/05/10 12:40:32  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.5  2002/02/18 14:07:22  juengel
 * streaming operators send and receive numberOfDebugKeys
 *
 * Revision 1.4  2002/01/23 07:50:03  loetzsch
 * removed workaround in the >> streaming operator
 *
 * Revision 1.3  2001/12/20 17:13:42  loetzsch
 * streaming operators implemented,  simplified
 *
 * Revision 1.2  2001/12/13 15:03:02  loetzsch
 * DebugKeyToolBar fertig
 *
 * Revision 1.1  2001/12/12 15:21:09  juengel
 * CDebugKeyToolBar in eigenes .cpp und .h File verlagert.
 *
 * Revision 1.5  2001/12/10 17:47:05  risler
 * change log added
 *
 */
