/**
* @file Tools/Debugging/Debugging.h
*
* Macros and functions for debugging
* 
* @author Martin Ltzsch
*/ 
#ifndef __Debugging_h_
#define __Debugging_h_

// Please don't include GTAssert.h here. GTAssert.h includes afx.h in the Win32 version,
// which makes it a bit slow and is not needed in the most files where Debugging.h is used.
// M.L.
//#include "Platform/GTAssert.h"

#include "Tools/Debugging/DebugKeyTable.h"
#include "Tools/MessageQueue/OutMessage.h"

/**
* Sets a pointer to an OutMessage and to a DebugKeyTable to be used by the macros.
*
* @param queue A pointer to an OutMessage.
* @param table A pointer to a DebugKeyTable.
*/
void initDebugging(OutMessage* queue, DebugKeyTable* table);

/**
* Returns a reference to a process wide OutMessage.
*/
OutMessage& getDebugOut();

/**
* Returns a reference to a process wide debug key table
*/
DebugKeyTable& getDebugKeyTable();


#ifdef NDEBUG

#define INFO(key,type,format,expression) /**/
#define OUTPUT(type,format,expression) /**/
#define WATCH(key,type,format,expression) /**/

#else //NDEBUG

/**
* A macro for sending debug messages depending on a debug key.
*
* @param key A key from the DebugKeyTable::debugKeyID enum.
* @param type The type of the message from the MessageID enum in MessageIDs.h
* @param format The message format of the message (bin or text)
* @param expression A streamable expression
*
* Examples:
* <pre>
* INFO(sendImages,idImage,bin,myImage);
* INFO(sendInterestingValue,idInterestingValue,text,"value: " << value);
* INFO(sendWorldState,idWorldState,bin,worldState);
* </pre>
*/
#define INFO(key,type,format,expression) \
  if (getDebugKeyTable().isActive(DebugKeyTable::key)) {\
    getDebugOut().format << expression;\
    getDebugOut().finishMessage(type);\
  }\

/**
* A macro for sending debug messages.
*
* @param type The type of the message from the MessageID enum in MessageIDs.h
* @param format The message format of the message (bin or text)
* @param expression A streamable expression
*
* Examples:
* <pre>
* OUTPUT(idImage, bin, *pMyImage);
* OUTPUT(idText, text, "MyObject::myFunction() invoked");
* OUTPUT(idText, text, "i: " << i << ", j:" << j);
* </pre>
*/
#define OUTPUT(type,format,expression) \
  { getDebugOut().format << expression;\
  getDebugOut().finishMessage(type); }

#if defined(_WIN32) && !defined(SIMROBOT)

/**
* A macro for sending debug messages that differs between Windows and Aperios.
* The parameters equal to the parameters if the INFO macro.
* Different from INFO, the output is sent automatically on the WIN32 platform
*/
#define WATCH(key,type,format,expression) \
  OUTPUT(type,format,expression);


#else

/**
* A macro for sending debug messages that differs between Windows and Aperios.
* The parameters equal to the parameters if the INFO macro.
* Different from INFO, the output is sent automatically on the WIN32 platform
*/
#define WATCH(key,type,format,expression) \
  INFO(key,type,format,expression);

#endif

#endif //NDEBUG

#endif //__Debugging_h_

/*
 * Change log :
 * 
 * $Log: Debugging.h,v $
 * Revision 1.2  2004/03/28 12:02:47  roefer
 * All drawings can be switched on and off in simulator
 *
 * Revision 1.1  2003/10/07 10:13:22  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.2  2003/09/26 15:28:10  juengel
 * Renamed DataTypes to representations.
 *
 * Revision 1.1.1.1  2003/07/02 09:40:28  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.6  2003/06/05 12:15:59  dueffert
 * if () OUTPUT() is allowed now
 *
 * Revision 1.5  2002/10/01 11:14:34  loetzsch
 * Redesigned DebugKey and DebugKeyTable
 *
 * Revision 1.4  2002/09/29 18:02:51  loetzsch
 * removed the INIT_DEBUG_KEY_TABLE macro from debugging.h
 *
 * Revision 1.3  2002/09/29 12:32:37  juengel
 * Changed semantics of "debug key is active".
 * If a debug key is active changes only before the execution of a process.
 *
 * Revision 1.2  2002/09/17 23:55:23  loetzsch
 * - unraveled several datatypes
 * - changed the WATCH macro
 * - completed the process restructuring
 *
 * Revision 1.1  2002/09/10 15:53:58  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.2  2002/07/23 13:48:28  loetzsch
 * - new streaming classes
 * - removed many #include statements
 * - exchanged StaticQueue by MessageQueue
 * - new debug message handling
 * - general clean up
 *
 * Revision 1.1.1.1  2002/05/10 12:40:32  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.11  2002/05/06 11:50:21  risler
 * removed test for debugOut
 *
 * Revision 1.10  2002/05/05 13:51:51  risler
 * test if debugOut exists in OUTPUT,INFO
 *
 * Revision 1.9  2002/04/20 15:52:23  roefer
 * Project simpified, WATCH and WATCH_PART added
 *
 * Revision 1.8  2002/01/22 00:08:00  loetzsch
 * Warnings for meaningless static const int ....() functions fixed
 *
 * Revision 1.7  2001/12/12 18:55:23  loetzsch
 * Representations/DebugKeyTable.h entfernt
 *
 * Revision 1.6  2001/12/12 18:08:56  loetzsch
 * Streaming- Operatoren fr Bilder eingebaut, DebugKeyTable nicht- statisch gemacht, Debuggin Mechanismen weitergemacht, Bilder aus Logfiles in RobotControl anzeigen, Logfiles in HU1/Debug auf den Stick schreiben
 *
 * Revision 1.5  2001/12/10 17:47:10  risler
 * change log added
 *
 */
