/**
 * \class IniFile  
 * Key/Value based .INI-File handling.
 *
 * The values are stored in a double layered map.  The first
 * layer contains the section names, the second layer contains
 * the actual key/value pairs. All names are case-insensitive,
 * and are kept in sorted order. Insertion, lookup and removal
 * of keys should take logarithmic time.
 *
 * \author  Thomas Kindler, thomas.kindler@gmx.de
 */

#include "IniFile.h"
#include "Streams/InStreams.h" 
#include "Streams/OutStreams.h" 
#include <strstream>

/**
 * Construct an empty IniFile.
 */
IniFile::IniFile() 
{
}


/**
 * Construct an IniFile and load settings.
 *
 * \param  filename  name of .ini file to load
 */
IniFile::IniFile(const char *filename) 
{
  load(filename);  
}


/**
 * Remove a key from a section.
 * \note    section and key names are case-insensitive.
 *
 * \param  section  section of key
 * \param  key      name of key to remove
 */
void IniFile::removeKey(const char *section, const char *key)
{
  Sections::iterator  s = sections.find(section);
  if (s != sections.end())
    s->second.erase(key);
}


/**
 * Load settings from an .ini file.
 *
 * Lines can be arbitrarily long and may use \-escaping
 * and quoted whitespace. Comments must start with an ';'.
 * 
 * \note  
 *   The contents of the file are <i>added</i> to the current
 *   list of settings. You can safely load multiple .ini-files
 *   into one IniFile instance.
 *
 * \param   filename  name of .ini file to load
 * \return  true if successful, false otherwise
 */
bool IniFile::load(const char *filename)
{
  InBinaryFile  ini(filename);
  if (!ini.exists())
    return false;

  // ini-parser state machine
  //
  enum {
    INITIAL, IGNORE,
    SECTION, KEY, AFTER_KEY, 
    BEFORE_VALUE, VALUE
  } state = INITIAL;
  
  string  s, section, key, value;
  bool  escape = false, quote = false;

  for (;;) {
    // read one character
    // (i hate the GT2004 stream functions!)
    //  
    unsigned char uc;
    ini.read(&uc, 1);
    int c = ini.getEof() ? EOF : uc;

    // check for special characters
    //
    if (escape)    { s+=c; escape=false; continue; } 
    if (c == '\\') { escape = true;      continue; }
    if (c == '"')  { quote  = !quote;    continue; }

    // end of line (or file) reached
    // 
    if (c == '\n' || c == '\r' || c == EOF) {
      if (state == VALUE)
        value = s;       
      if (key != "")
        sections[section][key] = value;

      state = INITIAL;
      quote = false;

      if (c == EOF)  
        break;
      else
        continue;
    }

    // parse normal characters
    //
    switch (state) {
      case INITIAL:  // wait for first char
        s = key = value = "";
        if      (c == '[')    { state = SECTION; }
        else if (c == ';')    { state = IGNORE;  }
        else if (!isspace(c)) { state = KEY;     }
        break;

      case SECTION:  // read section
        if (quote) break;
        if (c == ']') { 
          section = s.substr(1);
          state   = IGNORE;
        }
        break;
      
      case KEY:  // read key
        if (quote)  break;
        if (isspace(c)) {
          key   = s;
          s     = "";
          state = AFTER_KEY;
        } else if (c == '=') {
          key   = s;
          s     = "";
          state = BEFORE_VALUE;
        }
        break;

      case AFTER_KEY:  // wait for '='
        if (c == '=') {
          state = BEFORE_VALUE;
        } else if (!isspace(c)) {
          state = IGNORE;
        }
        break;

      case BEFORE_VALUE:  // wait for value
        if (!(isspace(c))) {
          s = "";
          state = VALUE;
        }
        break;

      case VALUE:  // read value
        if (quote)  break;
        if (isspace(c)) {
          value = s;
          state = IGNORE;
        }
        break;   

      case IGNORE:
        break;
    }
    s += c;
  }  
    
  return true;
}


/**
 * Escape special characters and add quotes if necessary. 
 *
 * \param   s  string to escape
 * \return  ini-safe string
 */
static const string esc(const string s)
{
  if (s.size() == 0)
    return "\"\"";
  
  string t = "";
  bool  mustQuote = false;

  for (unsigned i=0; i<s.size(); i++) {
    if (isspace(s[i]) || 
        s[i] == '=' || s[i] == ';' || 
        s[i] == '[' || s[i] == ']'  )
      mustQuote = true;

    if (!isprint(s[i]) || s[i] == '\\' || s[i] == '"')
      t += '\\';

    t += s[i];
  }
    
  return mustQuote ? "\"" + t + "\"" : t;
}


/**
 * Save setting to an .ini-File.
 *
 * \note  
 *   Sections and keys are saved in undefined order,
 *   formatting and comments are not preserved.
 *
 * \param  filename  name of .ini file to save into
 */
bool IniFile::save(const char *filename) const
{
  OutTextRawFile  ini(filename, false);
  if (!ini.exists())
    return false;

  Sections::const_iterator s;
  for (s = sections.begin(); s != sections.end(); s++) {
    ini << "[" << esc(s->first).c_str() << "]\n";
    
    Keys::const_iterator k;
    for (k = s->second.begin(); k != s->second.end(); k++)
      ini << esc(k->first).c_str()  << " = "
          << esc(k->second).c_str() << "\n";

    ini << "\n";
  }
  return true;
}


/**
 * Set value of key.
 * \note    section and key names are case-insensitive.
 *
 * \param   section  section of key
 * \param   key      name of key to set
 * \param   val      value to set
 */
void IniFile::setString(const char *section, const char *key, const char *val)
{
  sections[section][key] = val;
}


/**
 * Get value of key.
 * \note    section and key names are case-insensitive.
 *
 * \param   section  section of key
 * \param   key      name of key to retrieve
 * \param   def      default value
 *
 * \return  value of key, or def if key doesn't exist
 */
const char *IniFile::getString(const char *section, const char *key, const char *def) const
{
  Sections::const_iterator  s = sections.find(section);
  if (s != sections.end()) {
    Keys::const_iterator  k = s->second.find(key);
    if (k != s->second.end())
      return  k->second.c_str();
  } 
  return  def;
}


int  IniFile::getInt(const char *section, const char *key, int def) const
{
  string s = getString(section, key, "");
  return s == "" ? def : strtol(s.c_str(), NULL, 0);
}


double  IniFile::getDouble(const char *section, const char *key, double def) const
{
  string s = getString(section, key, "");
  return s == "" ? def : strtod(s.c_str(), NULL);
}


bool  IniFile::getBool(const char *section, const char *key, bool def) const
{
  const char *s = getString(section, key, "");
  if (*s == '\0')
    return def;
  
  if (!stricmp(s, "true") || !stricmp(s, "yes") ||
      !stricmp(s, "on")   || strtol(s, NULL, 0) != 0)
    return true;
  else
    return false;
}
    

void  IniFile::setInt(const char *section, const char *key, int val)
{
  strstream s; s << val;
  setString(section, key, s.str());
}


void  IniFile::setDouble(const char *section, const char *key, double val)
{
  strstream s; s << val;
  setString(section, key, s.str());
}


void  IniFile::setBool(const char *section, const char *key, bool val)
{
  setString(section, key, val ? "true" : "false");
}


/**
 * Change log:
 *
 * $Log: IniFile.cpp,v $
 * Revision 1.1  2004/04/13 23:35:16  kindler
 * initial import
 *
 */
