/** 
* @file KickSelectionTable.h
* Declaration of class KickSelectionTable.
*
* @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Jngel</A>
* @author Martin Ltzsch
*/

#ifndef _KickSelectionTable_h_
#define _KickSelectionTable_h_

#include "Tools/Math/Vector2.h"
#include "Platform/GTAssert.h"

/**
* @class KickSelectionTable
*
* @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Jngel</A>
* @author Martin Ltzsch
*
* Provides for a set of tables which provides a kick selection depending 
* on a desired kick angle and ball position.
*/
class KickSelectionTable
{
public:
  /** Constructor */
  KickSelectionTable();
  
  /** Destructor */
  ~KickSelectionTable();
  
  /** Ids for all possible actions */
  enum ActionID{
    nothing = 0,
      athErs7ChestKick2,
      athErs7fastDiveKick,
      athErs7ForwardLeapingKick,
      athErs7HeadKickLeft,
      athErs7HeadKickRight,
      athErs7KickForwardWithLeftPaw,
	    athErs7KickForwardWithRightPaw,
      athErs7SideHeadKickL,
      athErs7SideHeadKickR,
      armKickLeft,
      armKickRight,
      msh7AnyLeft,
      msh7AnyRight,
      msh7LeftHook,
      msh7RightHook,
      msh7ForwardLeft,
      msh7ForwardRight,
      kickWithHeadRight,    //head control mode
      kickWithHeadLeft,     //head control mode
      numberOfActions
  };
  
  /** returns names for action ids */
  static const char* getActionName(ActionID id)
  {
    switch (id)
    {
    case nothing: return "nothing";
    case athErs7ForwardLeapingKick: return "athErs7ForwardLeapingKick";
    case athErs7ChestKick2: return "athErs7ChestKick2";
    case athErs7fastDiveKick: return "athErs7fastDiveKick";
    case athErs7SideHeadKickL: return "athErs7SideHeadKickL";
    case athErs7SideHeadKickR: return "athErs7SideHeadKickR";
    case athErs7KickForwardWithLeftPaw: return "athErs7KickForwardWithLeftPaw";
	  case athErs7KickForwardWithRightPaw: return "athErs7KickForwardWithRightPaw";
    case athErs7HeadKickLeft: return "athErs7HeadKickLeft";
    case athErs7HeadKickRight: return "athErs7HeadKickRight";
    case armKickLeft: return "armKickLeft";
    case armKickRight: return "armKickRight";
      
    case kickWithHeadRight: return "kickWithHeadRight";
    case kickWithHeadLeft: return "kickWithHeadLeft";
    case msh7AnyLeft: return "msh7AnyLeft";
    case msh7AnyRight: return "msh7AnyRight";
    case msh7LeftHook: return "msh7LeftHook";
    case msh7RightHook: return "msh7RightHook";
    case msh7ForwardLeft: return "msh7ForwardLeft";
    case msh7ForwardRight: return "msh7ForwardRight";

    default: return "please edit KickSelectionTable::getActionName";
    }
  }
  
  /** returns short names for action ids (used by streaming operator)*/
  static const char* getShortActionName(ActionID id)
  {
    switch (id)
    {
    case nothing: return "no";
    case athErs7ForwardLeapingKick: return "7fLeapK";
    case athErs7ChestKick2: return "7ChestK2";
    case athErs7fastDiveKick: return "7fastDV";
    case athErs7SideHeadKickL: return "7SideHKL";
    case athErs7SideHeadKickR: return "7SideHKR";
    case athErs7KickForwardWithLeftPaw: return "7KfwlP";
    case athErs7KickForwardWithRightPaw: return "7KfwrP";
    case athErs7HeadKickLeft: return "7HeadKL";
    case athErs7HeadKickRight: return "7HeadKR";
    case kickWithHeadRight: return "kWHR";
    case kickWithHeadLeft: return "kWHL";
    case armKickLeft: return "aKL";
    case armKickRight: return "aKR";

    case msh7AnyLeft: return "m7AnyL";
    case msh7AnyRight: return "m7AnyR";
    case msh7LeftHook: return "m7LHook";
    case msh7RightHook: return "m7RHook";
    case msh7ForwardLeft: return "m7ForwardL";
    case msh7ForwardRight: return "m7ForwardR";

    default: 
      ASSERT(false);
      return 0;
    }
  }
  
  /** IDs for different kick selection tables */
  enum KickSelectionTableID
  {
    inCenterOfField, 
      atLeftBorder, 
      atRightBorder, 
      atLeftOpponentBorder, 
      atRightOpponentBorder, 
      nearOwnGoal, 
      goalie,
      numberOfKickSelectionTableIDs
  };
  
  /** returns a name for an action table id */
  static const char* getKickSelectionTableIDName(KickSelectionTableID id)
  {
    switch (id)
    {
    case inCenterOfField: return "inCenterOfField";
    case atLeftBorder: return "atLeftBorder";
    case atRightBorder: return "atRightBorder";
    case atLeftOpponentBorder: return "atLeftOpponentBorder";
    case atRightOpponentBorder: return "atRightOpponentBorder";
    case nearOwnGoal: return "nearOwnGoal"; 
    case goalie: return "goalie";
    default: 
      ASSERT(false);
      return 0;
    }
  }
  
  
  /**  */ 
  enum{xRange = 20}; 
  
  /**  */ 
  enum{yRange = 40};
  
  /**  */ 
  enum{numberOfSectors = 12};
  
  /** 
  * Calculates the best kick based on the position of the ball relative to the 
  * robot and a target angle. 
  */
  ActionID retrieveKick(
    double ballOffsetX, double ballOffsetY, 
    double destinationAngle,
    KickSelectionTableID kickSelectionTableID
    ) const;
  
  /** The action tables */
  ActionID action[xRange][yRange][numberOfSectors][numberOfKickSelectionTableIDs];
  
  /** Returns a table id for a name */
  static KickSelectionTableID getTableIDFromName(const char* name);
  
  /** Returns an action id for a short name */
  static ActionID getActionIDFromShortName(const char* name);
  
  /** Loads a kick selection table from a file */
  void load(const char* fileName);
};

/**
* Streaming operator that reads KickSelectionTable from a stream.
* @param stream The stream from which is read.
* @param kickSelectionTable The KickSelectionTable object.
* @return The stream.
*/ 
In& operator>>(In& stream, KickSelectionTable& kickSelectionTable);

/**
* Streaming operator that writes KickSelectionTable to a stream.
* @param stream The stream to write on.
* @param kickSelectionTable The KickSelectionTable object.
* @return The stream.
*/ 
Out& operator<<(Out& stream, const KickSelectionTable& kickSelectionTable);



#endif   //  _KickSelectionTable_h_

/*
* Change log :
* 
* $Log: KickSelectionTable.h,v $
* Revision 1.9  2004/05/04 18:10:01  juengel
* Added short names for msh kicks
*
* Revision 1.8  2004/05/04 14:44:37  loetzsch
* added a few kicks
*
* Revision 1.7  2004/04/05 17:56:50  loetzsch
* merged the local German Open CVS of the aibo team humboldt with the tamara CVS
*
* Revision 1.3  2004/04/03 15:31:10  juengel
* bug fixed
*
* Revision 1.2  2004/04/03 01:29:25  loetzsch
* added arm kicks
*
* Revision 1.1.1.1  2004/03/31 11:17:00  loetzsch
* created ATH repository for german open 2004
*
* Revision 1.6  2004/03/28 16:02:01  juengel
* beautified
*
* Revision 1.5  2004/03/27 20:33:22  loetzsch
* added table id for goalie
*
* Revision 1.4  2004/03/19 21:19:28  altmeyer
* SideHeadKick and SideHeadKick2 changed and renamed to SideHeadKick L and SideHeadKickR
* a few ATH ERS7-kicks removed
*
* Revision 1.3  2004/03/17 19:55:55  juengel
* Comments added.
*
* Revision 1.2  2004/03/16 16:26:16  altmeyer
* added new ATH-ERS7 kicks
*
* Revision 1.1  2004/03/16 14:00:23  juengel
* Integrated Improvments from "Gnne"
* -ATH2004ERS7Behavior
* -ATHHeadControl
* -KickSelectionTable
* -KickEditor
*
* Revision 1.11  2004/03/11 17:29:11  juengel
* Added kicks with head.
*
* Revision 1.10  2004/03/11 11:40:28  loetzsch
* renamed fastHeadKickRight to headKickRight
*
* Revision 1.9  2004/03/10 18:41:00  loetzsch
* first functioning version
*
* Revision 1.8  2004/03/10 14:06:04  juengel
* Implemented retrieveKick.
*
* Revision 1.7  2004/03/09 18:48:58  juengel
* Added ActionID "nothing".
*
* Revision 1.6  2004/03/09 18:43:45  juengel
* Added ActionID "nothing".
*
* Revision 1.5  2004/03/09 18:41:09  loetzsch
* no message
*
* Revision 1.4  2004/03/09 14:56:37  loetzsch
* bug fix
*
* Revision 1.3  2004/03/09 14:54:09  loetzsch
* changed comments
*
* Revision 1.2  2004/03/09 14:45:27  loetzsch
* automated generation of kick-selection-symbols.xml
*
* Revision 1.1  2004/03/09 13:40:30  juengel
* Moved KickSelectionTable to Tools.
*
* Revision 1.1  2004/03/09 01:14:36  juengel
* Added class KickSelectionTable.
*
*/
