/**
* @file Tools/Math/Geometry.h
* Declares class Geometry
*
* @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Jngel</A>
* @author <a href="mailto:walter.nistico@uni-dortmund.de">Walter Nistico</a>
*/

#ifndef __Geometry_h__
#define __Geometry_h__

#include "Tools/Math/Pose2D.h"
#include "Representations/Perception/CameraMatrix.h"
#include "Representations/Perception/CameraInfo.h"
#include "Representations/Cognition/RobotPose.h"
#include "Representations/Motion/HeadState.h"
#include "Tools/Actorics/RobotDimensions.h"




/**
* The class Geometry defines representations for geometric objects and Methods
* for calculations with such object.
*
*/
class Geometry
{
public:
  
  /** Defines a circle by its center and its radius*/
  struct Circle
  {
    Circle();
    Vector2<double> center;
    double radius;
  };
  
  /** Defines a line by two vectors*/
  struct Line
  {
    Vector2<double> base;
    Vector2<double> direction;
    
    Line() {};
    Line(Vector2<double> base, Vector2<double> direction) : 
    base(base), 
      direction(direction)
    {};
    
    Line(Vector2<int> base, Vector2<double> direction) : 
    direction(direction)
    {
      this->base.x = base.x;
      this->base.y = base.y;
    };

    Line(Vector2<int> base, Vector2<int> direction)
    {
      this->base.x = base.x;
      this->base.y = base.y;
      this->direction.x = direction.x;
      this->direction.y = direction.y;
    };

    
    Line(double baseX, double baseY, double directionX, double directionY) 
    {
      base.x = baseX; 
      base.y = baseY; 
      direction.x = directionX;
      direction.y = directionY;
    };
    
    void normalizeDirection();
  };
  
  struct PixeledLine
  {
    PixeledLine(int x1, int x2, int y1, int y2):x1(x1),y1(y1),x2(x2),y2(y2) 
    {
      calculatePixels();
    };

    PixeledLine(Vector2<int> start, Vector2<int> end):x1(start.x),y1(start.y),x2(end.x),y2(end.y)
    {
      calculatePixels();
    };

    void calculatePixels()
    {
      char sign;
      if(x1 == x2 && y1 == y2)
      {
        numberOfPixels = 0;
      }
      else //begin and end differ
      {
        if(abs(x2 - x1) > abs(y2 - y1) )
        {
          if(x1 < x2) sign = 1; else sign = -1;
          numberOfPixels = abs(x2 - x1) + 1;
          for(int x = 0; x < numberOfPixels; x++)
          {
            int y = (int)(x * (y2 - y1) / (x2 - x1));
            x_coordinate[x] = x1 + x*sign;
            y_coordinate[x] = y1 + y*sign;
          }
        }
        else
        {
          if(y1 < y2) sign = 1; else sign = -1;
          numberOfPixels = abs(y2 - y1) + 1;
          for(int y = 0; y < numberOfPixels; y++)
          {
            int x = (int)(y * (x2 - x1) / (y2 - y1));
            x_coordinate[y] = x1 + x*sign;
            y_coordinate[y] = y1 + y*sign;
          }
        }
      } //begin and end differ
    }// calculatePixels
    
    inline int getNumberOfPixels() const
    {
      return numberOfPixels;
    }

    inline int getPixelX(int i) const
    {
      return x_coordinate[i];
    }

    inline int getPixelY(int i) const
    {
      return y_coordinate[i];
    }

  private:
    int x1, y1, x2, y2;
    int numberOfPixels;
    //enum{maxNumberOfPixelsInLine = 228}; //diagonal size of image
	enum{maxNumberOfPixelsInLine = 600}; //diagonal size of BW image 
    int x_coordinate[maxNumberOfPixelsInLine];
    int y_coordinate[maxNumberOfPixelsInLine];
  };

  template <class V, int maxNumberOfPoints> class SetOfPoints
  {
  public:
    /** Constructor */
    SetOfPoints() {init();}
    void add(V x, V y)
    {
      if(numberOfPoints >= maxNumberOfPoints) return;
      points[numberOfPoints].x = y;
      points[numberOfPoints].y = y;
      numberOfPoints++;
    }
    void add(Vector2<V> newPoint)
    {
      if(numberOfPoints >= maxNumberOfPoints) return;
      points[numberOfPoints] = newPoint;
      numberOfPoints++;
    }

    void init() {numberOfPoints = 0;};
    
    double linearRegression(Line& line)
    {
      if(numberOfPoints == 0) return 0;
      double col1[maxNumberOfPoints], col2[maxNumberOfPoints], col3[maxNumberOfPoints], col4[maxNumberOfPoints], col5[maxNumberOfPoints];
      double sumX, sumY, averageX, averageY, c3sum, c4sum, c5sum, m, n;
      
      sumX=0; sumY=0; c3sum=0; c4sum=0; c5sum=0;

      int i;
      for (i = 0; i < numberOfPoints; i++)
      {
        sumX = sumX + points[i].x;
        sumY = sumY + points[i].y;
      }
      
      averageX = sumX / numberOfPoints;
      averageY = sumY / numberOfPoints;
      
      for (i = 0; i < numberOfPoints; i++)
      {
        col1[i] = points[i].x - averageX;
        col2[i] = points[i].y - averageY;
        col3[i] = col1[i] * col2[i];
        col4[i] = col1[i] * col1[i];
        col5[i] = col2[i] * col2[i];
      }
      
      for (i = 0; i < numberOfPoints; i++)
      {
        c3sum = c3sum + col3[i];
        c4sum = c4sum + col4[i];
        c5sum = c5sum + col5[i];
      }
      
      m = c3sum / c4sum;
      n = averageY - m * averageX;
      
      line.base.x = 0;
      line.base.y = n;
      line.direction.x = 1;
      line.direction.y = m;

      // reliability:
      return fabs(c3sum / sqrt(c4sum * c5sum));
    }
  private:
    Vector2<V> points[maxNumberOfPoints];
    int numberOfPoints;
  };

  /** 
  * Calculates the angle between a pose and a position
  * @param from The base pose.
  * @param to The other position.
  * @return the angle from the pose to the position.
  */
  static double angleTo(const Pose2D& from, 
    const Vector2<double>& to);

  /** 
  * Calculates the distance from a pose to a position
  * @param from The base pose.
  * @param to The other position.
  * @return the distance from the pose to the position.
  */
  static double distanceTo(const Pose2D& from, 
    const Vector2<double>& to);

  /** 
  * Calculates the relative vector from a pose to a position
  * @param from The base pose.
  * @param to The other position.
  * @return the vector from the pose to the position.
  */
  static Vector2<double> vectorTo(const Pose2D& from, 
    const Vector2<double>& to);

  /**
  * Returns the circle defined by the three points.
  * @param point1 The first point.
  * @param point2 The second point.
  * @param point3 The third point.
  * @return The circle defined by point1, point2 and point3.
  */
  static Circle getCircle(
    const Vector2<int> point1,
    const Vector2<int> point2,
    const Vector2<int> point3
    );
  
  static bool getIntersectionOfLines(
    const Line line1,
    const Line line2,
    Vector2<double>& intersection
    );

  static bool getIntersectionOfLines(
    const Line line1,
    const Line line2,
    Vector2<int>& intersection
    );

  static bool getIntersectionOfRaysFactor(
    const Line ray1,
    const Line ray2,
    double& intersection
    );
  
  static double getDistanceToLine(
    const Line line,
    const Vector2<double>& point
    );
  
  static double getDistanceToEdge(
    const Line line,
    const Vector2<double>& point
    );

  static double distance(
    const Vector2<double>& point1,
    const Vector2<double>& point2
    );

  static double distance(
    const Vector2<int>& point1,
    const Vector2<int>& point2
    );

  static void calculateAnglesForPoint(
    const Vector2<int>& point, 
    const CameraMatrix& cameraMatrix, 
    const CameraInfo& cameraInfo, 
    Vector2<double>& angles
    );
    
  static void calculateAnglesForPointIntrinsic(
    const Vector2<int>& point, 
    const CameraMatrix& cameraMatrix, 
    const CameraInfo& cameraInfo, 
    Vector2<double>& angles
    );

  static void calculatePointByAngles(
    const Vector2<double>& angles,
    const CameraMatrix& cameraMatrix, 
    const CameraInfo& cameraInfo, 
    Vector2<int>& point
    );

  static void calculatePointByAnglesIntrinsic(
    const Vector2<double>& angles,
    const CameraMatrix& cameraMatrix, 
    const CameraInfo& cameraInfo, 
    Vector2<int>& point
    );

  static bool clipLineWithQuadrangle(
    const Line lineToClip,
    const Vector2<double>& corner0, 
    const Vector2<double>& corner1, 
    const Vector2<double>& corner2, 
    const Vector2<double>& corner3, 
    Vector2<double>& clipPoint1, 
    Vector2<double>& clipPoint2
    );

  static bool clipLineWithQuadrangle(
    const Line lineToClip,
    const Vector2<double>& corner0, 
    const Vector2<double>& corner1, 
    const Vector2<double>& corner2, 
    const Vector2<double>& corner3, 
    Vector2<int>& clipPoint1, 
    Vector2<int>& clipPoint2
    );

  static bool isPointInsideRectangle(
    const Vector2<double>& bottomLeftCorner, 
    const Vector2<double>& topRightCorner,
    const Vector2<double>& point
    );
  
  static bool isPointInsideRectangle(
    const Vector2<int>& bottomLeftCorner, 
    const Vector2<int>& topRightCorner,
    const Vector2<int>& point
    );

  /** 
  * Calculates where a pixel in the image lies on the ground (relative to the robot).
  * @param x Specifies the x-coordinate of the pixel.
  * @param y Specifies the y-coordinate of the pixel.
  * @param cameraMatrix The camera matrix of the image.
  * @param pointOnField The resulting point.
  */
  static bool calculatePointOnField(
    const int x,
    const int y,
    const CameraMatrix& cameraMatrix,
    const CameraInfo& cameraInfo,
    Vector2<int>& pointOnField
    );

  static bool calculatePointOnFieldIntrinsic(
    const int x,
    const int y,
    const CameraMatrix& cameraMatrix,
    const CameraInfo& cameraInfo,
    Vector2<int>& pointOnField
    );

  /** 
  * Calculates where a relative point on the ground appears in an image.
  * @param point The coordinates of the point relative to the robot's origin.
  * @param cameraMatrix The camera matrix of the image.
  * @param pointInImage The resulting point.
  */
  static void calculatePointInImage
  ( 
   const Vector2<int>& point,
   const CameraMatrix& cameraMatrix,
   const CameraInfo& cameraInfo,
   Vector2<int>& pointInImage
   );

  static void calculatePointInImageIntrinsic
  ( 
   const Vector2<int>& point,
   const CameraMatrix& cameraMatrix,
   const CameraInfo& cameraInfo,
   Vector2<int>& pointInImage
   );

  /**
  * Clips a line with a rectangle
  * @param bottomLeft The bottom left corner of the rectangle
  * @param topRight The top right corner of the rectangle
  * @param line The line to be clipped
  * @param point1 The starting point of the resulting line
  * @param point2 The end point of the resulting line
  * @return states whether clipping was necessary (and done)
  */
  static bool getIntersectionPointsOfLineAndRectangle(
   const Vector2<int>& bottomLeft, 
   const Vector2<int>& topRight,
   const Geometry::Line line,
   Vector2<int>& point1, 
   Vector2<int>& point2
   );


  /**
  * Clips a line with the Cohen-Sutherland-Algorithm
  * @param bottomLeft The bottom left corner of the rectangle
  * @param topRight The top right corner of the rectangle
  * @param point1 The starting point of the line
  * @param point2 The end point of the line
  * @return states whether clipping was necessary (and done)
  */
  static bool clipLineWithRectangleCohenSutherland(
   const Vector2<int>& bottomLeft, 
   const Vector2<int>& topRight,
   Vector2<int>& point1, 
   Vector2<int>& point2
   );

  /**
  * Clips a line with the Cohen-Sutherland-Algorithm if it has a point outside the rectangle.
  * Expands the line if it has a point inside the rectangle.
  * @param bottomLeft The bottom left corner of the rectangle
  * @param topRight The top right corner of the rectangle
  * @param point1 The starting point of the line
  * @param point2 The end point of the line
  */
  /*
  static bool ExpandOrClipLineWithRectangleCohenSutherland(
   const Vector2<int>& bottomLeft, 
   const Vector2<int>& topRight,
   Vector2<int>& point1, 
   Vector2<int>& point2
   );
*/

  /**
  * Calculates the Cohen Sutherland Code for a given point and a given rectangle
  */
  static int cohenSutherlandOutCode(
    const Vector2<int>& bottomLeft, 
    const Vector2<int>& topRight,
    Vector2<int>& point
    )
  { 
    int code;
    if ( point.x < bottomLeft.x )
      code = 1 ;
    else
      code = 0 ;
    if ( point.x > topRight.x )
      code += 2 ;
    if ( point.y < bottomLeft.y )
      code += 4 ;
    if ( point.y > topRight.y )
      code += 8 ;
    return(code) ;
  }

  /**
  * Calculates the intersection of an arbitrary line and a horizontal or vertical line.
  */
  static int intersection(int a1,int b1,int a2,int b2,int value);

  /**
  * Calculates the camera matrix.
  */
  static void calculateCameraMatrix(double tilt,double pan,double roll,
                                    const HeadState& headState,
                                    CameraMatrix& cameraMatrix);

  /**
  * Function does the transformation from 2d relative robot coordinates
	* to absolute field coordinates.
  * @param rp current Robot Pose.
  * @param x relative x-coordinate of ball (relative to robot)
  * @param y relative y-coordinate of ball (relative to robot)
  * @return Returns the ball positon in absolute coordinates
  */
	static Vector2<double> relative2FieldCoord(RobotPose rp, double x, double y);

  /**
  * The function approximates the shape of a ball in the camera image.
  * Note: currently, the approximation is not exact.
  * @param ballOffset The ball's position relative to the robot's body origin.
  * @param cameraMatrix The position and orientation of the robot's camera.
  * @param cameraInfo The resolution and the opening angle of the robot's camera.
  * @param circle The approximated shape generated by the function.
  * @return If false, only the center of the circle is valid, not the radius.
  */
  static bool calculateBallInImage(const Vector2<double>& ballOffset,
                                   const CameraMatrix& cameraMatrix, 
                                   const CameraInfo& cameraInfo, 
                                   Circle& circle);

  static bool calculateBallInImageIntrinsic(const Vector2<double>& ballOffset,
                                   const CameraMatrix& cameraMatrix, 
                                   const CameraInfo& cameraInfo, 
                                   Circle& circle);

  /** 
  * The function determines how far an object is away depending on its real size and the size in the image.
  * @param sizeInReality The real size of the object.
  * @param sizeInPixels The size in the image.
  * @param imageWidthPixels The horizontal resolution of the image.
  * @param imageWidthAngle The horizontal opening angle of the camera.
  * @return The distance between camera and object.
  */
  static int getDistanceBySize(
    int sizeInReality, 
    int sizeInPixels,
    int imageWidthPixels,
    double imageWidthAngle
    );
  static int getDistanceBySize
    (
     const CameraInfo& cameraInfo,
     int sizeInReality,
     int sizeInPixels
     );

  /** 
  * The function determines how far an object is away depending on its real size and the size in the image
  * along with its center position, using camera intrinsic parameters.
  * @param cameraInfo Class containing the intrinsic paramaters
  * @param sizeInReality The real size of the object.
  * @param sizeInPixels The size in the image.
  * @param centerX X coordinate (in image reference) of object's baricenter. 
  * @param centerY Y coordinate (in image reference) of object's baricenter. 
  * @return The distance between camera and object.
  */
  static double getDistanceBySize(
    const CameraInfo& cameraInfo,
    int sizeInReality, 
    double sizeInPixels,
    int centerX,
    int centerY
    );
  
  static double getDistanceBySizeIntrinsic(
    const CameraInfo& cameraInfo,
    int sizeInReality, 
    double sizeInPixels,
    int centerX,
    int centerY
    );

    /** 
  * The function determines how far an object is away depending on its real size and the size in the image.
  * @param sizeInReality The real size of the object.
  * @param sizeInPixels The size in the image.
  * @return The distance between camera and object.
  */
  static double getDistanceByAngleSize(
    int sizeInReality, 
    double sizeInPixels
    );

  /** 
  * The function determines how big an object appears in the image depending on its distance and size.
  * @param sizeInReality The real size of the object.
  * @param distance The distance to the object.
  * @param imageWidthPixels The horizontal resolution of the image.
  * @param imageWidthAngle The horizontal opening angle of the camera.
  * @return The size as it would appear in the image.
  */
  static int getSizeByDistance(
    int sizeInReality,
    int distance,
    int imageWidthPixels,
    double imageWidthAngle
    );
    
  static int getSizeByDistance
    (
     const CameraInfo& cameraInfo,
     int sizeInReality,
     int distance
    );

  /** 
  * The function determines how big an object appears in the image depending on its distance and size,
  * using intrinsic camera parameters.
  * @param cameraInfo Object containing camera paramters.
  * @param sizeInReality The real size of the object.
  * @param distance The distance to the object.
  * @return The size as it would appear in the image.
  */
  static int getSizeByDistanceIntrinsic
  (
   const CameraInfo& cameraInfo,
   int sizeInReality,
   int distance
  );


  /** 
  * The function calculates the horizon.
  * @param cameraMatrix The camera matrix.
  * @return The line of the horizon in the image.
  */
  static Geometry::Line calculateHorizon(
    const CameraMatrix& cameraMatrix,
    const CameraInfo& cameraInfo
    );

  static Geometry::Line calculateHorizonIntrinsic(
    const CameraMatrix& cameraMatrix,
    const CameraInfo& cameraInfo
    );

  /** 
  * The function calculates the expected size (pixel) of a field line in an image.
  * @param pointInImage The point where the line is expected.
  * @param cameraMatrix The camera matrix.
  * @return The size of a line pixel.
  */
  static int calculateLineSize(
    const Vector2<int>& pointInImage,
    const CameraMatrix& cameraMatrix,
    const CameraInfo& cameraInfo
    );

  static int calculateLineSizeIntrinsic(
    const Vector2<int>& pointInImage,
    const CameraMatrix& cameraMatrix,
    const CameraInfo& cameraInfo
    );

  /** 
  * Calculates the angle size for a given pixel size.
  */
  static double pixelSizeToAngleSize(double pixelSize, const CameraInfo& cameraInfo);

  /** 
  * Calculates the angle size for a given pixel size using camera intrinsic parameters.
  */
  static double pixelSizeToAngleSizeIntrinsic(double pixelSize, const CameraInfo& cameraInfo);

  /** 
  * Calculates the pixel size for a given angle size.
  */
  static double angleSizeToPixelSize(double angleSize, const CameraInfo& cameraInfo);
  
  static double angleSizeToPixelSizeIntrinsic(double angleSize, const CameraInfo& cameraInfo);

  /**
  * A radial distortion correction look-up table element 
  */
  struct CorrectedCoords {
    unsigned char x;
    unsigned char y;
  };  

  /** 
  * Corrects the radial distortion introduced by the camera lens (using a 4th order model) of a given pixel.
  * @param cameraInfo Class containing camera intrinsic parameters
  * @param srcX X coordinate (image reference) of pixel to be distortion-corrected.
  * @param srcY Y coordinate (image reference) of pixel to be distortion-corrected.
  * @param correctedX X coordinate (image reference) of resulting distortion-corrected pixel.
  * @param correctedY Y coordinate (image reference) of resulting distortion-corrected pixel.
  */
  static inline void radialDistortionCorrection(const CameraInfo& cameraInfo, const int srcX, const int srcY, double& correctedX, double& correctedY){
    double x1 = srcX - cameraInfo.opticalCenter.x;
    double y1 = srcY - cameraInfo.opticalCenter.y;
    double r_square = x1*x1 + y1*y1;
    double k1_num = cameraInfo.secondOrderRadialDistortion;
    double k1_den = cameraInfo.focalLenPow2;
    double k2_num = cameraInfo.fourthOrderRadialDistortion;
    double k2_den = cameraInfo.focalLenPow4;
    correctedX = srcX + x1*(k1_num*r_square/k1_den + k2_num*r_square*r_square/k2_den);
    correctedY = srcY + y1*(k1_num*r_square/k1_den + k2_num*r_square*r_square/k2_den);
  }

  /**
  * This function builds a radial distortion correction look-up table based on the parameters contained in cameraInfo.
  * @param cameraInfo Object containing intrinisc (and other) camera parameters.
  */
  static void setupRadialCorrection(const CameraInfo& cameraInfo);
  
  static inline void setupRadialCorrectionERS7()
  {
    CameraInfo cameraInfo(RobotDesign::ERS7);
    setupRadialCorrection(cameraInfo);
  }

  static inline void setupRadialCorrectionERS210()
  {
    CameraInfo cameraInfo(RobotDesign::ERS210);
    setupRadialCorrection(cameraInfo);
  }

	inline static bool insideCircle(Geometry::Circle &circle,double radius2,double x,double y){
		return  (((circle.center.x-x)*(circle.center.x-x)\
			+ (circle.center.y-y)*(circle.center.y-y)) < radius2);
	}

  /** 
  * Corrects the radial distortion introduced by the camera lens (using a 4th order model) of a given pixel,
  * using a pre-computed look-up table.
  * @param srcX X coordinate (image reference) of pixel to be distortion-corrected.
  * @param srcY Y coordinate (image reference) of pixel to be distortion-corrected.
  * @param correctedX X coordinate (image reference) of resulting distortion-corrected pixel.
  * @param correctedY Y coordinate (image reference) of resulting distortion-corrected pixel.
  */
  static inline void radialDistortionCorrectionFast(const int srcX, const int srcY, int& correctedX, int& correctedY)
  {
    CorrectedCoords result = radialCorrectionLUT[srcY][srcX];
    correctedX = result.x;
    correctedY = result.y;
  }

private:
  static CorrectedCoords radialCorrectionLUT[cameraResolutionHeight_ERS7][cameraResolutionWidth_ERS7]; /** Radius look up table, used to index the correction table. */
};

#endif //__Geometry_h____

/*
* Change log :
* 
* $Log: Geometry.h,v $
* Revision 1.15  2004/05/17 11:57:04  nistico
* Imported ball detection from RasterImageProcessor (integrated in MSH by Bernd in New Orleans)
*
* Revision 1.14  2004/05/12 11:38:22  nistico
* RadialDistortionCorrectionFast is now static
*
* Revision 1.13  2004/05/12 09:12:40  schumann
* removed gcc error
*
* Revision 1.12  2004/05/07 15:16:25  nistico
* All geometry calculations are making use of intrinsic functions.
* I updated most of the images processors to make correct use of this.
* Please test if there are any problems, because i'm going to remove the
* old code soon.
*
* Revision 1.11  2004/04/18 18:14:53  nistico
* USE_INTRINSIC layout removed.
* All functions properly replicated in intrinsic version.
* However, image processor (MSH2004) making use of them get distorted visualization
* of percepts, because drawing functions use the old parameters.
* It has to be decided wheter to fully move to intrinsic, or discard it.
*
* Revision 1.10  2004/04/09 21:32:33  roefer
* USE_INTRINSIC is inconsistent in the simulator, so it is switched off
* Do we really want to use it on the robot?
*
* Revision 1.9  2004/04/08 15:33:08  wachter
* GT04 checkin of Microsoft-Hellounds
*
* Revision 1.8  2004/04/07 13:00:46  risler
* ddd checkin after go04 - second part
*
* Revision 1.2  2004/03/30 20:21:45  park
* changed PixeldeLine to work with BW Images
*
* Revision 1.1.1.1  2004/03/29 08:28:42  Administrator
* initial transfer from tamara
*
* Revision 1.7  2004/03/09 11:35:23  nistico
* - Sorry, conditional compilation switch is really located (as it should be) in Geometry.h, not CameraInfo.h
*
* Revision 1.6  2004/03/09 11:33:21  nistico
* - Intrinsic parameters based measurements can now be triggered through a single conditional compilation
* switch located in CameraInfo.h
* - Implemented fast (look-up table based) radial distortion correction
*
* Revision 1.5  2004/02/10 10:48:05  nistico
* Introduced Intrinsic camera parameters to perform geometric calculations (distance, angle, size...) without opening angle
* Implemented radial distortion correction function
* Implemented ball distance calculation based on size and intrinsic params (potentially more stable)
* To Be Done: calculate intrinsic params for ERS7, as soon as we get our puppies back
*
* Revision 1.4  2004/01/28 08:27:40  dueffert
* doxygen bug fixed
*
* Revision 1.3  2003/12/15 11:47:00  juengel
* Introduced CameraInfo
*
* Revision 1.2  2003/11/10 11:34:05  juengel
* Added class SetOfPoints.
*
* Revision 1.1  2003/10/07 10:13:24  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.7  2003/09/26 15:28:10  juengel
* Renamed DataTypes to representations.
*
* Revision 1.6  2003/09/26 11:40:40  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.5  2003/08/18 11:50:21  juengel
* Added calculateLineSize.
*
* Revision 1.4  2003/08/09 14:53:10  dueffert
* files and docu beautified
*
* Revision 1.3  2003/07/29 12:40:00  juengel
* Added calculateHorizon
*
* Revision 1.2  2003/07/28 10:02:35  dueffert
* access implications of member methods corrected
*
* Revision 1.1.1.1  2003/07/02 09:40:28  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.23  2003/06/22 17:25:57  dueffert
* getIntersectionPointsOfLineAndRectangle added
*
* Revision 1.22  2003/06/22 17:09:58  juengel
* Added int version of function "distance".
*
* Revision 1.21  2003/06/15 14:26:44  jhoffman
* + moved "relative2FieldCoord" to Geometry
* + added member function to ballposition to calculate the propagated position and speed for a given time
* + propagated speed and time calculation using exponential decay instead of using an iterative calculation
* + in motion you can now use propageted ball pos at 125 Hz rather then the framerate determined by cognition
*
* Revision 1.20  2003/06/12 16:54:09  juengel
* Added getDistanceBySize and getSizeByDistance.
*
* Revision 1.19  2003/06/10 17:59:04  juengel
* Added constructor Line(Vector2<int> base, Vector2<int> direction)
*
* Revision 1.18  2003/04/16 07:00:17  roefer
* Bremen GO checkin
*
* Revision 1.17  2003/04/15 15:52:07  risler
* DDD GO 2003 code integrated
*
* Revision 1.16  2003/03/26 17:46:00  max
* added getDistanceToEdge
* optimized getDistanceToLine
*
* Revision 1.16  2003/04/01 16:51:09  roefer
* CameraMatrix contains validity and is calculated in Geometry
*
* Revision 1.15  2003/03/10 18:19:54  dueffert
* assignments replaced with initializations
*
* Revision 1.14  2003/02/20 15:31:52  dueffert
* docu added
*
* Revision 1.13  2003/01/19 11:38:07  juengel
* int versions of clipLineWithQuadrangle and isPointInsideRectangle
*
* Revision 1.12  2003/01/15 08:20:21  juengel
* Added "int" - version of getIntersectionOfLines.
* Added struct PixeledLine.
*
* Revision 1.11  2003/01/13 18:24:43  juengel
* Comment corrected
*
* Revision 1.10  2002/12/16 12:03:51  dueffert
* doxygen bug fixed
*
* Revision 1.9  2002/12/14 19:45:54  roefer
* paintLinesPerceptToImageView added
*
* Revision 1.8  2002/12/04 12:24:35  juengel
* Changed parameter "pointOnField" of method calculatePointOnField() from Vector2<double> to Vector2<int>.
*
* Revision 1.7  2002/12/02 12:02:10  juengel
* Finished cohen sutherland line clipping.
*
* Revision 1.6  2002/12/01 17:37:45  juengel
* Worked at clipping functions.
*
* Revision 1.5  2002/11/28 18:53:52  juengel
* RadarViewer3D shows images projected on ground.
*
* Revision 1.4  2002/11/19 15:43:04  dueffert
* doxygen comments corrected
*
* Revision 1.3  2002/09/22 18:40:52  risler
* added new math functions, removed GTMath library
*
* Revision 1.2  2002/09/22 13:10:50  risler
* new Math headers added
*
* Revision 1.1  2002/09/22 09:15:34  risler
* Geometry.h moved to directory Math
*
* Revision 1.1  2002/09/10 15:53:58  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.4  2002/06/13 18:22:10  dueffert
* ray intersection added
*
* Revision 1.3  2002/06/07 12:11:09  risler
* moved max min to GTMath
*
* Revision 1.2  2002/05/14 12:39:04  dueffert
* corrected some documentation mistakes
*
* Revision 1.1.1.1  2002/05/10 12:40:33  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.14  2002/04/25 14:50:37  kallnik
* changed double/float to double
* added several #include GTMath
*
* PLEASE use double
*
* Revision 1.13  2002/04/16 15:44:04  dueffert
* vectorTo(Pose, Vector) added
*
* Revision 1.2  2002/04/11 14:20:04  dueffert
* dribble skill filled
*
* Revision 1.12  2002/04/09 11:18:03  loetzsch
* added min and max
*
* Revision 1.11  2002/04/08 19:53:14  juengel
* Drawing of percept collections in images added.
*
* Revision 1.10  2002/04/06 02:30:55  loetzsch
* added angleTo and distanceTo functions
*
* Revision 1.9  2002/04/04 18:43:31  juengel
* Disatance and distance from point to line added.
*
* Revision 1.8  2002/03/29 14:55:48  juengel
* "horizon aligned grid" started.
*
* Revision 1.7  2002/03/24 17:47:02  juengel
* LinesPercept defined and LinesPerceptor added.
*
* Revision 1.6  2002/03/18 09:45:48  kallnik
* GTMathTable updated
* GTMathConfig updated
* several doubles changed in GTMathValue
*
* Revision 1.5  2002/02/12 22:42:42  juengel
* ImageProcessorTester improved.
*
* Revision 1.4  2002/02/11 00:56:36  loetzsch
* added a constructor for class Circle
*
* Revision 1.3  2002/01/30 01:08:56  dueffert
* fixed endless-compile-bug caused by up/downcase difference MathLib vs. Mathlib, makefile should be caseUNsensitive (in such cases) now
*
* Revision 1.2  2002/01/23 12:34:18  juengel
* Kommentare eingefgt
*
* Revision 1.1  2002/01/22 14:54:47  juengel
* Geometry eingefhrt
*
*
*/
