/**
 * @file Matrix.cpp
 * Implements RotationMatrix
 *
 * @author <a href="mailto:martin.kallnik@gmx.de" > Martin Kallnik</a>
 * @author Max Risler
 */

#include "Matrix.h"
#include "Common.h"

Matrix_6x6::Matrix_6x6():
c00(1),c01(0),c02(0),c03(0),c04(0),c05(0),
c10(0),c11(1),c12(0),c13(0),c14(0),c15(0),
c20(0),c21(0),c22(1),c23(0),c24(0),c25(0),
c30(0),c31(0),c32(0),c33(1),c34(0),c35(0),
c40(0),c41(0),c42(0),c43(0),c44(1),c45(0),
c50(0),c51(0),c52(0),c53(0),c54(0),c55(1)
{
}

Matrix_6x6::Matrix_6x6(
                       double c00, double c01, double c02, double c03, double c04, double c05,
                       double c10, double c11, double c12, double c13, double c14, double c15,
                       double c20, double c21, double c22, double c23, double c24, double c25,
                       double c30, double c31, double c32, double c33, double c34, double c35,
                       double c40, double c41, double c42, double c43, double c44, double c45,
                       double c50, double c51, double c52, double c53, double c54, double c55):
c00(c00),c01(c01),c02(c02),c03(c03),c04(c04),c05(c05),
c10(c10),c11(c11),c12(c12),c13(c13),c14(c14),c15(c15),
c20(c20),c21(c21),c22(c22),c23(c23),c24(c24),c25(c25),
c30(c30),c31(c31),c32(c32),c33(c33),c34(c34),c35(c35),
c40(c40),c41(c41),c42(c42),c43(c43),c44(c44),c45(c45),
c50(c50),c51(c51),c52(c52),c53(c53),c54(c54),c55(c55)
{
}

Matrix_6x6& Matrix_6x6::operator=(const Matrix_6x6& o)
{
  c00=o.c00; c01=o.c01; c02=o.c02; c03=o.c03; c04=o.c04; c05=o.c05; 
  c10=o.c10; c11=o.c11; c12=o.c12; c13=o.c13; c14=o.c14; c15=o.c15; 
  c20=o.c20; c21=o.c21; c22=o.c22; c23=o.c23; c24=o.c24; c25=o.c25; 
  c30=o.c30; c31=o.c31; c32=o.c32; c33=o.c33; c34=o.c34; c35=o.c35; 
  c40=o.c40; c41=o.c41; c42=o.c42; c43=o.c43; c44=o.c44; c45=o.c45; 
  c50=o.c50; c51=o.c51; c52=o.c52; c53=o.c53; c54=o.c54; c55=o.c55;
  return *this;
}


Matrix_6x6 Matrix_6x6::operator-(const Matrix_6x6& o) const
{
  return Matrix_6x6(
    c00-o.c00, c01-o.c01, c02-o.c02, c03-o.c03, c04-o.c04, c05-o.c05,
    c10-o.c10, c11-o.c11, c12-o.c12, c13-o.c13, c14-o.c14, c15-o.c15,
    c20-o.c20, c21-o.c21, c22-o.c22, c23-o.c23, c24-o.c24, c25-o.c25,
    c30-o.c30, c31-o.c31, c32-o.c32, c33-o.c33, c34-o.c34, c35-o.c35,
    c40-o.c40, c41-o.c41, c42-o.c42, c43-o.c43, c44-o.c44, c45-o.c45,
    c50-o.c50, c51-o.c51, c52-o.c52, c53-o.c53, c54-o.c54, c55-o.c55
  );
}

Matrix_6x6 Matrix_6x6::operator*(const Matrix_6x6& o) const
{
  return Matrix_6x6(
    c00*o.c00+c01*o.c10+c02*o.c20+c03*o.c30+c04*o.c40+c05*o.c50,
    c00*o.c01+c01*o.c11+c02*o.c21+c03*o.c31+c04*o.c41+c05*o.c51,
    c00*o.c02+c01*o.c12+c02*o.c22+c03*o.c32+c04*o.c42+c05*o.c52,
    c00*o.c03+c01*o.c13+c02*o.c23+c03*o.c33+c04*o.c43+c05*o.c53,
    c00*o.c04+c01*o.c14+c02*o.c24+c03*o.c34+c04*o.c44+c05*o.c54,
    c00*o.c05+c01*o.c15+c02*o.c25+c03*o.c35+c04*o.c45+c05*o.c55,
    
    c10*o.c00+c11*o.c10+c12*o.c20+c13*o.c30+c14*o.c40+c15*o.c50,
    c10*o.c01+c11*o.c11+c12*o.c21+c13*o.c31+c14*o.c41+c15*o.c51,
    c10*o.c02+c11*o.c12+c12*o.c22+c13*o.c32+c14*o.c42+c15*o.c52,
    c10*o.c03+c11*o.c13+c12*o.c23+c13*o.c33+c14*o.c43+c15*o.c53,
    c10*o.c04+c11*o.c14+c12*o.c24+c13*o.c34+c14*o.c44+c15*o.c54,
    c10*o.c05+c11*o.c15+c12*o.c25+c13*o.c35+c14*o.c45+c15*o.c55,
    
    c20*o.c00+c21*o.c10+c22*o.c20+c23*o.c30+c24*o.c40+c25*o.c50,
    c20*o.c01+c21*o.c11+c22*o.c21+c23*o.c31+c24*o.c41+c25*o.c51,
    c20*o.c02+c21*o.c12+c22*o.c22+c23*o.c32+c24*o.c42+c25*o.c52,
    c20*o.c03+c21*o.c13+c22*o.c23+c23*o.c33+c24*o.c43+c25*o.c53,
    c20*o.c04+c21*o.c14+c22*o.c24+c23*o.c34+c24*o.c44+c25*o.c54,
    c20*o.c05+c21*o.c15+c22*o.c25+c23*o.c35+c24*o.c45+c25*o.c55,
    
    c30*o.c00+c31*o.c10+c32*o.c20+c33*o.c30+c34*o.c40+c35*o.c50,
    c30*o.c01+c31*o.c11+c32*o.c21+c33*o.c31+c34*o.c41+c35*o.c51,
    c30*o.c02+c31*o.c12+c32*o.c22+c33*o.c32+c34*o.c42+c35*o.c52,
    c30*o.c03+c31*o.c13+c32*o.c23+c33*o.c33+c34*o.c43+c35*o.c53,
    c30*o.c04+c31*o.c14+c32*o.c24+c33*o.c34+c34*o.c44+c35*o.c54,
    c30*o.c05+c31*o.c15+c32*o.c25+c33*o.c35+c34*o.c45+c35*o.c55,
    
    c40*o.c00+c41*o.c10+c42*o.c20+c43*o.c30+c44*o.c40+c45*o.c50,
    c40*o.c01+c41*o.c11+c42*o.c21+c43*o.c31+c44*o.c41+c45*o.c51,
    c40*o.c02+c41*o.c12+c42*o.c22+c43*o.c32+c44*o.c42+c45*o.c52,
    c40*o.c03+c41*o.c13+c42*o.c23+c43*o.c33+c44*o.c43+c45*o.c53,
    c40*o.c04+c41*o.c14+c42*o.c24+c43*o.c34+c44*o.c44+c45*o.c54,
    c40*o.c05+c41*o.c15+c42*o.c25+c43*o.c35+c44*o.c45+c45*o.c55,
    
    c50*o.c00+c51*o.c10+c52*o.c20+c53*o.c30+c54*o.c40+c55*o.c50,
    c50*o.c01+c51*o.c11+c52*o.c21+c53*o.c31+c54*o.c41+c55*o.c51,
    c50*o.c02+c51*o.c12+c52*o.c22+c53*o.c32+c54*o.c42+c55*o.c52,
    c50*o.c03+c51*o.c13+c52*o.c23+c53*o.c33+c54*o.c43+c55*o.c53,
    c50*o.c04+c51*o.c14+c52*o.c24+c53*o.c34+c54*o.c44+c55*o.c54,
    c50*o.c05+c51*o.c15+c52*o.c25+c53*o.c35+c54*o.c45+c55*o.c55
    );
}

Matrix_6x3 Matrix_6x6::operator*(const Matrix_6x3& o) const
{
  return Matrix_6x3(
    c00*o.c00+c01*o.c10+c02*o.c20+c03*o.c30+c04*o.c40+c05*o.c50,
    c00*o.c01+c01*o.c11+c02*o.c21+c03*o.c31+c04*o.c41+c05*o.c51,
    c00*o.c02+c01*o.c12+c02*o.c22+c03*o.c32+c04*o.c42+c05*o.c52,
    
    c10*o.c00+c11*o.c10+c12*o.c20+c13*o.c30+c14*o.c40+c15*o.c50,
    c10*o.c01+c11*o.c11+c12*o.c21+c13*o.c31+c14*o.c41+c15*o.c51,
    c10*o.c02+c11*o.c12+c12*o.c22+c13*o.c32+c14*o.c42+c15*o.c52,
    
    c20*o.c00+c21*o.c10+c22*o.c20+c23*o.c30+c24*o.c40+c25*o.c50,
    c20*o.c01+c21*o.c11+c22*o.c21+c23*o.c31+c24*o.c41+c25*o.c51,
    c20*o.c02+c21*o.c12+c22*o.c22+c23*o.c32+c24*o.c42+c25*o.c52,
    
    c30*o.c00+c31*o.c10+c32*o.c20+c33*o.c30+c34*o.c40+c35*o.c50,
    c30*o.c01+c31*o.c11+c32*o.c21+c33*o.c31+c34*o.c41+c35*o.c51,
    c30*o.c02+c31*o.c12+c32*o.c22+c33*o.c32+c34*o.c42+c35*o.c52,
    
    c40*o.c00+c41*o.c10+c42*o.c20+c43*o.c30+c44*o.c40+c45*o.c50,
    c40*o.c01+c41*o.c11+c42*o.c21+c43*o.c31+c44*o.c41+c45*o.c51,
    c40*o.c02+c41*o.c12+c42*o.c22+c43*o.c32+c44*o.c42+c45*o.c52,
    
    c50*o.c00+c51*o.c10+c52*o.c20+c53*o.c30+c54*o.c40+c55*o.c50,
    c50*o.c01+c51*o.c11+c52*o.c21+c53*o.c31+c54*o.c41+c55*o.c51,
    c50*o.c02+c51*o.c12+c52*o.c22+c53*o.c32+c54*o.c42+c55*o.c52
    );
}


Matrix_6x1 Matrix_6x6::operator*(const Matrix_6x1& o) const
{
  return Matrix_6x1(
    c00*o.c00+c01*o.c10+c02*o.c20+c03*o.c30+c04*o.c40+c05*o.c50,
    c10*o.c00+c11*o.c10+c12*o.c20+c13*o.c30+c14*o.c40+c15*o.c50,
    c20*o.c00+c21*o.c10+c22*o.c20+c23*o.c30+c24*o.c40+c25*o.c50,
    c30*o.c00+c31*o.c10+c32*o.c20+c33*o.c30+c34*o.c40+c35*o.c50,
    c40*o.c00+c41*o.c10+c42*o.c20+c43*o.c30+c44*o.c40+c45*o.c50,
    c50*o.c00+c51*o.c10+c52*o.c20+c53*o.c30+c54*o.c40+c55*o.c50
    );
}

Matrix_6x6 Matrix_6x6::transpose() const
{
  return Matrix_6x6(
    c00,c10,c20,c30,c40,c50,
    c01,c11,c21,c31,c41,c51,
    c02,c12,c22,c32,c42,c52,
    c03,c13,c23,c33,c43,c53,
    c04,c14,c24,c34,c44,c54,
    c05,c15,c25,c35,c45,c55
    );
}


Matrix_6x1::Matrix_6x1():
c00(0),
c10(0),
c20(0),
c30(0),
c40(0),
c50(0)
{
}

Matrix_6x1::Matrix_6x1(
                       double c00,
                       double c10,
                       double c20,
                       double c30,
                       double c40,
                       double c50):
c00(c00),
c10(c10),
c20(c20),
c30(c30),
c40(c40),
c50(c50)
{
}

Matrix_6x1& Matrix_6x1::operator=(const Matrix_6x1& o)
{
  c00=o.c00;
  c10=o.c10;
  c20=o.c20;
  c30=o.c30;
  c40=o.c40;
  c50=o.c50;
  return *this;
}

Matrix_6x1 Matrix_6x1::operator+(const Matrix_6x1& o) const
{
  return Matrix_6x1(
    c00+o.c00,
    c10+o.c10,
    c20+o.c20,
    c30+o.c30,
    c40+o.c40,
    c50+o.c50
    );
}

Matrix_6x3::Matrix_6x3():
c00(0),c01(0),c02(0),
c10(0),c11(0),c12(0),
c20(0),c21(0),c22(0),
c30(0),c31(0),c32(0),
c40(0),c41(0),c42(0),
c50(0),c51(0),c52(0)
{
}

Matrix_6x3::Matrix_6x3(
                       double c00, double c01, double c02,
                       double c10, double c11, double c12,
                       double c20, double c21, double c22,
                       double c30, double c31, double c32,
                       double c40, double c41, double c42,
                       double c50, double c51, double c52):
c00(c00),c01(c01),c02(c02),
c10(c10),c11(c11),c12(c12),
c20(c20),c21(c21),c22(c22),
c30(c30),c31(c31),c32(c32),
c40(c40),c41(c41),c42(c42),
c50(c50),c51(c51),c52(c52)
{
}

Matrix_6x3& Matrix_6x3::operator=(const Matrix_6x3& o)
{
  c00=o.c00; c01=o.c01; c02=o.c02;
  c10=o.c10; c11=o.c11; c12=o.c12;
  c20=o.c20; c21=o.c21; c22=o.c22;
  c30=o.c30; c31=o.c31; c32=o.c32;
  c40=o.c40; c41=o.c41; c42=o.c42;
  c50=o.c50; c51=o.c51; c52=o.c52;
  return *this;
}

Matrix_6x6 Matrix_6x3::operator*(const Matrix_3x6& o) const
{
  return Matrix_6x6(
    c00*o.c00+c01*o.c10+c02*o.c20,
    c00*o.c01+c01*o.c11+c02*o.c21,
    c00*o.c02+c01*o.c12+c02*o.c22,
    c00*o.c03+c01*o.c13+c02*o.c23,
    c00*o.c04+c01*o.c14+c02*o.c24,
    c00*o.c05+c01*o.c15+c02*o.c25,
    
    c10*o.c00+c11*o.c10+c12*o.c20,
    c10*o.c01+c11*o.c11+c12*o.c21,
    c10*o.c02+c11*o.c12+c12*o.c22,
    c10*o.c03+c11*o.c13+c12*o.c23,
    c10*o.c04+c11*o.c14+c12*o.c24,
    c10*o.c05+c11*o.c15+c12*o.c25,
    
    c20*o.c00+c21*o.c10+c22*o.c20,
    c20*o.c01+c21*o.c11+c22*o.c21,
    c20*o.c02+c21*o.c12+c22*o.c22,
    c20*o.c03+c21*o.c13+c22*o.c23,
    c20*o.c04+c21*o.c14+c22*o.c24,
    c20*o.c05+c21*o.c15+c22*o.c25,
    
    c30*o.c00+c31*o.c10+c32*o.c20,
    c30*o.c01+c31*o.c11+c32*o.c21,
    c30*o.c02+c31*o.c12+c32*o.c22,
    c30*o.c03+c31*o.c13+c32*o.c23,
    c30*o.c04+c31*o.c14+c32*o.c24,
    c30*o.c05+c31*o.c15+c32*o.c25,
    
    c40*o.c00+c41*o.c10+c42*o.c20,
    c40*o.c01+c41*o.c11+c42*o.c21,
    c40*o.c02+c41*o.c12+c42*o.c22,
    c40*o.c03+c41*o.c13+c42*o.c23,
    c40*o.c04+c41*o.c14+c42*o.c24,
    c40*o.c05+c41*o.c15+c42*o.c25,
    
    c50*o.c00+c51*o.c10+c52*o.c20,
    c50*o.c01+c51*o.c11+c52*o.c21,
    c50*o.c02+c51*o.c12+c52*o.c22,
    c50*o.c03+c51*o.c13+c52*o.c23,
    c50*o.c04+c51*o.c14+c52*o.c24,
    c50*o.c05+c51*o.c15+c52*o.c25
    );
}

Matrix_6x3 Matrix_6x3::operator*(const Matrix_3x3& o) const
{
  return Matrix_6x3(
    c00*o.c00+c01*o.c10+c02*o.c20,
    c00*o.c01+c01*o.c11+c02*o.c21,
    c00*o.c02+c01*o.c12+c02*o.c22,
    
    c10*o.c00+c11*o.c10+c12*o.c20,
    c10*o.c01+c11*o.c11+c12*o.c21,
    c10*o.c02+c11*o.c12+c12*o.c22,
    
    c20*o.c00+c21*o.c10+c22*o.c20,
    c20*o.c01+c21*o.c11+c22*o.c21,
    c20*o.c02+c21*o.c12+c22*o.c22,
    
    c30*o.c00+c31*o.c10+c32*o.c20,
    c30*o.c01+c31*o.c11+c32*o.c21,
    c30*o.c02+c31*o.c12+c32*o.c22,
    
    c40*o.c00+c41*o.c10+c42*o.c20,
    c40*o.c01+c41*o.c11+c42*o.c21,
    c40*o.c02+c41*o.c12+c42*o.c22,
    
    c50*o.c00+c51*o.c10+c52*o.c20,
    c50*o.c01+c51*o.c11+c52*o.c21,
    c50*o.c02+c51*o.c12+c52*o.c22
    );
}

Matrix_6x1 Matrix_6x3::operator*(const Vector3<double>& o) const
{
  return Matrix_6x1(
    c00*o.x+c01*o.y+c02*o.z,
    c10*o.x+c11*o.y+c12*o.z,
    c20*o.x+c21*o.y+c22*o.z,
    c30*o.x+c31*o.y+c32*o.z,
    c40*o.x+c41*o.y+c42*o.z,
    c50*o.x+c51*o.y+c52*o.z
    );
}


Matrix_3x6 Matrix_6x3::transpose() const
{
  return Matrix_3x6(
    c00,c10,c20,c30,c40,c50,
    c01,c11,c21,c31,c41,c51,
    c02,c12,c22,c32,c42,c52
    );
}


Matrix_3x6::Matrix_3x6(
                       double c00, double c01, double c02, double c03, double c04, double c05,
                       double c10, double c11, double c12, double c13, double c14, double c15,
                       double c20, double c21, double c22, double c23, double c24, double c25):
c00(c00),c01(c01),c02(c02),c03(c03),c04(c04),c05(c05),
c10(c10),c11(c11),c12(c12),c13(c13),c14(c14),c15(c15),
c20(c20),c21(c21),c22(c22),c23(c23),c24(c24),c25(c25)
{
}

Matrix_3x6& Matrix_3x6::operator=(const Matrix_3x6& o)
{
  c00=o.c00; c01=o.c01; c02=o.c02; c03=o.c03; c04=o.c04; c05=o.c05; 
  c10=o.c10; c11=o.c11; c12=o.c12; c13=o.c13; c14=o.c14; c15=o.c15; 
  c20=o.c20; c21=o.c21; c22=o.c22; c23=o.c23; c24=o.c24; c25=o.c25; 
  return *this;
}

Vector3<double> Matrix_3x6::operator*(const Matrix_6x1& o) const
{
  return Vector3<double>(
    c00*o.c00+c01*o.c10+c02*o.c20+c03*o.c30+c04*o.c40+c05*o.c50,
    c10*o.c00+c11*o.c10+c12*o.c20+c13*o.c30+c14*o.c40+c15*o.c50,
    c20*o.c00+c21*o.c10+c22*o.c20+c23*o.c30+c24*o.c40+c25*o.c50
    );
}

Matrix_3x3 Matrix_3x6::operator*(const Matrix_6x3& o) const
{
  return Matrix_3x3(
    c00*o.c00+c01*o.c10+c02*o.c20+c03*o.c30+c04*o.c40+c05*o.c50,
    c00*o.c01+c01*o.c11+c02*o.c21+c03*o.c31+c04*o.c41+c05*o.c51,
    c00*o.c02+c01*o.c12+c02*o.c22+c03*o.c32+c04*o.c42+c05*o.c52,
    
    c10*o.c00+c11*o.c10+c12*o.c20+c13*o.c30+c14*o.c40+c15*o.c50,
    c10*o.c01+c11*o.c11+c12*o.c21+c13*o.c31+c14*o.c41+c15*o.c51,
    c10*o.c02+c11*o.c12+c12*o.c22+c13*o.c32+c14*o.c42+c15*o.c52,
    
    c20*o.c00+c21*o.c10+c22*o.c20+c23*o.c30+c24*o.c40+c25*o.c50,
    c20*o.c01+c21*o.c11+c22*o.c21+c23*o.c31+c24*o.c41+c25*o.c51,
    c20*o.c02+c21*o.c12+c22*o.c22+c23*o.c32+c24*o.c42+c25*o.c52
    );
}

Matrix_3x6 Matrix_3x6::operator*(const Matrix_6x6& o) const
{
  return Matrix_3x6(
    c00*o.c00+c01*o.c10+c02*o.c20+c03*o.c30+c04*o.c40+c05*o.c50,
    c00*o.c01+c01*o.c11+c02*o.c21+c03*o.c31+c04*o.c41+c05*o.c51,
    c00*o.c02+c01*o.c12+c02*o.c22+c03*o.c32+c04*o.c42+c05*o.c52,
    c00*o.c03+c01*o.c13+c02*o.c23+c03*o.c33+c04*o.c43+c05*o.c53,
    c00*o.c04+c01*o.c14+c02*o.c24+c03*o.c34+c04*o.c44+c05*o.c54,
    c00*o.c05+c01*o.c15+c02*o.c25+c03*o.c35+c04*o.c45+c05*o.c55,
    
    c10*o.c00+c11*o.c10+c12*o.c20+c13*o.c30+c14*o.c40+c15*o.c50,
    c10*o.c01+c11*o.c11+c12*o.c21+c13*o.c31+c14*o.c41+c15*o.c51,
    c10*o.c02+c11*o.c12+c12*o.c22+c13*o.c32+c14*o.c42+c15*o.c52,
    c10*o.c03+c11*o.c13+c12*o.c23+c13*o.c33+c14*o.c43+c15*o.c53,
    c10*o.c04+c11*o.c14+c12*o.c24+c13*o.c34+c14*o.c44+c15*o.c54,
    c10*o.c05+c11*o.c15+c12*o.c25+c13*o.c35+c14*o.c45+c15*o.c55,
    
    c20*o.c00+c21*o.c10+c22*o.c20+c23*o.c30+c24*o.c40+c25*o.c50,
    c20*o.c01+c21*o.c11+c22*o.c21+c23*o.c31+c24*o.c41+c25*o.c51,
    c20*o.c02+c21*o.c12+c22*o.c22+c23*o.c32+c24*o.c42+c25*o.c52,
    c20*o.c03+c21*o.c13+c22*o.c23+c23*o.c33+c24*o.c43+c25*o.c53,
    c20*o.c04+c21*o.c14+c22*o.c24+c23*o.c34+c24*o.c44+c25*o.c54,
    c20*o.c05+c21*o.c15+c22*o.c25+c23*o.c35+c24*o.c45+c25*o.c55
    );
}

Matrix_6x3 Matrix_3x6::transpose() const
{
  return Matrix_6x3(
    c00,c10,c20,
    c01,c11,c21,
    c02,c12,c22,
    c03,c13,c23,
    c04,c14,c24,
    c05,c15,c25
    );
}


Matrix_3x3::Matrix_3x3():
c00(1),c01(0),c02(0),
c10(0),c11(1),c12(0),
c20(0),c21(0),c22(1)
{
}

Matrix_3x3::Matrix_3x3(
                       double c00, double c01, double c02,
                       double c10, double c11, double c12,
                       double c20, double c21, double c22):
c00(c00),c01(c01),c02(c02),
c10(c10),c11(c11),c12(c12),
c20(c20),c21(c21),c22(c22)
{
}

Matrix_3x3& Matrix_3x3::operator=(const Matrix_3x3& o)
{
  c00=o.c00; c01=o.c01; c02=o.c02;
  c10=o.c10; c11=o.c11; c12=o.c12;
  c20=o.c20; c21=o.c21; c22=o.c22;
  return *this;
}

Matrix_3x3 Matrix_3x3::operator*(const Matrix_3x3& o) const
{
  return Matrix_3x3(
    c00*o.c00+c01*o.c10+c02*o.c20,
    c00*o.c01+c01*o.c11+c02*o.c21,
    c00*o.c02+c01*o.c12+c02*o.c22,
    
    c10*o.c00+c11*o.c10+c12*o.c20,
    c10*o.c01+c11*o.c11+c12*o.c21,
    c10*o.c02+c11*o.c12+c12*o.c22,
    
    c20*o.c00+c21*o.c10+c22*o.c20,
    c20*o.c01+c21*o.c11+c22*o.c21,
    c20*o.c02+c21*o.c12+c22*o.c22
    );
}

Matrix_3x6 Matrix_3x3::operator*(const Matrix_3x6& o) const
{
  return Matrix_3x6(
    c00*o.c00+c01*o.c10+c02*o.c20,
    c00*o.c01+c01*o.c11+c02*o.c21,
    c00*o.c02+c01*o.c12+c02*o.c22,
    c00*o.c03+c01*o.c13+c02*o.c23,
    c00*o.c04+c01*o.c14+c02*o.c24,
    c00*o.c05+c01*o.c15+c02*o.c25,
    
    c10*o.c00+c11*o.c10+c12*o.c20,
    c10*o.c01+c11*o.c11+c12*o.c21,
    c10*o.c02+c11*o.c12+c12*o.c22,
    c10*o.c03+c11*o.c13+c12*o.c23,
    c10*o.c04+c11*o.c14+c12*o.c24,
    c10*o.c05+c11*o.c15+c12*o.c25,
    
    c20*o.c00+c21*o.c10+c22*o.c20,
    c20*o.c01+c21*o.c11+c22*o.c21,
    c20*o.c02+c21*o.c12+c22*o.c22,
    c20*o.c03+c21*o.c13+c22*o.c23,
    c20*o.c04+c21*o.c14+c22*o.c24,
    c20*o.c05+c21*o.c15+c22*o.c25
    );
}

Matrix_3x3 Matrix_3x3::transpose() const
{
  return Matrix_3x3(
    c00,c10,c20,
    c01,c11,c21,
    c02,c12,c22
    );
}

double Matrix_3x3::det() const 
{
  return c00 * (c11*c22 - c12*c21) +
    c01 * (c12*c20 - c10*c22) +
    c02 * (c10*c21 - c11*c20);
}

Matrix_3x3 Matrix_3x3::inverse() const 
{
  double d=det();
  if (d!=0)
  {
    return Matrix_3x3(
      (-c12*c21+c11*c22)/d, (c02*c21-c01*c22)/d, (-c02*c11+c01*c12)/d,
      (c12*c20-c10*c22)/d, (-c02*c20+c00*c22)/d, (c02*c10-c00*c12)/d,
      (-c11*c20+c10*c21)/d, (c01*c20-c00*c21)/d, (-c01*c10+c00*c11)/d
      );
  }
  else
  {
    return Matrix_3x3();
    //@todo error
  }
}


RotationMatrix& RotationMatrix::fromKardanRPY
(const double yaw, const double pitch, const double roll){

  double cy=cos(yaw);
  double sy=sin(yaw);
  double cp=cos(pitch);
  double sp=sin(pitch);
  double cr=cos(roll);
  double sr=sin(roll);

  c[0].x=cr*cp ;
  c[0].y=-sr*cy+cr*sp*sy ;
  c[0].z=sr*sy+cr*sp*cy ;
  c[1].x=sr*cp ;
  c[1].y=cr*cy+sr*sp*sy ;
  c[1].z=-cr*sy+sr*sp*cy ;
  c[2].x=-sp ;
  c[2].y=cp*sy ;
  c[2].z=cp*cy ;

  return *this;
}
			
RotationMatrix& RotationMatrix::rotateX(const double angle)
{
  double c = cos(angle),
         s = sin(angle);
  *this *= RotationMatrix(Vector3<double>(1,0,0),
                          Vector3<double>(0,c,s),
                          Vector3<double>(0,-s,c));
  return *this;
}

RotationMatrix& RotationMatrix::rotateY(const double angle)
{
  double c = cos(angle),
         s = sin(angle);
  *this *= RotationMatrix(Vector3<double>(c,0,-s),
                          Vector3<double>(0,1,0),
                          Vector3<double>(s,0,c));
  return *this;
}

RotationMatrix& RotationMatrix::rotateZ(const double angle)
{
  double c = cos(angle),
         s = sin(angle);
  *this *= RotationMatrix(Vector3<double>(c,s,0),
                          Vector3<double>(-s,c,0),
                          Vector3<double>(0,0,1));
  return *this;
}

double RotationMatrix::getXAngle() const
{
  double h = sqrt(c[2].y * c[2].y + c[2].z * c[2].z);
  return h ? acos(c[2].z / h) * (c[2].y > 0 ? -1 : 1) : 0;
}

double RotationMatrix::getYAngle() const
{
  double h = sqrt(c[0].x * c[0].x + c[0].z * c[0].z);
  return h ? acos(c[0].x / h) * (c[0].z > 0 ? -1 : 1) : 0;
}

double RotationMatrix::getZAngle() const
{
  double h = sqrt(c[0].x * c[0].x + c[0].y * c[0].y);
  return h ? acos(c[0].x / h) * (c[0].y < 0 ? -1 : 1) : 0;
}

template <class V> In& operator>>(In& stream, Matrix3x3<V>& matrix3x3)
{
  stream >> matrix3x3.c[0];
  stream >> matrix3x3.c[1];
  stream >> matrix3x3.c[2];
  return stream;
}

template <class V> Out& operator<<(Out& stream, const Matrix3x3<V>& matrix3x3)
{
  stream << matrix3x3.c[0];
  stream << matrix3x3.c[1];
  stream << matrix3x3.c[2];
  return stream;
}

In& operator>>(In& stream, RotationMatrix& rotationMatrix)
{
  stream >> rotationMatrix.c[0];
  stream >> rotationMatrix.c[1];
  stream >> rotationMatrix.c[2];
  return stream;
}

Out& operator<<(Out& stream, const RotationMatrix& rotationMatrix)
{
  stream << rotationMatrix.c[0];
  stream << rotationMatrix.c[1];
  stream << rotationMatrix.c[2];
  return stream;
}


/*
* Change log :
* 
* $Log: Matrix.cpp,v $
* Revision 1.2  2003/12/02 13:44:55  cesarz
* added streaming operators
*
* Revision 1.1  2003/10/07 10:13:24  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:28  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.4  2002/11/19 15:43:04  dueffert
* doxygen comments corrected
*
* Revision 1.3  2002/11/12 23:00:47  dueffert
* started restore greenhills compatibility
*
* Revision 1.2  2002/09/22 18:40:52  risler
* added new math functions, removed GTMath library
*
* Revision 1.1  2002/09/22 13:10:50  risler
* new Math headers added
*
*
*/
