/** 
* @file PIDsmoothedValue.h
* Contains class PIDsmoothedValue
* that calculates the PIDvalue for a given value
*
* @author <a href="mailto:goehring@informatik.hu-berlin.de">Daniel Goehring</a>
*/


#ifndef __PIDsmoothedValue_h__
#define __PIDsmoothedValue_h__


/**@class PIDsmoothedValue
* Calculation of the PID-value for a given target value,
* with or without boundary checks 
*/

class PIDsmoothedValue 
{
  
  double lastDifference; //previous difference between object and its target
  double lastTargetVal;  //previous target position
  double value;          //value contains the so called PIDvalue to be returned
  double lastValue;      //the previously returned value
  bool	 started;		     //is set true after the first difference is set

  double oldTargetVal;  //last value of target position
  double integralError; //last integralsumme
  double lastTime;	    //last timespot the PID-value was calculated
    
  double p;             //p-Koeffizient
  double i;             //i-Koeffizient
  double d;             //d-Koeffizient
  double min;           //obere Bereichsgrenze
  double max;           //untere Bereichsgrenze
  double maxaction;     //maximale Wertaenderung
  
  
public: 
 
  
/**
* Default constructor contains default vaules and weights.
* Value types are: First result value = 0; P-weight,I-weight,D-weight, 
* Min PID - Value, Max PID - Value, Max PID - value change since recent PID - value.
* 
*/  
  PIDsmoothedValue();
  

/**
* Constructor for user chosen weights and value 
* types as above in the default constructor.
*/
  PIDsmoothedValue(double value, double p, double i, double d, double min = -100, double max = 100.0, double maxaction = 100.0);

/**
* Set the min value
*/
  void setMin(double m);

/**
* Set the max value
*/
  void setMax(double m);

/** set the P-weight */    
  void setWeightP(double p);
/** set the P-weight */    
  void setWeightI(double i);
/** set the P-weight */    
  void setWeightD(double d);
  
/**
* Calculation of the PID-value to return
* thereby it will not just calculate the PID-sum but also ensure that it lies
* within the given boundaries
*/
  double approximateVal(double newTargetVal);
  
  /* 
  * 2do: do these work properly?
  */
  double addOffset(double offset);
  double setTo(double val);
  


/** Discreet and iterative value calculation
*/
  double apprDiscrVal(double targetVal, double time, double maxIteration);



/**  return the - if necessary modified - PID-value */
  double getVal();
  
  
/**
* Calculation of the PID-value to return, just an alternative approach
* without any boundary checks and the D-value is not just calculated by
* the change of the target to object difference but by the object position change.
* During first tests the upper algorithm "approximateValue" showed a better
* convergence behavior that this one and is considered as the original algorithm
*/
  double approximateValNew(double newTargetVal);



/** Calculate the PID value as a continuous function*/
  double apprContVal(double targetVal, double time);


/** get the P-weight */  
  double getWeightP();
/** get the I-weight */
  double getWeightI();
/** get the D-weight */
  double getWeightD();

/** get the Min-border */
  double getMin();
/** get the Max-border */
  double getMax();


/**
* reset all values and the sums necessary for its calculation
*/
  void reset(void );

/**
* reset all values and the sums necessary for its calculation
* and set the value to val
*/
  void resetTo(double val);
};

#endif // __PIDsmoothedValue_h__

