/**
 * @file Vector2.h
 * Contains template class Vector2 of type V
 *
 * @author <a href="mailto:martin.kallnik@gmx.de" > Martin Kallnik</a>
 * @author Max Risler
 */

#ifndef __Vector2_h__
#define __Vector2_h__

#include <math.h>
#include "Tools/Streams/InOut.h"

template <class V> class Matrix2x2;

/** This class represents a 2-vector */
template <class V> class Vector2 {
  public:
  /** The vector values */
  V x,y;

  /** Default constructor. */
  Vector2<V>():x(0),y(0)
  {
  }
  /** Default constructor. */
  Vector2<V>(V x, V y):x(x),y(y)
  {
  }

  /** Assignment operator
  *\param other The other vector that is assigned to this one
  *\return A reference to this object after the assignment.
  */
  Vector2<V>& operator=(const Vector2<V>& other)
  {
    x=other.x;y=other.y;
    return *this;
  }

  /** Copy constructor
  *\param other The other vector that is copied to this one
  */
  Vector2<V>(const Vector2<V>& other) {*this = other;}

  /** Addition of another vector to this one.
  *\param other The other vector that will be added to this one
  *\return A reference to this object after the calculation.
  */
  Vector2<V>& operator+=(const Vector2<V>& other)
  {
    x += other.x;
    y += other.y;
    return *this;
  }

  /** Substraction of this vector from another one.
  *\param other The other vector this one will be substracted from
  *\return A reference to this object after the calculation.
  */
  Vector2<V>& operator-=(const Vector2<V>& other)
  {
    x -= other.x;
    y -= other.y;
    return *this;
  }

  /** Multiplication of this vector by a factor.
  *\param factor The factor this vector is multiplied by
  *\return A reference to this object after the calculation.
  */
  Vector2<V>& operator*=(const V& factor)
  {
    x *= factor;
    y *= factor;
    return *this;
  }

  /** Division of this vector by a factor.
  *\param factor The factor this vector is divided by
  *\return A reference to this object after the calculation.
  */
  Vector2<V>& operator/=(const V& factor)
  {
    if (factor == 0) return *this;
    x /= factor;
    y /= factor;
    return *this;
  }

  /** Addition of another vector to this one.
  *\param other The other vector that will be added to this one
  *\return A new object that contains the result of the calculation.
  */
  Vector2<V> operator+(const Vector2<V>& other) const
    {return Vector2<V>(*this) += other;}

  /** Subtraction of another vector to this one.
  *\param other The other vector that will be added to this one
  *\return A new object that contains the result of the calculation.
  */
  Vector2<V> operator-(const Vector2<V>& other) const
    {return Vector2<V>(*this) -= other;}

  /** Negation of this vector.
  *\return A new object that contains the result of the calculation.
  */
  Vector2<V> operator-() const
    {return Vector2<V>() -= *this;}

  /** Inner product of this vector and another one.
  *\param other The other vector this one will be multiplied by
  *\return The inner product.
  */
  V operator*(const Vector2<V>& other) const
  {
    return x * other.x + y * other.y;
  }

  /** Multiplication of this vector by a factor.
  *\param factor The factor this vector is multiplied by
  *\return A new object that contains the result of the calculation.
  */
  Vector2<V> operator*(const V& factor) const
    {return Vector2<V>(*this) *= factor;}

  /** Multiplication of this vector by a matrix.
  *\param mat The matrix this vector is multiplied by
  *\return A new object that contains the result of the calculation.
  */
  Vector2<V> operator*(const Matrix2x2<V>& mat) const;

  /** Division of this vector by a factor.
  *
  *\param factor The factor this vector is divided by
  *\return A new object that contains the result of the calculation.
  */
  Vector2<V> operator/(const V& factor) const
    {return Vector2<V>(*this) /= factor;}

  /** Comparison of another vector with this one.
  *\param other The other vector that will be compared to this one
  *\return Whether the two vectors are equal.
  */
  bool operator==(const Vector2<V>& other) const
  {
    return (x==other.x && y==other.y);
  }

  /** Comparison of another vector with this one.
  *\param other The other vector that will be compared to this one.
  *\return Whether the two vectors are unequal.
  */
  bool operator!=(const Vector2<V>& other) const
    {return !(*this == other);}

  /** Calculation of the length of this vector.
  *\return The length.
  */
  V abs() const
  {return (V) sqrt(((double)x)*x+((double)y)*y);}

  /** normalize this vector.
  *\param len The length, the vector should be normalized to, default=1.
  *\return the normalized vector.
  */
  Vector2<V> normalize(V len)
  {
    if (abs() == 0) return *this;
    return *this = (*this * len) / abs();
   }

  /** normalize this vector.
  *\return the normalized vector.
  */
  Vector2<V> normalize()
  {
    if (abs() == 0) return *this;
    return *this /= abs();
  }

  /** transpose this vector.
  *\return the transposed vector.
  */
  Vector2<V> transpose()
  { V buffer = x;
    x = y;
    y = buffer;
    return *this;
  }

  /** the vector is rotated left by 90 degrees.
  *\return the rotated vector.
  */
  Vector2<V> rotateLeft()
  { V buffer = -y;
    y = x;
    x = buffer;
    return *this;
  }

  /** the vector is rotated right by 90 degrees.
  *\return the rotated vector.
  */
  Vector2<V> rotateRight()
  { V buffer = -x;
    x = y;
    y = buffer;
    return *this;
  }

  /** Calculation of the angle of this vector */
  double angle() const
  {return atan2((double)y,(double)x);}
};

/**
* Streaming operator that reads a Vector2<V> from a stream.
* @param stream The stream from which is read.
* @param vector2 The Vector2<V> object.
* @return The stream.
*/ 
template <class V> In& operator>>(In& stream, Vector2<V>& vector2)
{
  stream >> vector2.x;
  stream >> vector2.y;
  return stream;
}

/**
* Streaming operator that writes a Vector2<V> to a stream.
* @param stream The stream to write on.
* @param vector2 The Vector2<V> object.
* @return The stream.
*/ 
template <class V> Out& operator<<(Out& stream, const Vector2<V>& vector2)
{
  stream << vector2.x;
  stream << vector2.y;
  return stream;
}

#endif // __Vector2_h__

/*
* Change log :
*
* $Log: Vector2.h,v $
* Revision 1.3  2004/01/25 14:25:17  roefer
* Missing include added
*
* Revision 1.2  2003/12/02 13:46:00  cesarz
* - added streaming operators
* - added functions rotateLeft() and rotateRight()
*
* Revision 1.1  2003/10/07 10:13:24  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:28  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.10  2003/04/15 15:52:07  risler
* DDD GO 2003 code integrated
*
* Revision 1.11  2003/04/09 14:50:32  kallnik
* error in normalize(V len) fixed. (error if V is an int)
*
* Revision 1.10  2003/04/01 17:46:05  dthomas
* added: negation operator
*
* Revision 1.9  2003/03/22 00:10:41  dueffert
* abs() now works in larger range
*
* Revision 1.8  2003/03/11 09:01:24  dueffert
* Greenhills compilability restored
*
* Revision 1.7  2003/03/10 18:22:38  dueffert
* assignments replaced with initializations
*
* Revision 1.6  2003/02/17 10:40:05  dueffert
* greenhills backport
*
* Revision 1.5  2002/11/19 15:43:04  dueffert
* doxygen comments corrected
*
* Revision 1.4  2002/11/12 23:00:47  dueffert
* started restore greenhills compatibility
*
* Revision 1.3  2002/10/14 13:14:25  dueffert
* doxygen comments corrected
*
* Revision 1.2  2002/09/22 18:40:52  risler
* added new math functions, removed GTMath library
*
* Revision 1.1  2002/09/22 13:10:50  risler
* new Math headers added
*
*
*/
