/**
 * @file Vector3.h
 * Contains template class Vector3 of type V
 *
 * @author <a href="mailto:martin.kallnik@gmx.de" > Martin Kallnik</a>
 * @author Max Risler
 */

#ifndef __Vector3_h__
#define __Vector3_h__

#include "Tools/Streams/InOut.h"

/** This class represents a 3-vector */
template <class V> class Vector3 {
  public:
  /** The vector values */
  V x,y,z;

  /** Default constructor 4 gcc. */
  Vector3<V>():x(0),y(0),z(0)
  {
  }

  /** Default constructor. */
  Vector3<V>(V x, V y, V z):x(x),y(y),z(z)
  {
  }

  /** Assignment operator
  *\param other The other vector that is assigned to this one
  *\return A reference to this object after the assignment.
  */
  Vector3<V>& operator=(const Vector3<V>& other) 
  {
    x = other.x;
    y = other.y;
    z = other.z;
    return *this;
  }

  /** Copy constructor
  *\param other The other vector that is copied to this one
  */
  Vector3<V>(const Vector3<V>& other) {*this = other;}

  /** Addition of another vector to this one. 
  *\param other The other vector that will be added to this one
  *\return A reference to this object after the calculation.
  */
  Vector3<V>& operator+=(const Vector3<V>& other)
  {
    x += other.x;
    y += other.y;
    z += other.z;
    return *this;
  }

  /** Substraction of this vector from another one.
  *\param other The other vector this one will be substracted from 
  *\return A reference to this object after the calculation.
  */
  Vector3<V>& operator-=(const Vector3<V>& other)
  {
    x -= other.x;
    y -= other.y;
    z -= other.z;
    return *this;
  }

  /** Multiplication of this vector by a factor.
  *\param factor The factor this vector is multiplied by 
  *\return A reference to this object after the calculation.
  */
  Vector3<V>& operator*=(const V& factor)
  {
    x *= factor;
    y *= factor;
    z *= factor;
    return *this;
  }

  /** Division of this vector by a factor.
  *\param factor The factor this vector is divided by 
  *\return A reference to this object after the calculation.
  */
  Vector3<V>& operator/=(const V& factor)
  {
    if (factor == 0) return *this;
    x /= factor;
    y /= factor;
    z /= factor;
    return *this;
  }

  /** Addition of another vector to this one.
  *\param other The other vector that will be added to this one
  *\return A new object that contains the result of the calculation.
  */
  Vector3<V> operator+(const Vector3<V>& other) const
    {return Vector3<V>(*this) += other;}

  /** Subtraction of another vector to this one.
  *\param other The other vector that will be added to this one
  *\return A new object that contains the result of the calculation.
  */
  Vector3<V> operator-(const Vector3<V>& other) const
    {return Vector3<V>(*this) -= other;}

  /** Inner product of this vector and another one.
  *\param other The other vector this one will be multiplied by 
  *\return The inner product.
  */
  V operator*(const Vector3<V>& other) const
  {
    return (x*other.x + y*other.y + z*other.z);
  }

  /** Multiplication of this vector by a factor.
  *\param factor The factor this vector is multiplied by 
  *\return A new object that contains the result of the calculation.
  */
  Vector3<V> operator*(const V& factor) const
    {return Vector3<V>(*this) *= factor;}

  /** Division of this vector by a factor.
  *
  *\param factor The factor this vector is divided by 
  *\return A new object that contains the result of the calculation.
  */
  Vector3<V> operator/(const V& factor) const
    {return Vector3<V>(*this) /= factor;}

  /** Comparison of another vector with this one.
  *\param other The other vector that will be compared to this one
  *\return Whether the two vectors are equal.
  */
  bool operator==(const Vector3<V>& other) const
  {
    return (x==other.x && y==other.y && z==other.z);
  }

  /** Comparison of another vector with this one.
  *\param other The other vector that will be compared to this one
  *\return Whether the two vectors are unequal.
  */
  bool operator!=(const Vector3<V>& other) const
    {return !(*this == other);}

  /** Calculation of the length of this vector.
  *\return The length.
  */
  V abs() const 
  {return (V) sqrt(double((*this) * (*this)));}

  /** Crossproduct of this vector and another vector.
  *\param other The factor this vector is multiplied with.
  *\return A new object that contains the result of the calculation.
  */
  Vector3<V> operator^(const Vector3<V>& other)
    {return Vector3<V>(y * other.z - z * other.y, 
                    z * other.x - x * other.z, 
                    x * other.y - y * other.x);}

  /** Crossproduct of this vector and another vector.
  *\param other The factor this vector is multiplied with.
  *\return A reference to this object after the calculation.
  */
  Vector3<V>& operator^=(const Vector3<V>& other)
    {*this = *this ^ other; return *this;}

	/** normalize this vector.
  *\param len The length, the vector should be normalized to, default=1.
  *\return the normalized vector.
  */
  Vector3<V> normalize(V len)
  {
    if (abs() == 0) return *this;
    return *this = (*this * len) / abs();
   }

 /** normalize this vector.
  *\return the normalized vector.
  */
  Vector3<V> normalize()
  {
    if (abs() == 0) return *this;
    return *this /= abs();
  }
};

/**
* Streaming operator that reads a Vector3<V> from a stream.
* @param stream The stream from which is read.
* @param vector3 The Vector3<V> object.
* @return The stream.
*/ 
template <class V> In& operator>>(In& stream, Vector3<V>& vector3)
{
  stream >> vector3.x;
  stream >> vector3.y;
  stream >> vector3.z;
  return stream;
}

/**
* Streaming operator that writes a Vector3<V> to a stream.
* @param stream The stream to write on.
* @param vector3 The Vector3<V> object.
* @return The stream.
*/ 
template <class V> Out& operator<<(Out& stream, const Vector3<V>& vector3)
{
  stream << vector3.x;
  stream << vector3.y;
  stream << vector3.z;
  return stream;
}

#endif // __Vector3_h__

/*
* Change log :
* 
* $Log: Vector3.h,v $
* Revision 1.4  2004/03/10 08:11:12  roefer
* abs() fixed
*
* Revision 1.3  2003/12/05 08:04:08  jhoffman
* added normalize() to Vector3
*
* Revision 1.2  2003/12/02 13:44:56  cesarz
* added streaming operators
*
* Revision 1.1  2003/10/07 10:13:24  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:28  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.6  2003/03/10 18:22:51  dueffert
* assignments replaced with initializations
*
* Revision 1.5  2002/11/19 15:43:04  dueffert
* doxygen comments corrected
*
* Revision 1.4  2002/11/12 23:00:47  dueffert
* started restore greenhills compatibility
*
* Revision 1.3  2002/10/14 13:14:25  dueffert
* doxygen comments corrected
*
* Revision 1.2  2002/09/22 18:40:52  risler
* added new math functions, removed GTMath library
*
* Revision 1.1  2002/09/22 13:10:50  risler
* new Math headers added
*
*
*/
