/**
* @file InMessage.h
*
* Declaration of class InMessageQueue, InBinaryMessage, InTextMessage,
* InConfigMessage, InMessage and MessageHandler.
*
* Include that file for the definition of classes that derive from MessageHandler.
*
* @author Martin Ltzsch
*/

#ifndef __InMessage_h_
#define __InMessage_h_

#include "Tools/Streams/InStreams.h"
#include "Platform/MessageQueueBase.h"

/**
* @class InMessageQueue.
*
* A PhysicalInStream that reads the data from a MessageQueue.
*/
class InMessageQueue : public PhysicalInStream
{
private:
  /** The queue where the data are read from */
  MessageQueueBase* queue; 
  
public:
  /** Default constructor */
  InMessageQueue();
  
  
  /**
  * The function states whether the stream actually exists.
  * @return Does the file exist?
  */
  virtual bool exists() const;
  
  /**
  * The function states whether the end of the file has been reached.
  * @return End of file reached?
  */
  virtual bool getEof() const;
  
protected:
/**
* opens the stream.
* @param q A pointer to the message queue base
  */
  void open(MessageQueueBase* q);
  
  /**
  * The function reads a number of bytes from the stream.
  * @param p The address the data is written to. Note that p
  *          must point to a memory area that is at least
  *          "size" bytes large.
  * @param size The number of bytes to be read.
  */
  virtual void readFromStream(void* p,int size);
};

/** 
* @class InBinaryMessage
*
* A binary stream from a message queue.
*/
class InBinaryMessage : public InStream<InMessageQueue,InBinary>
{
public:
/** 
* Constructor 
* @param q A pointer to the message queue base
  */
  InBinaryMessage(MessageQueueBase* q);
};

/** 
* @class InTextMessage
*
* A text stream from a message queue.
*/
class InTextMessage : public InStream<InMessageQueue,InText>
{
public:
/** 
* Constructor 
* @param q A pointer to the message queue base
  */
  InTextMessage(MessageQueueBase* q);

  /** 
  * Reads the complete stream into a character buffer rechanging the escaped characters.
  * @param buf The buffer to fill. Note that buf must point
  *              to a memory area that is large enough to carry 
  *              the whole string.
  */
  void readAll(char* buf);
};

/** 
* @class InConfigMessage
*
* A config-file-style text stream from a message queue.
*/
class InConfigMessage : public InStream<InMessageQueue,InConfig>
{
public:
/** 
* Constructor 
* @param q A pointer to the message queue base
  */
  InConfigMessage(MessageQueueBase* q);
};

/**
* @class InMessage
*
* An Interface for reading single messages from a MessageQueue that is used by 
* MessageHandler derivates. 
*
* Use the bin, text or config member for formated reading from a message queue. 
*/
class InMessage
{
private:
/** 
* The message queue where the messages are read from. Note that the selection which message is 
* read is not done by InMessageQueue but by MessageQueue itself.
  */
  MessageQueueBase& queue;
  
public:
  /** An interface for reading binary messages from the queue */ 
  InBinaryMessage bin;
  
  /** An interface for reading text messages from the queue */ 
  InTextMessage text;
  
  /** An interface for reading config-file-style text messages from the queue */ 
  InConfigMessage config;
  
  /** 
  * Constructor
  * @param queue A reference to a MessageQueueBase
  */
  InMessage(MessageQueueBase& queue);

  /** returns the time stamp of the current message */
  unsigned long getTimeStamp() const;

  /** returns the message id of the current message */
  MessageID getMessageID() const;

  /** returns the message size of the current message */
  int getMessageSize() const;

  /** returns the team color of the robot that sent this message. Can be undefined. */
  Player::teamColor getTeamColor() const;

  /** returns the player number of the robot that sent this message. Can be undefined. */
  Player::playerNumber getPlayerNumber() const;

  /** 
  * returns the robot number of the robot that sent this message. 
  * @return 0: red1, 3:red4, 4:blue1, 7:blue4, 8:undefined */
  int getRobotNumber() const;

  /** 
  * returns whether the message was sent from a physical robot.
  * This is true, when the message was sent from the Aperios/OpenR platform,
  * otherwise false.
  */
  bool getMessageWasSentFromAPhysicalRobot() const;

  /** sets the read position to 0 so that the message can be read again */
  void resetReadPosition();

protected:
  /** 
  * Gives a direct read access to the message.
  * @return The adress of the first byte of the message.
  */
  const char* getData() const;

  /** gives the class MessageQueue access to protected members */
  friend class MessageQueue;

  /** gives the operator that copies a InMessage to another queue access to protected members */
  friend void operator >> (InMessage& message, MessageQueue& queue);

  /** Gives the stream operator access to protected members */
  friend Out& operator<<(Out& stream, const MessageQueue& messageQueue);
};

/**
* @class MessageHandler
*
* Responsible for distribution of incoming messages.
* Derive any class that shall receive messages from MessageHandler and implement the 
* handleMessage() function. 
*
* The MessageQueue needs a reference to a MessageHandler to distribute incoming messages.
*/
class MessageHandler
{
public:
  /** 
  * Called from a MessageQueue to distribute messages.
  * Use message.getMessageID to decide if the message is relavant for 
  * the MesssageHandler derivate.
  * Use message.bin, message.text or message.config as In streams to get the data from.
  * @param message The message that can be read.
  * @return true if the message was read (handled).
  */
  virtual bool handleMessage(InMessage& message) = 0;
};

#endif //__InMessage_h_

/*
* Change Log:
*
* $Log: InMessage.h,v $
* Revision 1.3  2003/12/06 23:39:07  loetzsch
* added getRobotNumber()
*
* Revision 1.2  2003/12/06 23:23:55  loetzsch
* messages in a MessageQueue now contain
* - the team color of the robot which sent the message
* - the player number of the robot which sent the message
* - if the message was sent from a physical robot or not
*
* Revision 1.1  2003/10/07 10:13:24  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:28  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.2  2002/09/10 17:52:58  loetzsch
* corrected comment
*
* Revision 1.1  2002/09/10 15:53:59  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.1  2002/07/23 13:47:14  loetzsch
* - new streaming classes
* - new debug message handling
* - exchanged StaticQueue by MessageQueue
*
*/
