/**
* @file MessageQueue.cpp
*
* Implementation of class MessageQueue and helper classes
* 
* @author Martin Ltzsch
*/

#include "MessageQueue.h"
#include "Platform/GTAssert.h"

MessageQueue::MessageQueue()
: in(queue), out(queue)
{
  ASSERT(idJPEGImage == 37); // otherwise log files will not work anymore
}

void MessageQueue::setSize(int size)
{
  queue.setSize(size);
}

void MessageQueue::setPlayerForNewMessages(const Player& player)
{
  out.teamColorForNewMessages = player.getTeamColor();
  out.playerNumberForNewMessages = player.getPlayerNumber();
}

void MessageQueue::handleSpecificMessages(MessageID id, MessageHandler& handler)
{
  for (int i=0; i<queue.getNumberOfMessages(); i++)
  {
    queue.setSelectedMessageForReading(i);
    if (in.getMessageID()==id)
    {
      in.config.reset();
      in.text.reset();
      handler.handleMessage(in);
    }
  }
}

void MessageQueue::handleAllMessages(MessageHandler& handler)
{
  for (int i=0; i<queue.getNumberOfMessages(); i++)
  {
    queue.setSelectedMessageForReading(i);
    in.config.reset();
    in.text.reset();
    handler.handleMessage(in);
  }
}

void MessageQueue::copyAllMessages(MessageQueue& other)
{
  for (int i=0; i<queue.getNumberOfMessages(); i++)
  {
    copyMessage(i,other);
  }
}

void MessageQueue::moveAllMessages(MessageQueue& other)
{
  copyAllMessages(other);
  clear();
}

void MessageQueue::clear()
{
  queue.clear();
}

bool MessageQueue::isEmpty() const
{
  return queue.getNumberOfMessages() == 0;
}

void MessageQueue::copyMessage(int message, MessageQueue& other)
{
  queue.setSelectedMessageForReading(message);
  other.out.bin.write(queue.getData(),queue.getMessageSize());
  other.out.finishMessage(
    queue.getMessageID(),
    queue.getTimeStamp(),
    queue.getTeamColor(),
    queue.getPlayerNumber(),
    queue.getMessageWasSentFromAPhysicalRobot());
}

//
// Format of a message in a log file:
//
// byte
// 0     Bit 0: if 1, then this is the last message in the queue
//       Bit 1-2: the team color. 0:undef, 1: red, 2:blue, 3:undef
//       Bit 2-5: the player number: 0:undef, 1:0, 2:1, 3:2, 4:3, 5:undef
//       Bit 6  : where the message is from: 0:physical robot, 1:simulated robot
// 1-4   Size of the message (without header) in bytes
// 5     id of the message
// 6-9  timestamp
// 10-?  content
//

In& operator>>(In& stream,MessageQueue& messageQueue)
{
  unsigned int messageSize;
  char id,c;
  unsigned long timeStamp;
  bool last=false;
  char buffer[100000];
  while (!last)
  {
    stream >> c >> messageSize >> id >> timeStamp;
    ASSERT(messageSize<10*1024*1024);
    while(messageSize > 0)
    {
      unsigned int blockSize = messageSize > sizeof(buffer) ? sizeof(buffer) : messageSize;
      stream.read(buffer,blockSize); 
      messageQueue.out.bin.write(buffer,blockSize);
      messageSize -= blockSize;
    }

    // This might look strange but is needed for compability with old log files
    last = (c%2==1);
    messageQueue.out.finishMessage((MessageID)id,timeStamp,
      ((c>>1)%4==0? Player::undefinedTeamColor : (Player::teamColor)(((c>>1)%4)-1)),
      ((c>>3)%8==0? Player::undefinedPlayerNumber : (Player::playerNumber)(((c>>3)%8)-1)),
      (c>>6)==0);
#ifdef _WIN32
    last &= stream.eof();
#endif
  }
  return stream;
}

int MessageQueue::getStreamedSize()
{
  int size = 0;

  for (int i=0; i< queue.getNumberOfMessages(); i++)
  {
    queue.setSelectedMessageForReading(i);
    size += in.getMessageSize() + 10;
  }
  return size;
}

Out& operator<<(Out& stream, const MessageQueue& messageQueue)
{
  for (int i=0; i< messageQueue.queue.getNumberOfMessages(); i++)
  {
    // check if the message is the last streamed message
    bool last = (i == messageQueue.queue.getNumberOfMessages()-1);

    ((MessageQueue&)messageQueue).queue.setSelectedMessageForReading(i);

    // This might look strange but is needed for compability with old log files
    char c = (char)(last)
      + (((char)messageQueue.in.getTeamColor()+1) << 1)
      + (((char)messageQueue.in.getPlayerNumber()+1) << 3)
      + (((char)!messageQueue.in.getMessageWasSentFromAPhysicalRobot()) << 6);

    // write message parameters to the stream
    stream << c
      << messageQueue.in.getMessageSize() 
      << (char)messageQueue.in.getMessageID()
      << messageQueue.in.getTimeStamp();

    // write message content
    stream.write(messageQueue.in.getData(),messageQueue.in.getMessageSize());
  }
  return stream;
}

void operator >> (InMessage& message, MessageQueue& queue)
{
  queue.out.bin.write(message.getData(),message.getMessageSize());
  queue.out.finishMessage(message.getMessageID(),
    message.getTimeStamp(),
    message.getTeamColor(),
    message.getPlayerNumber(),
    message.getMessageWasSentFromAPhysicalRobot());
}

/*
* Change Log:
*
* $Log: MessageQueue.cpp,v $
* Revision 1.6  2004/07/07 15:16:58  dueffert
* support for appended logfiles added
*
* Revision 1.5  2004/03/18 18:04:25  roefer
* idJPEGImage repaired
*
* Revision 1.4  2004/01/17 20:55:34  loetzsch
* bug fix in getStreamedSize()
*
* Revision 1.3  2004/01/10 18:04:58  loetzsch
* added MessageQueue::getStreamedSize()
*
* Revision 1.2  2003/12/06 23:23:55  loetzsch
* messages in a MessageQueue now contain
* - the team color of the robot which sent the message
* - the player number of the robot which sent the message
* - if the message was sent from a physical robot or not
*
* Revision 1.1  2003/10/07 10:13:24  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:28  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.6  2003/05/09 11:40:50  goehring
* Assertion for >10MByte messages implemented
*
* Revision 1.5  2003/03/06 11:56:46  dueffert
* signed comparison warnings removed
*
* Revision 1.4  2003/02/05 13:31:44  loetzsch
* added
*       in.config.reset();
*       in.text.reset();
* to MessageQueue::handleSpecificMessages()
*
* Revision 1.3  2003/02/05 12:37:14  dueffert
* handleSpecificMessage added
*
* Revision 1.2  2003/01/26 19:37:16  loetzsch
* In ::handleMessage() now for every message
*     in.config.reset();
*     in.text.reset();
* is called.
*
* Revision 1.1  2002/09/10 15:53:59  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.3  2002/08/08 16:38:47  loetzsch
* moved some members to the derived class LogPlayer because they are not
* needed in the normal MessageQueue
*
* Revision 1.2  2002/07/27 19:07:50  roefer
* Streaming operator is more efficient now
*
* Revision 1.1  2002/07/23 13:47:14  loetzsch
* - new streaming classes
* - new debug message handling
* - exchanged StaticQueue by MessageQueue
*
*/

