/**
* @file ModuleHandler.cpp
*
* Implementation of class ModuleHandler.
*
* @author Max Risler
* @author Martin Ltzsch
*/

#include "ModuleHandler.h"
#include "Tools/Debugging/Debugging.h"

ModuleHandler::ModuleHandler()
{
  for(int i = 0; i < SolutionRequest::numOfModules; i++)
    pModuleSelectors[i] = 0;
}

ModuleSelector* ModuleHandler::pGetModuleSelector(SolutionRequest::ModuleID id) const
{
  return pModuleSelectors[id];
}

bool ModuleHandler::handleMessage(InMessage& message)
{
  bool handled = false;
  for(int i = 0; i < SolutionRequest::numOfModules; i++)
  {
    if (pModuleSelectors[i] != 0)
    {
      if (pModuleSelectors[i]->handleMessage(message))
      {
        handled = true;
        message.resetReadPosition();
      }
    }
  }
  return handled;
}

void ModuleHandler::setModuleSelector(SolutionRequest::ModuleID id, ModuleSelector *module)
{
  pModuleSelectors[id] = module;
  if (module!=0) module->init();
}

SolutionRequest::ModuleSolutionID ModuleHandler::getSelectedSolution(SolutionRequest::ModuleID id) const
{
  if (pModuleSelectors[id] != 0)
    return pModuleSelectors[id]->getSelectedSolution();
  else
    return SolutionRequest::disabled;
}

void ModuleHandler::selectSolution(SolutionRequest::ModuleID id, SolutionRequest::ModuleSolutionID solution)
{
  if (pModuleSelectors[id] != 0 && pModuleSelectors[id]->getSelectedSolution()!=solution)
  {
    OUTPUT(idText, text, "ModuleHandler::selectSolution :" << 
      " Switched solution for " << SolutionRequest::getModuleName(id) <<
      " from " << SolutionRequest::getModuleSolutionName(id,pModuleSelectors[id]->getSelectedSolution()) <<
      " to " << SolutionRequest::getModuleSolutionName(id,solution) );
    pModuleSelectors[id]->selectSolution(solution);
  }
}

void ModuleHandler::selectSolutions(SolutionRequest request)
{
  for (int i=0; i < SolutionRequest::numOfModules; i++) 
    selectSolution((SolutionRequest::ModuleID)i,request.solutions[i]);
}

/*
 * Change log :
 * 
 * $Log: ModuleHandler.cpp,v $
 * Revision 1.2  2004/01/21 14:31:58  loetzsch
 * Module Selectors create only the selected solution.
 * When the solution changes, the old solution is erased and the new
 * one ist created using createSolution(..)
 *
 * Revision 1.1  2003/10/07 10:13:24  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1.1.1  2003/07/02 09:40:29  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.5  2002/11/19 15:43:04  dueffert
 * doxygen comments corrected
 *
 * Revision 1.4  2002/11/12 18:52:08  risler
 * reset read position of handled messaged (fixing crash when e.g. sending walking engine parameters)
 *
 * Revision 1.3  2002/09/25 10:25:13  loetzsch
 * removed the "executeVisionModules" variable
 * from SolutionRequest and ModuleHandler.
 *
 * Revision 1.2  2002/09/10 17:53:26  loetzsch
 * First draft of new Module/Solution Architecture
 *
 * Revision 1.1  2002/09/10 15:53:59  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.3  2002/07/23 13:48:40  loetzsch
 * - new streaming classes
 * - removed many #include statements
 * - exchanged StaticQueue by MessageQueue
 * - new debug message handling
 * - general clean up
 *
 * Revision 1.2  2002/06/04 00:12:45  loetzsch
 * added bool executeVisionModules
 *
 * Revision 1.1.1.1  2002/05/10 12:40:32  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.7  2002/02/08 18:36:03  risler
 * removed destructor again
 *
 * Revision 1.6  2002/02/08 17:48:21  risler
 * destructor added
 *
 * Revision 1.5  2002/01/20 13:00:18  risler
 * changed implementation of selecting default solution
 *
 * Revision 1.4  2002/01/19 12:43:16  risler
 * enabled SolutionRequest, changed HandleDebugMessage calls
 *
 * Revision 1.3  2002/01/17 14:35:48  risler
 * SolutionRequest added
 *
 * Revision 1.2  2001/12/10 17:47:10  risler
 * change log added
 *
 */
