/**
* @file SolutionRequest.cpp
*
* Implementation of class SolutionRequest
*
* @author Martin Ltzsch
* @author Max Risler
*/

#include "SolutionRequest.h"
#include "Tools/Debugging/Debugging.h"
#include "Platform/GTAssert.h"
#include "Tools/Streams/InStreams.h"
#include "Tools/Location.h"

SolutionRequest::SolutionRequest(bool setDefault)
{
  errorWhileReading = false;
  if (setDefault) 
    setDefaultSolutions();
  else
  {
    for (int i=0;i<numOfModules;i++)
    {
      solutions[i] = disabled;
    }
  }
}

void SolutionRequest::setDefaultSolutions()
{
  InConfigFile modulesFile(getLocation().getFilename("modules.cfg"));
  if (!modulesFile.exists() || modulesFile.eof()) {
    errorWhileReading = true;
//    OUTPUT(idText,text,"SolutionRequest : Error, modules.cfg not found.");
    for (int i=0;i<numOfModules;i++)
    {
      solutions[i] = disabled;
    }
  } else {
    modulesFile >> *this;
    if (errorWhileReading) {
//      PRINT("SolutionRequest : Error, modules.cfg contained errors. Please check or write new modules.cfg with Settings Dialog.");
    }
  }
}

bool SolutionRequest::operator == (const SolutionRequest& other)
{
  for (int i=0;i<numOfModules;i++)
  {
    if (solutions[i] != other.solutions[i]) return false;
  }
  return true;
}

In& operator>>(In& stream,SolutionRequest& solutionRequest)
{
  char module[80];
  char solution[80];
  int i;

  solutionRequest.errorWhileReading = false;
  for (i=0; i<SolutionRequest::numOfModules; i++)
    solutionRequest.solutions[i] = (SolutionRequest::ModuleSolutionID)-2;

  while (!stream.eof())
  {
    stream >> module;
    if (strlen(module) > 0) {
      if (stream.eof())
      {
        solutionRequest.errorWhileReading = true;
        //OUTPUT(idText,text,"SolutionRequest : Error while reading from stream, unexpected end.");
        break;
      }
      stream >> solution;
      for (i=0; i<SolutionRequest::numOfModules; i++) {
        if (strcmp(module,SolutionRequest::getModuleName((SolutionRequest::ModuleID)i))==0) 
        {
          int j;
          for (j=-1; j<SolutionRequest::getNumOfSolutions((SolutionRequest::ModuleID)i); j++) {
            if (strcmp(solution,SolutionRequest::getModuleSolutionName((SolutionRequest::ModuleID)i,(SolutionRequest::ModuleSolutionID)j))==0) {
              solutionRequest.solutions[i] = (SolutionRequest::ModuleSolutionID)j;
              break;
            }
          }
          if (j == SolutionRequest::getNumOfSolutions((SolutionRequest::ModuleID)i)) {
            solutionRequest.errorWhileReading = true;
            //OUTPUT(idText,text,"SolutionRequest : Error while reading from stream, invalid solution " << solution << " was requested for module " << module << ".");
          }
          break;
        }
      }
      if (i == SolutionRequest::numOfModules) {
        solutionRequest.errorWhileReading = true;
//        OUTPUT(idText,text,"SolutionRequest : Error while reading from stream, invalid module " << module << ".");
      }
    }
  }

  for (i=0; i<SolutionRequest::numOfModules; i++)
    if (solutionRequest.solutions[i] == -2) {
      solutionRequest.errorWhileReading = true;
      solutionRequest.solutions[i] = SolutionRequest::disabled;
    }
    
  return stream;
}      
  

Out& operator<<(Out& stream, const SolutionRequest& solutionRequest)
{
  stream << endl;
  for (int i=0;i<SolutionRequest::numOfModules;i++)
  {
    stream << SolutionRequest::getModuleName((SolutionRequest::ModuleID)i);
    if (solutionRequest.solutions[i] < SolutionRequest::disabled || 
      solutionRequest.solutions[i] >= SolutionRequest::getNumOfSolutions((SolutionRequest::ModuleID)i))
    {
      //OUTPUT is a bad idea here, since this may be called from another OUTPUT
      //OUTPUT(idText,text,"SolutionRequest : Error while writing to stream, SolutionRequest contained invalid value for module " << SolutionRequest::getModuleName((SolutionRequest::ModuleID)i) << ", value was " << solutionRequest.solutions[i] << ".");
      stream << "disabled";
    }
    else
      stream << SolutionRequest::getModuleSolutionName((SolutionRequest::ModuleID)i,solutionRequest.solutions[i]);
    stream << endl;
  }
  return stream;
}

/*
* Change log :
* 
* $Log: SolutionRequest.cpp,v $
* Revision 1.2  2004/01/24 14:55:29  loetzsch
* created ATH AiboControl
*
* Revision 1.1  2003/10/07 10:13:24  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/09/01 15:19:01  juengel
* SolutionRequest clean-up.
*
* Revision 1.1.1.1  2003/07/02 09:40:29  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.13  2003/05/11 17:03:00  risler
* added location.cfg
*
* Revision 1.12  2003/04/25 19:50:17  goehring
* Added new module CollisionDetector
*
* Revision 1.11  2002/11/11 11:53:02  risler
* added error message for wrong modules.cfg
*
* Revision 1.10  2002/11/09 11:22:51  risler
* added errorWhileReading flag to SolutionRequest
*
* Revision 1.9  2002/11/07 15:36:58  risler
* removed OUTPUT in SolutionRequest streaming operator
*
* Revision 1.8  2002/11/06 14:10:47  risler
* added error message
*
* Revision 1.7  2002/10/11 10:47:54  risler
* added error checking to streaming operator <<
*
* Revision 1.6  2002/10/01 15:31:02  loetzsch
* fixed gcc errors
*
* Revision 1.5  2002/09/25 10:25:13  loetzsch
* removed the "executeVisionModules" variable
* from SolutionRequest and ModuleHandler.
*
* Revision 1.4  2002/09/24 18:41:58  risler
* human readable solutionrequest streaming operator
* default module solutions read from modules.cfg
*
* Revision 1.3  2002/09/24 16:28:20  risler
* preparation for reading default module solutions from file
*
* Revision 1.2  2002/09/10 17:53:26  loetzsch
* First draft of new Module/Solution Architecture
*
* Revision 1.1  2002/09/10 15:53:59  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.2  2002/06/04 00:12:24  loetzsch
* added == and != operator
* added bool executeVisionModules
*
* Revision 1.1.1.1  2002/05/10 12:40:32  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.4  2002/02/18 14:07:00  juengel
* streaming operators send and receive numberOfSolutions
*
* Revision 1.3  2002/01/20 13:00:18  risler
* changed implementation of selecting default solution
*
* Revision 1.2  2002/01/17 14:35:48  risler
* SolutionRequest added
*
*/
