/**
* @file Player.h
*
* Definition of class Player
* 
* @author Martin Ltzsch
*/
#ifndef __Player_h_
#define __Player_h_

#include <string.h>
#include "Tools/Streams/InOut.h"

class Player;

/**
* Returns a reference to a process wide player configuration.
*/
Player& getPlayer();

/** 
* @class Player
*
* A class that represents the team color and player number of the robot. 
*
* @author Martin Ltzsch
*/
class Player
{
public:
  /** Constructor. */
  Player();

  /** Loads the data from player.cfg */
  void load();

  /** possible team colors */
  enum teamColor {red, blue, undefinedTeamColor};

  /** returns the name of a color */
  static const char* getTeamColorName(teamColor color)
  {
    switch (color)
    {
    case red: return "red";
    case blue: return "blue";
    default: return "undefined";
    }
  }

  /** returns the team color */
  teamColor getTeamColor() const { return theTeamColor; }

  /** sets the team color */
  void setTeamColor(teamColor t) { theTeamColor=t; }

  /** returns the color for a given name */
  static teamColor getTeamColorFromString(char* name)
  {
    if (strcmp(name,"red") == 0) return red;
    else if (strcmp(name,"blue") == 0) return blue;
    else return undefinedTeamColor;
  }

 // #define FIVEDOGS  
/* -> Martin : Bitte vorlufig drinlassen.
 * Ich berblicke es im Augenblick nicht was mehr als 4 TeamMessageSender / Receiver
 * fr Auswirkungen in RobotControl haben. Wenn FIVEDOGS nicht gesetzt ist, werden nur 3 
 * angelegt.
 *
 * Michael.
 */

  /** the possible player numbers */
  enum playerNumber
  {
    one = 0, two, three, four, five, 
      undefinedPlayerNumber, numOfPlayerNumbers = undefinedPlayerNumber
  };

  /** returns the name of a number */
  static const char* getPlayerNumberName(playerNumber number)
  {
    switch (number)
    {
    case one: return "1";
    case two: return "2";
    case three: return "3";
    case four: return "4";
    case five: return "5";

    default: return "undefined";
    }
  }

  char* getTeamName()
  {
    return(theTeamName);
  }

  /** returns the player number */
  playerNumber getPlayerNumber() const { return thePlayerNumber; }

  /** sets the player number */
  void setPlayerNumber(playerNumber n) { thePlayerNumber=n; }

  /** returns the number for a given name */
  static playerNumber getPlayerNumberFromString(char* name)
  {
    if (strcmp(name,"1")==0) return one;
    else if (strcmp(name,"2")==0) return two; 
    else if (strcmp(name,"3")==0) return three; 
    else if (strcmp(name,"4")==0) return four; 
    else if (strcmp(name,"5")==0) return five; 

    else return undefinedPlayerNumber;
  }

private:
  /** the team color of the player */
  teamColor theTeamColor;

  /** the number of the player */
  playerNumber thePlayerNumber;

  /** the team-name of the player */
  char theTeamName[16];
};

/**
 * Streaming operator that reads a Player from a stream.
 * @param stream The stream from which is read.
 * @param player The Player object.
 * @return The stream.
 */ 
In& operator>>(In& stream,Player& player);
 
/**
 * Streaming operator that writes a Player to a stream.
 * @param stream The stream to write on.
 * @param player The Player object.
 * @return The stream.
 */ 
Out& operator<<(Out& stream, const Player& player);


#endif //__Player_h_

/*
* Change Log:
*
* $Log: Player.h,v $
* Revision 1.10  2004/05/17 17:17:50  kerdels
* added symbols for the sequence control of the open challenge,
* fixed a bug concerning 5-dog-mode and distribution of option-ratings,
* set one = 0 in enum Player::playerNumber --> just to be sure ;-)
*
* Revision 1.9  2004/05/14 14:12:08  wachter
* - Added communication support for 5 robots
* - rewrote parts of team-communication to be faster and more stable
*
* Revision 1.8  2004/05/13 18:08:33  loetzsch
* removed #ifdef statements, as they are not needed
*
* Revision 1.7  2004/05/12 19:33:14  kerdels
* merged the behavior changes during australian, american and japan open
*
* Revision 1.6  2004/01/09 15:44:58  wachter
* Added TeamName for Dog-Discovery-Protokol
*
* Revision 1.5  2003/12/06 23:38:46  loetzsch
* bug fix
*
* Revision 1.4  2003/12/06 23:21:59  loetzsch
* made the read functions const
*
* Revision 1.3  2003/12/06 19:01:05  loetzsch
* replaced undefinedColor by undefinedTeamColor
*
* Revision 1.2  2003/12/06 17:45:34  loetzsch
* replaced Player::playerRole (goalie, defender, striker1, striker2)
* by Player::playerNumber (one, two, three, four)
*
* Revision 1.1  2003/10/07 10:13:21  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/09/25 11:21:59  juengel
* Removed BlobCollection.
*
* Revision 1.1.1.1  2003/07/02 09:40:28  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.4  2003/05/03 16:20:01  roefer
* robot.cfg added
*
* Revision 1.3  2003/04/16 07:00:17  roefer
* Bremen GO checkin
*
* Revision 1.3  2003/04/09 14:47:31  roefer
* bodyTiltOffset is read from player.cfg
*
* Revision 1.2  2003/03/06 11:54:03  dueffert
* missing case warning removed
*
* Revision 1.1  2002/09/10 15:53:58  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.7  2002/09/08 16:21:42  roefer
* barChallenge name added
*
* Revision 1.6  2002/08/30 19:38:03  loetzsch
* removed a few roles
*
* Revision 1.5  2002/07/23 13:48:40  loetzsch
* - new streaming classes
* - removed many #include statements
* - exchanged StaticQueue by MessageQueue
* - new debug message handling
* - general clean up
*
* Revision 1.4  2002/06/12 17:54:12  risler
* only one role barChallenger
*
* Revision 1.3  2002/06/10 12:29:33  tim
* added roles for ball challenge
*
* Revision 1.2  2002/05/15 10:46:22  brunn
* New Roles for Colaboration Challenge 2002
* Extended draft for bar-perception
*
* Revision 1.1.1.1  2002/05/10 12:40:32  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.4  2002/05/05 14:04:36  loetzsch
* added numOfPlayerRoles to the enum
*
* Revision 1.3  2002/04/08 13:34:14  dueffert
* PlayersToolBar added
*
* Revision 1.2  2002/02/07 19:57:06  risler
* DefaultLEDControl uses getPlayer now, removed team and role from LEDRequest
*
* Revision 1.1  2002/02/05 03:42:46  loetzsch
* added to the cvs
*
*/
