/**
* @file FutureWorldModelGenerator.h
* 
* Definition of class FutureWorldModelGenerator
*
* @author <a href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</a>
*/

#ifndef FUTUREWORLDMODELGENERATOR_H_
#define FUTUREWORLDMODELGENERATOR_H_


#include <string>
#include <vector>

class Potentialfield;
class Actionfield;
class PotentialfieldTransformation;
class Rotation;
class Action;
class Object;
class PfPose;
class Polygon;
class PfVec;


/**
* @class FutureWorldModelGenerator
*
* A class for computing a world model after excuting a specific action.
*/
class FutureWorldModelGenerator
{
private:
  /** Constructor */
  FutureWorldModelGenerator() {};

public:
  /** Returns a pointer to the class (which is a singleton)
  * @return The pointer
  */
  static FutureWorldModelGenerator* getFutureWorldModelGenerator()
  {
    static FutureWorldModelGenerator instance;
    return &instance;
  }

  /** Transforms the world state by applying an action
  * @param poseNow The original pose
  * @param poseThen The pose after the action
  * @param action The action
  * @param callingField The potential field calling this function
  * @param worldStateNow The state of all objects before the action
  * @param worldStateThen The state of all objects after the action
  * @param worldStateStatic All static objects (never move, cannot be moved)
  */
  void transformWorldState(const PfPose& poseNow, PfPose& poseThen,
                           Action& action, Actionfield* callingField,
                           std::vector<Object*>& worldStateNow, 
                           std::vector<Object*>& worldStateThen,
                           std::vector<Object*>& worldStateStatic);

private:
  /** Executes a transformation
  * @param pose The robot pose (may be changed by this function)
  * @param action The action to which the transformation is assigned
  * @param transformation The transformation to be executed
  * @param callingField The potential field using this class
  * @param dynamicWorldState All dynamic objects (may be changed by this function)
  * @param staticWorldState All static objects
  */
  void executeSingleTransformation(PfPose& pose, Action& action, 
                                   PotentialfieldTransformation* transformation,
                                   Actionfield* callingField,
                                   std::vector<Object*>& dynamicWorldState,
                                   std::vector<Object*>& staticWorldState);

  /** Checks a planned transformation for collisions
  * @param objectPose The pose of the object to be transformed
  * @param translation The planned translation
  * @param otherObjects All dynamic objects
  * @param otherStaticObjects All static objects
  * @param excludedObject The index (in otherObjects) of the object to be checked
  * @return A value [0.0 .. 1.0] indicated "how much" of the translation may be executed
  */
  double getMaxTranslationForObject(const PfPose& objectPose,
                                    const PfVec& translation,
                                    std::vector<Object*>& otherObjects,
                                    std::vector<Object*>& otherStaticObjects,
                                    int excludedObject = -1);

  /** Checks a planned rotation for collisions
  * @param objectPose The pose of the object to be transformed
  * @param rotationPose The position to rotate around also including the current rotation
  * @param rotation The planned rotation
  * @param otherObjects All dynamic objects
  * @param otherStaticObjects All static objects
  * @param excludedObject The index (in otherObjects) of the object to be checked
  * @return A value [0.0 .. 1.0] indicated "how much" of the translation may be executed
  */
  double getMaxRotationForObject(const PfPose& objectPose,
                                 const PfPose& rotationPose,
                                 double rotation,
                                 std::vector<Object*>& otherObjects,
                                 std::vector<Object*>& otherStaticObjects,
                                 int excludedObject = -1);
};


#endif //FUTUREWORLDMODELGENERATOR_H_



/*
* $Log: FutureWorldModelGenerator.h,v $
* Revision 1.3  2004/03/05 17:59:34  tim
* corrected doxygen comments
*
* Revision 1.2  2004/01/28 08:27:16  dueffert
* doxygen bugs fixed
*
* Revision 1.1  2004/01/20 15:42:19  tim
* Added potential fields implementation
*
* Revision 1.7  2003/05/22 14:23:47  tim
* Changed representation of transformations
*
* Revision 1.6  2003/05/12 00:03:29  dueffert
* doxygen bugs fixed
*
* Revision 1.5  2003/05/08 15:26:06  tim
* no message
*
* Revision 1.4  2003/04/22 14:35:17  tim
* Merged changes from GO
*
* Revision 1.4  2003/04/09 19:03:06  tim
* Last commit before GermanOpen
*
* Revision 1.3  2003/04/04 14:50:53  tim
* Fixed bugs, added minor features
*
* Revision 1.2  2003/03/23 20:32:37  loetzsch
* removed green compiler warning: no newline at end of file
*
* Revision 1.1  2003/03/23 17:51:27  tim
* Added potentialfields
*
*/
