/**
* @file PfcParser.cpp
* 
* Implementation of class Parser
*
* @author <a href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</a>
*/


#include "PfcParser.h"
#include "Actionfield.h"
#include "Motionfield.h"
#include "PfieldDatatypes.h"
#include "PotentialFunctions.h"
#include "PfieldGeometry.h"
#include "FieldObject.h"
#include "FormationObject.h"
#include "RandomMotionGenerator.h"
#include "AStarSearch.h"
#include <cassert>
#include <iostream>


inline double toDouble(const std::string& s)
{
  return atof(s.c_str());
}


inline int toInt(const std::string& s)
{
  return atoi(s.c_str());
}


inline bool toBool(const std::string& s)
{
  if(s=="true")
  {
    return true;
  }
  else
  {
    return false;
  }
}


inline ChangeType toChangeType(const std::string& s)
{
  if(s == "calls")
  {
    return CALLS;
  }
  else
  {
    return MILLISECONDS;
  }
}



Parser::~Parser()
{
  std::vector < Object* >::iterator object;
  for (object = objects.begin (); object != objects.end (); ++object)
  {
    delete (*object);
  }
  objects.clear();
  instanceGroups.clear();
}


inline char Parser::nextChar()
{
  char ch;
#ifdef POTENTIALFIELDS_FOR_GT2004_
  (*file) >> ch;
#else
  ch = file.get();
#endif //POTENTIALFIELDS_FOR_GT2004_
  return ch;
}


inline bool Parser::endOfFile()
{
#ifdef POTENTIALFIELDS_FOR_GT2004_
  return file->eof();
#else
  return file.eof();
#endif //POTENTIALFIELDS_FOR_GT2004_
}


std::string Parser::nextToken()
{
  std::string token="";
  char ch;
  do
  {
    ch = nextChar(); 
  } while (((ch == ' ') || (ch == '\t') || (ch == '\n') || 
           (ch == 0x0A) || (ch == 0x0D) || (ch == ';')) && !endOfFile());
  while ((ch != ';') && !endOfFile())
  {
    token += ch;
    ch = nextChar(); 
  }
  return token;
}


inline double Parser::parseAngle()
{
  return (toDouble(nextToken()) * pi_180);
}


void Parser::parseGeometricObject(PfieldGeometricObject*& geometricObject,
                                  const std::string& preparsedToken) 
{
  std::string geometryType;
  if(preparsedToken == "")
  {
    geometryType = nextToken();
  }
  else
  {
    geometryType = preparsedToken;
  }
  if(geometryType == "line")
  {
    Line* line= new Line();
    line->intersectable = toBool(nextToken());
    line->p1.x = toDouble(nextToken());
    line->p1.y = toDouble(nextToken());
    line->p2.x = toDouble(nextToken());
    line->p2.y = toDouble(nextToken());
    line->initRadiusOfCollisionCircle();  
    geometricObject = line;
  }
  else if(geometryType == "polygon")
  {
    Polygon* polygon = new Polygon();
    polygon->intersectable = toBool(nextToken()); 
    std::string x;
    PfVec p;
    x = nextToken();
    while(x != "EOL")
    {
      p.x = toDouble(x);
      p.y = toDouble(nextToken());
      polygon->addPoint(p);
      x = nextToken();
    }
    polygon->initRadiusOfCollisionCircle();  
    geometricObject = polygon;
  }
  else if(geometryType == "circle")
  {
    Circle* circle = new Circle();
    circle->radius = toDouble(nextToken());
    circle->intersectable = toBool(nextToken());
    circle->initRadiusOfCollisionCircle();  
    geometricObject = circle;
  }
  else if(geometryType == "no-geometry")
  {
    NoGeometry* noGeometry = new NoGeometry();
    noGeometry->intersectable = false;
    noGeometry->initRadiusOfCollisionCircle();
    geometricObject = noGeometry;
  }
}


void Parser::parseFunction(PotentialfieldFunction*& function)
{
  std::string functionType = nextToken();
  if(functionType == "linear-function")
  {
    double range = toDouble(nextToken());
    double atZero = toDouble(nextToken());
    LinearFunction* linearFunction = new LinearFunction(range, atZero);
    function = linearFunction;
  }
  else if(functionType == "parabolic-function")
  {
    double range = toDouble(nextToken());
    double atZero = toDouble(nextToken());
    ParabolicFunction* parabolicFunction = new ParabolicFunction(range, atZero);
    function = parabolicFunction;
  }
  if(functionType == "no-function")
  {
    NoFunction* noFunction = new NoFunction();
    function = noFunction;
  }
  else if(functionType == "asymptotic-function")
  {
    double range = toDouble(nextToken());
    double atZero = toDouble(nextToken());
    double solidCenter = toDouble(nextToken());
    assert(solidCenter > 0.0);
    AsymptoticFunction* asymptoticFunction = new AsymptoticFunction(range, atZero, solidCenter);
    function = asymptoticFunction;
  }
  else if(functionType == "social-function")
  {
    double cRep(toDouble(nextToken()));
    double omRep(toDouble(nextToken()));
    double cAtt(toDouble(nextToken()));
    double omAtt(toDouble(nextToken()));
    double epsilon(toDouble(nextToken()));
    double k(toDouble(nextToken()));
    assert((cRep >= 0.0) && (cAtt >= 0.0) && (omRep > omAtt) && (omAtt >= 0.0) && (epsilon >= 0.0));
    SocialFunction* socialFunction = new SocialFunction(cRep,omRep,cAtt,omAtt, epsilon, k);
    function = socialFunction;
  }
  if((functionType == "asymptotic-function") || 
     (functionType == "linear-function") ||
     (functionType == "parabolic-function"))
  {
    double smoothingAtObject(toDouble(nextToken())/100.0);
    double smoothingAtBorder(toDouble(nextToken())/100.0);
    double gradientAtObject(toDouble(nextToken()));
    double gradientAtBorder(toDouble(nextToken()));
    assert((smoothingAtObject >= 0.0) && (smoothingAtBorder >=0.0) &&
           (smoothingAtObject <= 1.0) && (smoothingAtBorder <=1.0));
    function->setSmoothingParameters(smoothingAtObject, smoothingAtBorder,
                                     gradientAtObject, gradientAtBorder);
  }
}


inline Object* Parser::getInstance(const std::string& name)
{
  NameToIndexMap::const_iterator pos;
  pos = composition->objectMap.find(name);
  assert(pos != composition->objectMap.end());
  return composition->objects[pos->second];
}


void Parser::parseObjectsForField(const std::string& nameOfFirst,Potentialfield* field)
{
  std::string objectName = nameOfFirst;
  Object* object;
  while(objectName != "EOL")
  {
    if(objectName == "include-group")
    {
      objectName = nextToken();
      addGroupToField(objectName, field);
      objectName = nextToken();
    }
    else
    {
      if(objectName == "include-formation")
      {
        objectName = nextToken();
      }
      object = getInstance(objectName);
      assert(object != 0);
      field->addObject(object);
      objectName = nextToken();
    }
  }
}


void Parser::addGroupToField(const std::string& groupName, Potentialfield* field)
{
  bool groupFound = false;
  for(unsigned int i=0; i<instanceGroups.size(); i++)
  {
    if(instanceGroups[i].name == groupName)
    {
      for(unsigned int j=0; j<instanceGroups[i].objects.size(); j++)
      {
        field->addObject(instanceGroups[i].objects[j]);
      }
      groupFound = true;
      break;
    }
  }
  assert(groupFound);
}


void Parser::parseObjectStateSymbol()
{
  std::string symbol(nextToken());
  composition->addDynamicObjectState(symbol);
}


void Parser::parseInstanceGroup()
{
  InstanceGroup newGroup;
  newGroup.name = nextToken();
  std::string objectName = nextToken();
  Object* object;
  while(objectName != "EOL")
  {
    object = getInstance(objectName);
    assert(object != 0);
    newGroup.objects.push_back(object);
    objectName = nextToken();
  }
  instanceGroups.push_back(newGroup);
}


Object* Parser::parseObject()
{
  std::string name(nextToken());
  std::string type(nextToken());
  ObjectType objectType;
  if(type=="attractive")
  {
    objectType = ATTRACTIVE;
  }
  else 
  {
    objectType = REPULSIVE;
  }
  Object* object = new Object(name, objectType);
  std::string tangentialField = nextToken();
  if(tangentialField == "clockwise")
  {
    object->setTangentialField(CLOCKWISE);
  }
  else if(tangentialField == "counter-clockwise")
  {
    object->setTangentialField(COUNTER_CLOCKWISE);
  }
  object->setPfieldCompositionRef(composition);
  PotentialfieldFunction* function;
  parseFunction(function);
  object->setFunction(function);
  delete function;
  std::string fieldType = nextToken();
  if(fieldType == "point-field")
  {
    object->setField(POINT_FIELD);
  }
  else if(fieldType == "shape-field")
  {
    object->setField(SHAPE_FIELD);
  }
  else if(fieldType == "sector-field")
  {
    Sector sector;
    sector.openingAngle = parseAngle();
    std::string crossFunctionString = nextToken();
    if(crossFunctionString == "linear-function")
    {
      sector.crossFunction = new LinearFunction(1.0,1.0);
    }
    else if(crossFunctionString == "parabolic-function")
    {
      sector.crossFunction = new ParabolicFunction(1.0,1.0);
    }
    else if(crossFunctionString == "asymptotic-function")
    {
      sector.crossFunction = new AsymptoticFunction(1.0,1.0,0.0001);
    }
    sector.crossFunction->setSmoothingParameters(
        toDouble(nextToken()), toDouble(nextToken()),
        toDouble(nextToken()), toDouble(nextToken()));
    object->setField(SECTOR_FIELD, sector);
  }
  PfieldGeometricObject* geometricObject;
  parseGeometricObject(geometricObject);
  object->setGeometry(geometricObject);
  delete geometricObject;
  return object;
}


void Parser::parseAndInsertObject()
{
  Object* object = parseObject();  
  objectMap[object->getName()] = objects.size();
  objects.push_back(object);
}


void Parser::parseInstance()
{
  Object* instance = 0;
  std::string type = nextToken();
  std::string name = nextToken();
  NameToIndexMap::iterator pos;
  pos = objectMap.find(type);
  assert(pos != objectMap.end());
  instance = objects[pos->second]->createInstance(name);
  std::string pose = nextToken();
  if(pose=="static-pose")
  {
	  PfPose pose;
    pose.pos.x = toDouble(nextToken());
    pose.pos.y = toDouble(nextToken());  
    pose.rotation = parseAngle();
    instance->setPose(pose);
    instance->setStatic(true);
    instance->setActivation(true);
  }
  else if(pose=="dynamic-pose")
  {
    std::string poseSymbol = nextToken();
    unsigned int dynamicPoseId = composition->getIdFromObjectStateSymbol(poseSymbol);
    instance->setStatic(false, dynamicPoseId);
    instance->setActivation(false);
  }
  composition->addObject(instance);
}


void Parser::parseSingleFormation(SingleFormation*& formation, 
                                  const std::string& formationToken)
{
  if(formationToken == "between")
  {
    BetweenFormation* betweenFormation = new BetweenFormation();
    betweenFormation->addObject(getInstance(nextToken()));
    betweenFormation->addObject(getInstance(nextToken()));
    formation = betweenFormation;
  }
  else if(formationToken == "among")
  {
    AmongFormation* amongFormation = new AmongFormation();
    std::string groupName(nextToken());
    bool groupFound = false;
    for(unsigned int i=0; i<instanceGroups.size(); i++)
    {
      if(instanceGroups[i].name == groupName)
      {
        for(unsigned int j=0; j<instanceGroups[i].objects.size(); j++)
        {
          amongFormation->addObject(instanceGroups[i].objects[j]);
        }
        groupFound = true;
        break;
      }
    }
    assert(groupFound);
    formation = amongFormation;
  }
  else if(formationToken == "relative-to")
  {
    RelativeFormation* relativeFormation = new RelativeFormation();
    relativeFormation->addObject(getInstance(nextToken()));
    double angle(parseAngle());
    std::string coordinateString(nextToken());
    if(coordinateString == "relative")
    {
      relativeFormation->setParameters(angle, RELATIVE_FORMATION);
    }
    else //coordinateString == "absolute"
    {
      relativeFormation->setParameters(angle, ABSOLUTE_FORMATION);
    }
    formation = relativeFormation;
  }
  formation->setPriority(toDouble(nextToken()));
  PotentialfieldFunction* function;
  parseFunction(function);
  formation->setFunction(function);
  delete function;
  formation->init();
}


void Parser::parseFormationObject()
{
  FormationObject* object;
  std::string name(nextToken());
  object = new FormationObject(name);
  std::string formationToken(nextToken());
  while(formationToken != "EOL")
  {
    if(formationToken != "best-fit")
    {
      SingleFormation* singleFormation;
      parseSingleFormation(singleFormation, formationToken);
      object->addSingleFormation(singleFormation);
    }
    else
    {
      BestFitFormation* bestFitFormation = new BestFitFormation();
      std::string selectionString(nextToken());
      if(selectionString == "max-gradient")
      {
        bestFitFormation->setBestFitSelection(SELECT_MAX_GRADIENT);
      }
      else if(selectionString == "min-gradient")
      {
        bestFitFormation->setBestFitSelection(SELECT_MIN_GRADIENT);
      }
      else if(selectionString == "min-distance")
      {
        bestFitFormation->setBestFitSelection(SELECT_MIN_DISTANCE);
      }
      else //selectionString == "max-priority"
      {
        bestFitFormation->setBestFitSelection(SELECT_MAX_PRIORITY);
      }
      formationToken = nextToken();
      while(formationToken != "EOL")
      { 
        SingleFormation* singleFormation;
        parseSingleFormation(singleFormation, formationToken);
        bestFitFormation->addFormation(singleFormation);
        formationToken = nextToken();
      }
      object->addSingleFormation(bestFitFormation);
    }
    formationToken = nextToken();
  }
  composition->addObject(object);
}


void Parser::parseTimeConstraintsForField(Potentialfield* field)
{
  std::string whatToKeepStr(nextToken());
  KeepType whatToKeep;
  if(whatToKeepStr == "result")
  {
    whatToKeep = KEEP_RESULT;
  }
  else
  {
    whatToKeep = KEEP_FIELD;
  }
  ChangeType changeTypeN(toChangeType(nextToken()));
  int n(toInt(nextToken()));
  ChangeType changeTypeM(toChangeType(nextToken()));
  int m(toInt(nextToken()));
  ChangeType changeTypeO(toChangeType(nextToken()));
  int o(toInt(nextToken()));
  field->setTimeConstraints(whatToKeep, changeTypeN, n, changeTypeM, m,
                            changeTypeO, o);
}


void Parser::parseMotionfield()
{
  std::string name = nextToken();
  Motionfield* motionField = new Motionfield(name);
  bool disableTranslation(toBool(nextToken()));
  bool disableRotation(toBool(nextToken()));
  motionField->disableDegreesOfFreedom(disableTranslation, disableRotation);
  double maxAccel(toDouble(nextToken()));
  double maxGradientDifference(parseAngle());
  motionField->setAccelerationLimits(maxAccel, maxGradientDifference);
  parseTimeConstraintsForField(motionField);
  std::string criterion = nextToken();
  if(criterion == "gradient")
  {
    motionField->setCriterion(CRITERION_GRADIENT);
  }
  else if(criterion == "const")
  {
    motionField->setCriterion(CRITERION_CONST,toDouble(nextToken()));
  }
  std::string combinedField(nextToken());
  while(combinedField != "EOL")
  {
    motionField->addCombinedField(combinedField);
    combinedField = nextToken();
  }
  std::string nameOfFirstObject(nextToken());
  if(nameOfFirstObject == "avoid-local-minima")
  {
    std::string pathPlannerUsage(nextToken());
    bool alwaysPlanning(pathPlannerUsage == "always");
    double gradientLength(toDouble(nextToken()));
    Object* goalObject = getInstance(nextToken());
    PotentialfieldAStarParameterSet parameterSet;
    parameterSet.distanceToGoal = toDouble(nextToken());
    parameterSet.minExpansionRadius = toDouble(nextToken());
    parameterSet.maxExpansionRadius = toDouble(nextToken());
    parameterSet.minBranchingFactor = toInt(nextToken());
    parameterSet.maxBranchingFactor = toInt(nextToken());
    parameterSet.endOfNear = toDouble(nextToken());
    parameterSet.endOfFar = toDouble(nextToken());
    parameterSet.standardGradientLength = toDouble(nextToken());
    parameterSet.numberOfCalls = 0;
    int maxNumberOfSearchNodes(toInt(nextToken()));
    int minCacheSize(toInt(nextToken()));
    nameOfFirstObject = nextToken();
    if(nameOfFirstObject == "stabilization-element")
    {
      parameterSet.useStabilization = true;
      parameterSet.stabilizationDistance = toDouble(nextToken());
      nameOfFirstObject = nextToken();
      parameterSet.stabilizationObject = parseObject();
      nameOfFirstObject = nextToken();
    }
    AStarSearch<PotentialfieldAStarNode, PotentialfieldAStarParameterSet,double>* pathPlanner = 
      new AStarSearch<PotentialfieldAStarNode, PotentialfieldAStarParameterSet,double>(minCacheSize,maxNumberOfSearchNodes,parameterSet);
    motionField->addPathPlanner(pathPlanner, alwaysPlanning, 
                                goalObject, parameterSet, gradientLength);
  }
  if(nameOfFirstObject == "include-random-motion-generator")
  {
    double minValue(toDouble(nextToken()));
    double maxValue(toDouble(nextToken()));
    double valueDx(toDouble(nextToken()));
    double directionDx(parseAngle());
    assert(minValue <= maxValue);
    assert(maxValue >= 0.0);
    assert(valueDx >= 0.0);
    assert(directionDx >= 0.0);
    ChangeType changeType;
    std::string changeString(nextToken());
    if(changeString == "calls") changeType = CALLS;
    else changeType = MILLISECONDS;
    unsigned long n = toInt(nextToken());
    RandomMotionGenerator* rmg = new RandomMotionGenerator
                          (minValue, maxValue, valueDx, directionDx, changeType,n);
    motionField->addRandomMotionGenerator(rmg);
    nameOfFirstObject = nextToken();
  }
  parseObjectsForField(nameOfFirstObject,motionField);
  composition->addField(motionField);
}


void Parser::parseTransformation(PotentialfieldTransformation*& transformation,
                                 const std::string& typeToken)
{
  if (typeToken == "translation")
  {
    Translation* translation = new Translation();
    translation->translation.x = toDouble(nextToken());
    translation->translation.y = toDouble(nextToken());
    translation->time = toDouble(nextToken())/1000.0;
    translation->speed = 0.0;
    translation->alongGradient = false;
    translation->stepLength = 0.0;
    translation->maxGradientDeviation = 0.0;
    translation->toObject = 0;
    transformation = translation;
  }
  else if (typeToken == "translation-to-object")
  {
    Translation* translation = new Translation();
    translation->alongGradient = false;
    translation->stepLength = 0.0;
    translation->maxGradientDeviation = 0.0;
    translation->toObject = getInstance(nextToken());
    translation->speed = toDouble(nextToken());
    translation->time = 0.0;
    transformation = translation;
  }
  else if (typeToken == "translation-along-gradient")
  {
    Translation* translation = new Translation();
    translation->alongGradient = true;
    translation->stepLength = toDouble(nextToken());
    translation->maxGradientDeviation = parseAngle();
    translation->maxLength = toDouble(nextToken());
    translation->speed = toDouble(nextToken());
    translation->time = 0.0;
    translation->toObject = 0;
    transformation = translation;
  }
  else if (typeToken == "rotation")
  {
    Rotation* rotation = new Rotation();
    rotation->toGradient = false;
    rotation->angle = parseAngle();
    rotation->speed = 0.0;
    rotation->time = toDouble(nextToken())/1000.0;
    rotation->toObject = 0;
    transformation = rotation;
  }
  else if (typeToken == "rotation-to-gradient")
  {
    Rotation* rotationToGradient = new Rotation();
    rotationToGradient->toGradient = true;
    rotationToGradient->toObject = 0;
    rotationToGradient->speed = parseAngle();
    rotationToGradient->time = 0.0;
    transformation = rotationToGradient;
  }
  else if (typeToken == "rotation-to-object")
  {
    Rotation* rotationToObject = new Rotation();
    rotationToObject->toGradient = false;
    rotationToObject->toObject = getInstance(nextToken());
    rotationToObject->speed = parseAngle();
    rotationToObject->time = 0.0;
    transformation = rotationToObject;
  }
  else if(typeToken == "no-transformation")
  {
    transformation = new NoTransformation();
    transformation->time = toDouble(nextToken());
    assert(transformation->time != 0.0);
  }
}


void Parser::parseAction(Action& action)
{
  std::string manipulationType = nextToken();
  if(manipulationType == "move-object")
  {
    action.actionType = MOVE_OBJECT;
    action.joinAction = toBool(nextToken());
    action.manipulatedObject = getInstance(nextToken());
    std::string polygonString = nextToken();
    PfieldGeometricObject* geoObj;
    while(polygonString == "polygon")
    {
      parseGeometricObject(geoObj, polygonString);
      Polygon* polygon;
      polygon = dynamic_cast< Polygon* >(geoObj);
      action.impactAreas.push_back(*polygon);
      delete geoObj;
      polygonString = nextToken();
    }
  }
  else if(manipulationType == "move-self")
  {
    action.actionType = MOVE_SELF;
    action.manipulatedObject = 0;
    action.joinAction = false;
  }
  else if(manipulationType == "measure-object")
  {
    action.actionType = MEASURE_OBJECT;
    action.manipulatedObject = getInstance(nextToken());
    action.joinAction = false;
  }
  else // (manipulationType == "measure-self")
  {
    action.actionType = MEASURE_SELF;
    action.manipulatedObject = 0;
    action.joinAction = false;
  }
  if((manipulationType == "move-object") || (manipulationType == "move-self"))
  {
    std::string transformationToken(nextToken());
    PotentialfieldTransformation* transformation;
    if(transformationToken == "probability-distribution")
    {
      transformationToken = nextToken();
      while(transformationToken != "EOL")
      {
        double probability = toDouble(transformationToken);
        transformationToken = nextToken();
        parseTransformation(transformation, transformationToken);
        transformation->probability = probability;
        action.transformations.push_back(transformation);
        transformationToken = nextToken();
      }
    }
    else
    {
      parseTransformation(transformation, transformationToken);
      transformation->probability = 1.0;
      action.transformations.push_back(transformation);
    }
  }
  else
  {
    PotentialfieldTransformation* transformation = new NoTransformation();
    transformation->time = 1.0;
    action.transformations.push_back(transformation);
  }
}


void Parser::parseActionfield()
{
  std::string name = nextToken();
  Actionfield* actionField = new Actionfield(name);
  actionField->setConsiderTime(toBool(nextToken()));
  parseTimeConstraintsForField(actionField);
  std::string criterion = nextToken();
  if(criterion == "gain")
  {
    actionField->setCriterion(CRITERION_GAIN);
  }
  else if(criterion == "gradient")
  {
    actionField->setCriterion(CRITERION_GRADIENT);
  }
  else if(criterion == "absolute")
  {
    actionField->setCriterion(CRITERION_ABSOLUTE);
  }
  else if(criterion == "const")
  {
    actionField->setCriterion(CRITERION_CONST,toDouble(nextToken()));
  }
  std::string nextAction(nextToken());
  if(nextAction == "fixed-sequence") 
  {
    actionField->setActionfieldType(FIXED_SEQUENCE_FIELD);
    nextAction = nextToken();
  }
  else if(nextAction == "find-best-sequence")
  {
    unsigned int depth(toInt(nextToken()));
    bool decreasingValuesOnly(toBool(nextToken()));
    actionField->setActionfieldType(BEST_SEQUENCE_FIELD, 
                                    decreasingValuesOnly, depth);
    nextAction = nextToken();
  }
  else
  {
    actionField->setActionfieldType(SINGLE_ACTION_FIELD);
  }
  while (nextAction != "EOL")
  {
    Action action;
    action.name = nextToken();
    parseAction(action);
    actionField->addAction(action);
    nextAction = nextToken();
  }
  std::string combinedField = nextToken();
  while(combinedField != "EOL")
  {
    actionField->addCombinedField(combinedField);
    combinedField = nextToken();
  }
  std::string nameOfFirstObject(nextToken());
  parseObjectsForField(nameOfFirstObject, actionField);
  composition->addField(actionField);
}


void Parser::parseComposition()
{
  std::string selectionProcedure(nextToken());
  int n = toInt(nextToken());
  if(selectionProcedure == "best-of-n")
  {
    composition->setFieldSelectionParameters(BEST_OF_N, n);
  }
  else
  {
    composition->setFieldSelectionParameters(SUCCESSIVE_N_TIMES, n);
  }
}


void Parser::parse(PotentialfieldComposition* composition, std::string filename)
{
  setlocale(LC_NUMERIC, "C");
  this->composition = composition;
#ifdef POTENTIALFIELDS_FOR_GT2004_
  std::string relativeFileName("Pfield/"+filename);  
  file = new InBinaryFile(relativeFileName.c_str()); 
  assert(file->exists());
#else
  file.open(filename.c_str());
#endif //POTENTIALFIELDS_FOR_GT2004_
  std::string mainToken;
  while(!endOfFile())
  {
    mainToken = nextToken();
    if(mainToken == "object")
    {
      parseAndInsertObject();
    }
	  else if(mainToken == "object-state-symbol")
    {
      parseObjectStateSymbol();
    }
    else if(mainToken == "instance-group")
    {
      parseInstanceGroup();
    }
    else if(mainToken == "formation-object")
    {
      parseFormationObject();
    }
    else if(mainToken == "object-instance")
    {
      parseInstance();
    }
    else if(mainToken == "motionfield")
    {
      parseMotionfield();
    }
    else if(mainToken == "actionfield")
    {
      parseActionfield();
    }
    else if(mainToken == "composition")
    {
      parseComposition();
    }
    else
    {
      assert(mainToken == ""); //Unknown element has been read
    }
  }
#ifdef POTENTIALFIELDS_FOR_GT2004_
  delete file;
#else
  file.close();
#endif //POTENTIALFIELDS_FOR_GT2004_
}



/*
* $Log: PfcParser.cpp,v $
* Revision 1.2  2004/02/23 11:43:26  tim
* fixed warnings
*
* Revision 1.1  2004/01/20 15:42:19  tim
* Added potential fields implementation
*
* Revision 1.15  2003/06/13 14:27:58  tim
* added random generator and tangential fields
*
* Revision 1.14  2003/06/09 21:00:18  tim
* fixed warnings
*
* Revision 1.13  2003/06/09 20:00:04  tim
* Changed potentialfield architecture
*
* Revision 1.12  2003/05/22 14:23:47  tim
* Changed representation of transformations
*
* Revision 1.11  2003/05/20 12:43:43  tim
* Changed function representation, fixed crash on SuperCore, integrated social functions
*
* Revision 1.10  2003/05/15 16:59:57  tim
* fixed warnings
*
* Revision 1.9  2003/05/08 15:26:06  tim
* no message
*
* Revision 1.8  2003/05/07 00:29:48  dueffert
* compatibility restored
*
* Revision 1.7  2003/04/22 14:35:17  tim
* Merged changes from GO
*
* Revision 1.8  2003/04/10 10:51:52  tim
* no message
*
* Revision 1.7  2003/04/09 19:03:06  tim
* Last commit before GermanOpen
*
* Revision 1.6  2003/04/04 14:50:53  tim
* Fixed bugs, added minor features
*
* Revision 1.5  2003/04/03 15:47:32  tim
* Added modelling for teammates
*
* Revision 1.4  2003/04/02 14:39:11  tim
* Changed Bremen Byters Behavior
*
* Revision 1.3  2003/03/28 14:07:53  dueffert
* usage of common pi, warnings removed
*
* Revision 1.2  2003/03/23 20:32:37  loetzsch
* removed green compiler warning: no newline at end of file
*
* Revision 1.1  2003/03/23 17:51:27  tim
* Added potentialfields
*
*/
