/**
* @file Process.h
*
* Contains the definition of class Process and macros for receivers and senders
*/

#ifndef __Process_h_
#define __Process_h_

#include "Tools/MessageQueue/MessageQueue.h"
#include "Tools/Debugging/DebugKeyTable.h"
#include "Tools/Module/ModuleHandler.h"
#include "Platform/ProcessFramework.h"

/**
 * @class Process
 *
 * System independent base class for Processes.
 *
 * Process is the common base for processes in the GT2002 Project. 
 * The embedding into the system environment is done by system dependent frameworks that use
 * derivates of the Process class.
 */
class Process : public PlatformProcess, public MessageHandler
{
public:
  
  /** 
   * Constructor.
   * @param debugIn A reference to an incoming debug queue
   * @param debugOut A reference to an outgoing debug queue
   */
  Process(MessageQueue& debugIn, MessageQueue& debugOut);
  
  /** 
   * The main function is called from the process framework once in each frame. 
   * It does the debug handling and calls the main function in the derivates.
   * @return If 0 is returned, the function main will not be called on a timed basis
   *         again. A positive number specifies the minimum number of microseconds to
   *         wait before the next frame starts. A negative number specifies a cycle 
   *         time, i.e. the time between two activations of the main() function.
   */
  int processMain();
  
  /**
   * The main funtion is called once in each frame.
   * It must be implemented.
   * @return If 0 is returned, the function main will not be called on a timed basis
   *         again. A positive number specifies the minimum number of microseconds to
   *         wait before the next frame starts. A negative number specifies a cycle 
   *         time, i.e. the time between two activations of the main() function.
   */
  virtual int main() = 0;

  /**
   * That function is called once before the first main(). It can be used 
   * for things that can't be done in the constructor.
   */
  virtual void init() {}

  /** 
   * Is called for every incoming debug message.
   * @param message An interface to read the message from the queue
   * @return true if message was handled
   */
  virtual bool handleMessage(InMessage& message);

private:
  /** 
   * Determines if the process is already initialized. Becomes true short 
   * before the first call of main()
   */
  bool initialized;

protected:

  /** A module handler for managing runtime exchangeable modules */
  ModuleHandler moduleHandler;

  /** A table of debug keys for runtime debug requests.*/
  DebugKeyTable debugKeyTable;
  
  /** A queue for incoming debug messages */
  MessageQueue& debugIn;
  
  /** A queue for outgoing debug messages */
  MessageQueue& debugOut;
};

/**
 * This template class implements a sender for debug packages.
 * It ensures that only a package is sent if it is not empty.
 */
template<class T> class MultiDebugSender : public Sender<T>
{
  public:
    /**
     * The constructor.
     * @param process The process this sender is associated with.
     * @param name The Aperios connection name of the sender without the process name.
     * @param blocking Decides whether this sender blocks the execution of the next frame
     *                 until all connected receivers have requested a new package.
     */
    MultiDebugSender(PlatformProcess* process,const char* name,bool blocking)
    : Sender<T>(process,name,blocking) {}

    /**
     * Marks the package for sending and transmits it to all receivers that already requested for it.
     * All other receiver may get it later if they request for it before the package is changed.
     * In function will only send a package if it is not empty.
     */
    virtual void send()
    {
      if(requestedNew() && !isEmpty())
      {
        Sender<T>::send();
        clear();
      }
    }
};

/**
 * This class implements a sender for MessageQueues.
 * It ensures that only a package is sent if it is not empty.
 */
class DebugSender : public MultiDebugSender<MessageQueue>
{
  public:
    /**
     * The constructor.
     * @param process The process this sender is associated with.
     * @param name The Aperios connection name of the sender without the process name.
     * @param blocking Decides whether this sender blocks the execution of the next frame
     *                 until all connected receivers have requested a new package.
     */
    DebugSender(PlatformProcess* process,const char* name,bool blocking)
    : MultiDebugSender<MessageQueue>(process,name,blocking) {}
};

/**
 * The macro declares two debugging queues.
 * The macro shall be the first entry in the declaration of a process.
 */
#define DEBUGGING \
  Receiver<MessageQueue> theDebugReceiver; \
  DebugSender theDebugSender

/**
 * The macro initializes the two debugging queues and the base class.
 * The macro shall be the first entry after the colon in constructor
 * of the process.
 */
#define INIT_DEBUGGING \
  Process(theDebugReceiver,theDebugSender), \
  theDebugReceiver(this,"Receiver.MessageQueue.O",false), \
  theDebugSender(this,"Sender.MessageQueue.S",false)

/**
 * The macro declares a receiver.
 * It must be used inside a declaration of a process, after the macro DEBUGGING.
 * @param type The type of the package. The variable actually declared has
 *             a type compatible to "type" and is called "thetypeReceiver".
 */
#define RECEIVER(type) \
  Receiver<type> the##type##Receiver

/**
 * The macro initializes a receiver for a certain type. It must be part of the 
 * initializer list of the constructor of the process.
 * @param type The type of the package. The variable actually declared has
 *             a type compatible to "type" and is called "thetypeReceiver".
 * @param blocking Decides whether this receiver blocks the execution of the next frame
 *                 until it has received a package.
 */
#define INIT_RECEIVER(type,blocking) \
  the##type##Receiver(this,"Receiver." #type ".O",blocking)

/**
 * The macro declares a sender.
 * It must be used inside a declaration of a process, after the macro DEBUGGING.
 * @param type The type of the package. The variable actually declared has
 *             a type compatible to "type" and is called "thetypeSender".
 */
#define SENDER(type) \
  Sender<type> the##type##Sender

/**
 * The macro initializes a sender for a certain type. It must be part of the 
 * initializer list of the constructor of the process.
 * @param type The type of the package. The variable actually declared has
 *             a type compatible to "type" and is called "thetypeSender".
 * @param blocking Decides whether this sender blocks the execution of the next frame
 *                 until all connected receivers have requested a new package.
 */
#define INIT_SENDER(type,blocking) \
  the##type##Sender(this,"Sender." #type ".S",blocking)

/**
 * The macro declares a receiver for a MessageQueue.
 * It must be used inside a declaration of a process, after the macro DEBUGGING.
 * It shall only be used in the Debug process.
 * @param source The source process of the package. The variable actually declared is
 *               of type MessageQueue and is called "thesourceReceiver".
 */
#define DEBUG_RECEIVER(source) \
  class Receive##source##MessageQueue : public MessageQueue {}; \
  Receiver<Receive##source##MessageQueue> the##source##Receiver

/**
 * The macro initializes a receiver for a MessageQueue. It must be part of the 
 * initializer list of the constructor of the process.
 * @param source The source process of the package. The variable actually declared is
 *               of type MessageQueue and is called "thesourceReceiver".
 */
#define INIT_DEBUG_RECEIVER(source) \
  the##source##Receiver(this,#source "Receiver.MessageQueue.O",false) \

/**
 * The macro declares a sender for a MessageQueue.
 * It must be used inside a declaration of a process, after the macro DEBUGGING.
 * It shall only be used in the Debug process.
 * @param target The target process of the package. The variable actually declared is
 *               of type MessageQueue and is called "thetargetReceiver".
 */
#define DEBUG_SENDER(target) \
  class Send##target##MessageQueue : public MessageQueue {}; \
  MultiDebugSender<Send##target##MessageQueue> the##target##Sender

/**
 * The macro initializes a sender for a MessageQueue. It must be part of the 
 * initializer list of the constructor of the process.
 * @param target The target process of the package. The variable actually declared is
 *               of type MessageQueue and is called "thetargetSender".
 */
#define INIT_DEBUG_SENDER(target) \
  the##target##Sender(this,#target "Sender.MessageQueue.S",false)

#endif //__Process_h_

/*
 * Change log :
 * 
 * $Log: Process.h,v $
 * Revision 1.1  2003/10/07 10:13:21  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1.1.1  2003/07/02 09:40:28  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.5  2003/03/06 11:52:54  dueffert
 * re-order warning removed
 *
 * Revision 1.4  2002/12/07 16:40:45  roefer
 * Blocking for theDebugReceiver changed
 *
 * Revision 1.3  2002/10/14 13:14:24  dueffert
 * doxygen comments corrected
 *
 * Revision 1.2  2002/09/10 21:07:30  loetzsch
 * continued change of module/solution mechanisms
 *
 * Revision 1.1  2002/09/10 15:53:58  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.5  2002/07/23 16:40:12  roefer
 * Router and SimGT2002 adapted to new message queue and streams
 *
 * Revision 1.4  2002/07/23 13:48:40  loetzsch
 * - new streaming classes
 * - removed many #include statements
 * - exchanged StaticQueue by MessageQueue
 * - new debug message handling
 * - general clean up
 *
 * Revision 1.3  2002/07/13 10:54:58  roefer
 * New command and sound sender
 *
 * Revision 1.2  2002/05/14 20:59:21  hebbel
 * added macro for SoundDataReceiver
 *
 * Revision 1.1.1.1  2002/05/10 12:40:32  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.21  2002/05/06 16:05:32  hebbel
 * Added Macro for buffered sender
 *
 * Revision 1.20  2002/05/05 14:06:43  roefer
 * SimGT2002 processes block as the ones on Aperios
 *
 * Revision 1.19  2002/04/21 21:03:07  roefer
 * Again, events block with &-logic
 *
 * Revision 1.18  2002/04/21 16:09:02  roefer
 * Anchor removed
 *
 * Revision 1.17  2002/04/20 15:52:22  roefer
 * Project simpified, WATCH and WATCH_PART added
 *
 * Revision 1.16  2002/04/06 09:55:53  roefer
 * Image and SensorData path through DebugQueues changed
 *
 * Revision 1.15  2002/02/21 14:22:22  loetzsch
 * added inclusion of Player.h and Stopwatch.h
 *
 * Revision 1.14  2002/02/05 20:02:15  risler
 * handleDebugMessage now returns bool, added debug message handling to ImageProcessor
 *
 * Revision 1.13  2002/01/30 17:29:13  loetzsch
 * handleDebugMessage um Parameter timestamp erweitert
 *
 * Revision 1.12  2002/01/22 22:42:25  loetzsch
 * Added init() again.
 *
 * Revision 1.11  2002/01/19 12:43:16  risler
 * enabled SolutionRequest, changed HandleDebugMessage calls
 *
 * Revision 1.10  2002/01/18 23:27:51  loetzsch
 * solutionRequest and setSolutionRequest added
 *
 * Revision 1.9  2002/01/15 16:25:09  roefer
 * init() removed
 *
 * Revision 1.8  2001/12/20 17:12:25  loetzsch
 * Virtual function init(), that is called before the first main(),
 * added.
 *
 * Revision 1.7  2001/12/19 15:20:53  loetzsch
 * Debug Sender nicht mehr blockierend
 *
 * Revision 1.6  2001/12/15 20:32:08  roefer
 * Senders and receivers are now part of the processes
 *
 * Revision 1.5  2001/12/12 18:08:56  loetzsch
 * Streaming- Operatoren fr Bilder eingebaut, DebugKeyTable nicht- statisch gemacht, Debuggin Mechanismen weitergemacht, Bilder aus Logfiles in RobotControl anzeigen, Logfiles in HU1/Debug auf den Stick schreiben
 *
 * Revision 1.4  2001/12/10 17:47:10  risler
 * change log added
 *
 */
