/**
 * @file RingBuffer.h
 *
 * Declaration of class RingBuffer
 *
 * @author Max Risler
 */

#ifndef __RingBuffer_h_
#define __RingBuffer_h_

/**
 * @class RingBuffer
 *
 * template class for cyclic buffering of the last n values of Type V
 */
template <class V, int n> class RingBuffer
{
  public:
    /** Constructor */
    RingBuffer() {init();}

    /**
     * initializes the Ringbuffer
     */
    void init () {current = n - 1; numberOfEntries = 0;}

    /**
     * adds an entry to the buffer
     * \param v value to be added
     */
    void add (const V& v) 
    {
      current++;
      if (current==n) current=0;
      if (++numberOfEntries >= n) numberOfEntries = n;
      buffer[current] = v;
    }

    /**
     * returns an entry
     * \param i index of entry counting from last added (last=0,...)
     * \return a reference to the buffer entry
     */
    V& getEntry (int i)
    {
      int j = current - i;
      j %= n;
      if (j < 0) j += n;
      return buffer[j];
   }

   /**
     * returns an entry
     * \param v the value the entry i shall be updated with
     * \param i index of entry counting from last added (last=0,...)
     */
    void updateEntry(const V& v, int i)
    {
      int j = current - i;
      j %= n;
      if (j < 0) j += n;
			buffer[j] = v;
   }

    /**
     * returns an entry
     * \param i index of entry counting from last added (last=0,...)
     * \return a reference to the buffer entry
     */
    V& operator[] (int i)
    {
      return getEntry(i);
    }

    /**
     * returns a constant entry.
     * \param i index of entry counting from last added (last=0,...)
     * \return a reference to the buffer entry
     */
    const V& operator[] (int i) const
    {
      return buffer[i > current ? n + current - i : current - i];
    }

    int getNumberOfEntries() const
    {
      return numberOfEntries;
    }

  private:
    int current;
    int numberOfEntries;
    V buffer[n];
};

#endif // __RingBuffer_h_

/*
 * Change log :
 * 
 * $Log: RingBuffer.h,v $
 * Revision 1.1  2003/10/07 10:13:21  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.2  2003/08/08 11:37:30  dueffert
 * doxygen docu corrected
 *
 * Revision 1.1.1.1  2003/07/02 09:40:28  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.6  2003/06/24 11:36:12  jhoffman
 * added "updateEntry()"
 *
 * Revision 1.5  2003/04/06 20:56:39  risler
 * bug fix: wrong array-index was used in getEntry and operator[]
 *
 * Revision 1.4  2003/03/12 22:25:35  roefer
 * Access functions speeded up
 *
 * Revision 1.3  2002/11/27 13:51:55  dueffert
 * doxygen docu corrected
 *
 * Revision 1.2  2002/11/19 15:43:03  dueffert
 * doxygen comments corrected
 *
 * Revision 1.1  2002/09/10 15:53:58  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.2  2002/06/20 00:40:22  Thomas Rfer
 * WLan crash removed
 *
 * Revision 1.1.1.1  2002/05/10 12:40:32  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.4  2002/04/25 20:29:58  roefer
 * New BallPercept and BallPosition, GTMath errors in SimGT2002 fixed
 *
 * Revision 1.3  2002/02/25 19:27:49  juengel
 * no message
 *
 * Revision 1.2  2002/02/21 16:29:43  risler
 * added constructor
 *
 * Revision 1.1  2002/02/03 09:32:54  risler
 * RingBuffer from DefaultSensorDataProcessor now is seperate class
 *
 *
 */
