/**
* @file GTXabsl2EngineExecutor.h
* 
* Implementation of class GTXabsl2EngineExecutor.
*
* @author Martin Ltzsch
*/

#ifndef __GTXabsl2EngineExecutor_h_
#define __GTXabsl2EngineExecutor_h_

#include "Tools/Xabsl2/Xabsl2Engine/Xabsl2Engine.h"
#include "Tools/Module/SolutionRequest.h"
#include "Tools/Streams/InStreams.h"
#include "Tools/MessageQueue/InMessage.h"

/** Implements Xabsl2ErrorHandler using the OUTPUT macro */
class GTXabsl2ErrorHandler : public Xabsl2ErrorHandler
{
public:
/** 
* Constructor 
* @param id The id of the Xabsl2Engine. 
  */
  GTXabsl2ErrorHandler(SolutionRequest::xabsl2EngineID id);
  
  /** 
  * Prints out an error
  * @param text The text to display
  */
  virtual void printError(const char* text);
  
  /**
  * Prints out a message
  * @param text The text to display
  */
  virtual void printMessage(const char* text);
  
private:
  /** The id of the Xabsl2Engine. */
  SolutionRequest::xabsl2EngineID id;
};

/** Implements Xabsl2InputSource using the InConfigFile class */
class Xabsl2FileInputSource : public Xabsl2InputSource, public Xabsl2NamedItem
{
public:
/** 
* Constructor. Does not open the file
* @param fileName The file name to open
  */
  Xabsl2FileInputSource(const char* fileName);
  
  /** Destructor */
  ~Xabsl2FileInputSource();
  
  /** opens the source that contains the intermediate code */
  virtual bool open();
  
  /** closes the source */
  virtual void close();
  
  /** reads a numeric value from the input source */
  virtual double readValue();
  
  /** 
  * reads a string from the input source
  * @param destination The position where to write the string
  * @param maxLength the maximum length of the string
  * @return if the read succeded
  */
  virtual bool readString(char* destination, int maxLength);
  
private:
  /** The file to read the data from */
  InConfigFile* file;
};

/** Implements Xabsl2InputSource using the InConfigMessage class */
class Xabsl2MessageInputSource : public Xabsl2InputSource
{
public:
/** 
* Constructor. Does not open the file
* @param message A reference to the message that contains the intermediate code
  */
  Xabsl2MessageInputSource(InConfigMessage& message);
  
  /** Destructor */
  ~Xabsl2MessageInputSource() {};
  
  /** opens the source that contains the intermediate code */
  virtual bool open() {return true;};
  
  /** closes the source */
  virtual void close() {};
  
  /** reads a numeric value from the input source */
  virtual double readValue();
  
  /** 
  * reads a string from the input source
  * @param destination The position where to write the string
  * @param maxLength the maximum length of the string
  * @return if the read succeded
  */
  virtual bool readString(char* destination, int maxLength);
  
private:
  /** The file to read the data from */
  InConfigMessage& message;
};

/**
* @class GTXabsl2EngineExecutor
*
* Executes an Xabsl2Engine in the GT - architecture
*
* @author Martin Ltzsch
*/ 
class GTXabsl2EngineExecutor 
{
public:
/** 
* Constructor.
* @param id The id of the Xabsl2Engine derivate.
* @param module The id of the module (not the solution) that embeds the engine
  */
  GTXabsl2EngineExecutor(SolutionRequest::xabsl2EngineID id,
    SolutionRequest::ModuleID module);
  
  /** destructor */
  ~GTXabsl2EngineExecutor();
  
  /** 
  * Creates a new engine 
  * @param input An input source to read to intermediate code from
  */
  void init(Xabsl2InputSource& input);
  
  /** Executes the engine */
  void executeEngine();
  
  /** Registers symbols and basic behaviors at the engine */
  virtual void registerSymbolsAndBasicBehaviors() = 0;
  
  /** Sets the selected Agent. If the last selected agent was different from
  * the new one, the root option is changed depending on the new agent.
  * @param name The name of the agent
  */
  void setSelectedAgent(const char* name);
  
  /** 
  * Is called for every incoming debug message.
  * @param message An interface to read the message from the queue
  * @return if the messag was read
  */
  virtual bool handleMessage(InMessage& message);
  
protected:
  
  /** An engine that executes the XML formalized behaviors */
  Xabsl2Engine* pEngine;
  
  /** Is invoked when errors occur */
  GTXabsl2ErrorHandler errorHandler;
  
  /** Is called if the engine could not be created */
  virtual void executeIfEngineCouldNotBeCreated() = 0;

  /** 
  * Prints the main action that was generated by the execution of the engine to a string
  * @param buf the string where to print the action
  */
  virtual void printGeneratedMainActionToString(char* buf) = 0;

private:
  
  /** The id of the Xabsl2Engine derivate. */
  SolutionRequest::xabsl2EngineID id;
  
  /** The id of the module (not the solution) that embeds the engine */
  SolutionRequest::ModuleID module;

  //!@name Debug interface to the Xabsl2 Dialog
  //!@{
  
  /** The requested debug mode */
  enum Xabsl2DebugMode { executeRootOption, executeOption, executeBasicBehavior } debugMode;
  
  /** Sends a debug message to the Xabsl2 dialog depending on the last request */
  void sendDebugMessage();
  
  /** The decimal input symbols that are watched by the Xabsl2 Dialog */
  Xabsl2Array<Xabsl2DecimalInputSymbol*> watchedDecimalInputSymbols;
  
  /** The boolean input symbols that are watched by the Xabsl2 Dialog */
  Xabsl2Array<Xabsl2BooleanInputSymbol*> watchedBooleanInputSymbols;
  
  /** The enumerated input symbols that are watched by the Xabsl2 Dialog */
  Xabsl2Array<Xabsl2EnumeratedInputSymbol*> watchedEnumeratedInputSymbols;
  
  /** The enumerated output symbols that are watched by the Xabsl2 Dialog */
  Xabsl2Array<Xabsl2EnumeratedOutputSymbol*> watchedEnumeratedOutputSymbols;
  
  /** The output symbols that are set from the Xabsl2 Dialog */
  Xabsl2Array<Xabsl2EnumeratedOutputSymbol*> setEnumeratedOutputSymbols;

  /** The values for the set output symbols */
  Xabsl2Array<int> setEnumeratedOutputSymbolValues;

  //!@}
};


#endif// __GTXabsl2EngineExecutor_h_

/*
* Change log :
* 
* $Log: GTXabsl2EngineExecutor.h,v $
* Revision 1.2  2004/05/17 18:35:23  loetzsch
* continued support for multiple Xabsl engines in different modules
*
* Revision 1.1  2004/05/14 11:37:08  loetzsch
* support for multiple xabsl2engines in different modules
* preliminary GT2004HeadControl (does not work at all)
*
*/
