/** 
* @file Xabsl2BooleanExpression.cpp
*
* Implementation of Xabsl2BooleanExpression and derivates
* 
* @author Martin Ltzsch
*/

#include "Xabsl2BooleanExpression.h"
#include "Xabsl2Option.h"

Xabsl2BooleanExpression::~Xabsl2BooleanExpression()
{
}

Xabsl2BooleanExpression* Xabsl2BooleanExpression::create(Xabsl2InputSource& input, 
                                                         Xabsl2Option* subsequentOption,
                                                         Xabsl2ErrorHandler& errorHandler,
                                                         Xabsl2Array<double>& parameters,
                                                         Xabsl2Symbols& symbols,
                                                         unsigned long& timeOfOptionExecution,
                                                         unsigned long& timeOfStateExecution)
{
  char c[100];
  Xabsl2BooleanExpression* booleanOperand = 0;
  Xabsl2DecimalExpression* decimalOperand1 = 0;
  Xabsl2DecimalExpression* decimalOperand2 = 0;
  Xabsl2RelationalAndEqualityOperator* relationalAndEqualityOperator = 0;
  input.readString(c,1);
  
  switch (*c)
  {
  case 'r':
    return new Xabsl2BooleanInputSymbolRef(input, errorHandler, symbols);
  case 't':
    return new Xabsl2subsequentOptionReachedTargetStateCondition(subsequentOption, errorHandler);
  case 'c':
    return new Xabsl2EnumeratedInputSymbolComparison(input, errorHandler,symbols);
  case '&':
    {
      XABSL2_DEBUG_INIT(errorHandler.message("creating and operator"));
      
      int numberOfOperands = (int)input.readValue();
      
      Xabsl2AndOperator* andOperator = new Xabsl2AndOperator();
      
      for (int i=0; i<numberOfOperands; i++)
      {
        if (!Xabsl2BooleanExpression::createOperand(booleanOperand, input, subsequentOption, errorHandler, parameters, symbols, timeOfOptionExecution, timeOfStateExecution))
        {
          delete andOperator;
          return 0;
        }
        
        andOperator->addOperand(booleanOperand);
      }
      
      return andOperator;
    }
  case '|':
    {
      XABSL2_DEBUG_INIT(errorHandler.message("creating or operator"));
      
      int numberOfOperands = (int)input.readValue();
      
      Xabsl2OrOperator* orOperator = new Xabsl2OrOperator();
      
      for (int i=0; i<numberOfOperands; i++)
      {
        if (!Xabsl2BooleanExpression::createOperand(booleanOperand, input, subsequentOption, errorHandler, parameters, symbols, timeOfOptionExecution, timeOfStateExecution))
        {
          delete orOperator;
          return 0;
        }
        orOperator->addOperand(booleanOperand);
      }
      
      return orOperator;
    }
  case '!':
    XABSL2_DEBUG_INIT(errorHandler.message("creating not operator"));
    
    if (!Xabsl2BooleanExpression::createOperand(booleanOperand, input, subsequentOption, errorHandler, parameters, symbols, timeOfOptionExecution, timeOfStateExecution))
      return 0;
    
    return new Xabsl2NotOperator(booleanOperand);
  case '=':
    XABSL2_DEBUG_INIT(errorHandler.message("creating == operator"));
    
    if (!Xabsl2DecimalExpression::createOperand(decimalOperand1,input, subsequentOption, errorHandler, parameters, symbols, timeOfOptionExecution, timeOfStateExecution))
      return 0;
    
    if (!Xabsl2DecimalExpression::createOperand(decimalOperand2,input, subsequentOption, errorHandler, parameters, symbols, timeOfOptionExecution, timeOfStateExecution))
      return 0;
    
    relationalAndEqualityOperator = new Xabsl2EqualToOperator();
    relationalAndEqualityOperator->create(decimalOperand1,decimalOperand2);
    return relationalAndEqualityOperator;
  case 'n':
    XABSL2_DEBUG_INIT(errorHandler.message("creating != operator"));
    
    if (!Xabsl2DecimalExpression::createOperand(decimalOperand1,input,subsequentOption,errorHandler, parameters, symbols, timeOfOptionExecution, timeOfStateExecution))
      return 0;
    
    if (!Xabsl2DecimalExpression::createOperand(decimalOperand2,input,subsequentOption,errorHandler, parameters, symbols, timeOfOptionExecution, timeOfStateExecution))
      return 0;
    
    relationalAndEqualityOperator = new Xabsl2NotEqualToOperator();
    relationalAndEqualityOperator->create(decimalOperand1,decimalOperand2);
    return relationalAndEqualityOperator;
  case '<':
    XABSL2_DEBUG_INIT(errorHandler.message("creating < operator"));
    
    if (!Xabsl2DecimalExpression::createOperand(decimalOperand1,input,subsequentOption,errorHandler, parameters, symbols, timeOfOptionExecution, timeOfStateExecution))
      return 0;
    
    if (!Xabsl2DecimalExpression::createOperand(decimalOperand2,input,subsequentOption,errorHandler, parameters, symbols, timeOfOptionExecution, timeOfStateExecution))
      return 0;
    
    relationalAndEqualityOperator = new Xabsl2LessThanOperator();
    relationalAndEqualityOperator->create(decimalOperand1,decimalOperand2);
    return relationalAndEqualityOperator;
  case 'l':
    XABSL2_DEBUG_INIT(errorHandler.message("creating <= operator"));
    
    if (!Xabsl2DecimalExpression::createOperand(decimalOperand1,input,subsequentOption,errorHandler, parameters, symbols, timeOfOptionExecution, timeOfStateExecution))
      return 0;
    
    if (!Xabsl2DecimalExpression::createOperand(decimalOperand2,input,subsequentOption,errorHandler, parameters, symbols, timeOfOptionExecution, timeOfStateExecution))
      return 0;
    
    relationalAndEqualityOperator = new Xabsl2LessThanOrEqualToOperator();
    relationalAndEqualityOperator->create(decimalOperand1,decimalOperand2);
    return relationalAndEqualityOperator;
  case '>':
    XABSL2_DEBUG_INIT(errorHandler.message("creating > operator"));
    
    if (!Xabsl2DecimalExpression::createOperand(decimalOperand1,input,subsequentOption,errorHandler, parameters, symbols, timeOfOptionExecution, timeOfStateExecution))
      return 0;
    
    if (!Xabsl2DecimalExpression::createOperand(decimalOperand2,input,subsequentOption,errorHandler, parameters, symbols, timeOfOptionExecution, timeOfStateExecution))
      return 0;
    
    relationalAndEqualityOperator = new Xabsl2GreaterThanOperator();
    relationalAndEqualityOperator->create(decimalOperand1,decimalOperand2);
    return relationalAndEqualityOperator;
  case 'g':
    XABSL2_DEBUG_INIT(errorHandler.message("creating >= operator"));
    
    if (!Xabsl2DecimalExpression::createOperand(decimalOperand1,input,subsequentOption,errorHandler, parameters, symbols, timeOfOptionExecution, timeOfStateExecution))
      return 0;
    
    if (!Xabsl2DecimalExpression::createOperand(decimalOperand2,input,subsequentOption,errorHandler, parameters, symbols, timeOfOptionExecution, timeOfStateExecution))
      return 0;
    
    relationalAndEqualityOperator = new Xabsl2GreaterThanOrEqualToOperator();
    relationalAndEqualityOperator->create(decimalOperand1,decimalOperand2);
    return relationalAndEqualityOperator;
  default:
    errorHandler.error("Xabsl2BooleanExpression::create(): unknown expression type \"%c\"",*c);
  }
  return 0;
}

bool Xabsl2BooleanExpression::createOperand(Xabsl2BooleanExpression*& operand,
                                            Xabsl2InputSource& input, 
                                            Xabsl2Option* subsequentOption,
                                            Xabsl2ErrorHandler& errorHandler,
                                            Xabsl2Array<double>& parameters,
                                            Xabsl2Symbols& symbols,
                                            unsigned long& timeOfOptionExecution,
                                            unsigned long& timeOfStateExecution)
{
  operand = Xabsl2BooleanExpression::create(input,subsequentOption,errorHandler,
    parameters, symbols, timeOfOptionExecution,timeOfStateExecution);
  
  if (operand == 0) 
  {
    errorHandler.error("Xabsl2BooleanExpression::createOperand(): created operand is 0");
    return false;
  }
  
  if (errorHandler.errorsOccurred)
  {
    errorHandler.error("Xabsl2BooleanExpression::createOperand(): could not create operand");
    if (operand != 0) delete operand;
    return false;
  }
  
  return true;
}

Xabsl2AndOperator::Xabsl2AndOperator() 
{
  operands.clear();
}

Xabsl2AndOperator::~Xabsl2AndOperator()
{
  for (int i=0; i< operands.getSize(); i++)
  {
    if (operands[i]!=0) delete operands[i];
  }
}

bool Xabsl2AndOperator::getValue()
{
  for (int i=0; i< operands.getSize(); i++)
  {
    if (operands[i]->getValue() == false) return false;
  }
  return true;
}

void Xabsl2AndOperator::addOperand(Xabsl2BooleanExpression* operand)
{
  operands.append("",operand);
}

Xabsl2OrOperator::Xabsl2OrOperator() 
{
  operands.clear();
}

Xabsl2OrOperator::~Xabsl2OrOperator()
{
  for (int i=0; i< operands.getSize(); i++)
  {
    if (operands[i]!=0) delete operands[i];
  }
}

bool Xabsl2OrOperator::getValue()
{
  for (int i=0; i< operands.getSize(); i++)
  {
    if (operands[i]->getValue() == true) return true;
  }
  return false;
}

void Xabsl2OrOperator::addOperand(Xabsl2BooleanExpression* operand)
{
  operands.append("",operand);
}

Xabsl2NotOperator::Xabsl2NotOperator(Xabsl2BooleanExpression* operand1) :
operand1(operand1)
{
}

Xabsl2NotOperator::~Xabsl2NotOperator()
{
  if (operand1!=0) delete operand1;
}

bool Xabsl2NotOperator::getValue()
{
  return !(operand1->getValue());
}

Xabsl2BooleanInputSymbolRef::Xabsl2BooleanInputSymbolRef(Xabsl2InputSource& input,
                                                         Xabsl2ErrorHandler& errorHandler, 
                                                         Xabsl2Symbols& symbols)
{
  char buf[100];
  input.readString(buf,99);
  
  XABSL2_DEBUG_INIT(errorHandler.message("creating reference to boolean input symbol \"%s\"",buf));
  
  if (!symbols.existsBooleanInputSymbol(buf))
  {
    errorHandler.error("Xabsl2BooleanInputSymbolRef::Xabls2BooleanInputSymbolRef(): boolean input symbol \"%s\" was not registered at the engine",buf);
    return;
  }
  
  symbol = symbols.getBooleanInputSymbol(buf);
}

bool Xabsl2BooleanInputSymbolRef::getValue()
{
  return symbol->getValue();
}

Xabsl2subsequentOptionReachedTargetStateCondition
::Xabsl2subsequentOptionReachedTargetStateCondition(Xabsl2Option* subsequentOption,
                                                   Xabsl2ErrorHandler& errorHandler)
                                                   : subsequentOption(subsequentOption)
{
  XABSL2_DEBUG_INIT(errorHandler.message("creating a \"subsequent-option-reached-target-state\" element"));
}

bool Xabsl2subsequentOptionReachedTargetStateCondition::getValue()
{
  if (subsequentOption==0)
  {
    return false;
  }
  else
  {
    return subsequentOption->getOptionReachedATargetState();
  }
}

Xabsl2EnumeratedInputSymbolComparison::Xabsl2EnumeratedInputSymbolComparison(Xabsl2InputSource& input,
                                                                             Xabsl2ErrorHandler& errorHandler, Xabsl2Symbols& symbols)
{
  char buf[100];
  input.readString(buf,99);
  
  XABSL2_DEBUG_INIT(errorHandler.message("creating comparison to enumerated input symbol \"%s\"",buf));
  
  if (!symbols.existsEnumeratedInputSymbol(buf))
  {
    errorHandler.error("Xabsl2EnumeratedInputSymbolComparison::Xabsl2EnumeratedInputSymbolComparison(): enumerated input symbol \"%s\" was not registered", buf);
    return;
  }
  
  symbol = symbols.getEnumeratedInputSymbol(buf);
  
  input.readString(buf,99);
  if (!symbol->enumElements.exists(buf))
  {
    errorHandler.error("Xabsl2EnumeratedInputSymbolComparison::Xabsl2EnumeratedInputSymbolComparison(): enum element \"%s\" enumerated input symbol \"%s\" was not registered", buf, symbol->n);
    return;
  }
  
  value = symbol->enumElements.getElement(buf)->v;
}

bool Xabsl2EnumeratedInputSymbolComparison::getValue()
{
  return (symbol->getValue() == value);
}

void Xabsl2RelationalAndEqualityOperator::create(Xabsl2DecimalExpression* operand1,
                                                 Xabsl2DecimalExpression* operand2)
{
  this->operand1 = operand1;
  this->operand2 = operand2;
}

Xabsl2RelationalAndEqualityOperator ::~Xabsl2RelationalAndEqualityOperator ()
{
  if (operand1!=0) delete operand1;
  if (operand2!=0) delete operand2;
}

bool Xabsl2EqualToOperator::getValue()
{
  return (operand1->getValue() == operand2->getValue());
}

bool Xabsl2NotEqualToOperator::getValue()
{
  return (operand1->getValue() != operand2->getValue());
}

bool Xabsl2LessThanOperator::getValue()
{
  return (operand1->getValue() < operand2->getValue());
}

bool Xabsl2LessThanOrEqualToOperator::getValue()
{
  return (operand1->getValue() <= operand2->getValue());
}

bool Xabsl2GreaterThanOperator::getValue()
{
  return (operand1->getValue() > operand2->getValue());
}

bool Xabsl2GreaterThanOrEqualToOperator::getValue()
{
  return (operand1->getValue() >= operand2->getValue());
}



/*
* Change Log:
*
* $Log: Xabsl2BooleanExpression.cpp,v $
* Revision 1.1  2003/10/07 10:13:25  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.8  2003/09/30 10:51:10  dueffert
* typos fixed
*
* Revision 1.7  2003/09/20 16:34:15  loetzsch
* renamed "following-option-..." to "subsequent-option-.." and
* "following-basic-behavior-.." to "subsequent-basic-behavior-.."
* for consistency with publications
*
* Revision 1.6  2003/09/03 20:16:18  loetzsch
* bug fix
*
* Revision 1.5  2003/07/23 22:25:52  loetzsch
* implemented question mark operator
*
* Revision 1.4  2003/07/22 16:54:19  loetzsch
* renamed and_ to andOperator and or_ to orOperator
*
* Revision 1.3  2003/07/22 09:35:20  dueffert
* bugs fixed: and and or are gcc synonyms for && and ||
*
* Revision 1.2  2003/07/21 19:18:06  loetzsch
* - Xabsl2 AND and OR operators now can contain more than 2 operands
* - speed improvements and cleanup
*
* Revision 1.1.1.1  2003/07/02 09:40:29  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.5  2003/05/05 17:47:55  loetzsch
* added in Xabsl2:
* - marking of states as target-states with attribute is-target-state="true"
* - common-decision-tree
*
* Revision 1.4  2003/03/06 11:59:02  dueffert
* unreachable warning removed
*
* Revision 1.3  2003/02/03 17:43:08  loetzsch
* fixed a bug in Xabsl2BooleanInputSymbolRef
*
* Revision 1.2  2003/01/22 12:16:40  dueffert
* doxygen docu corrected
*
* Revision 1.1  2003/01/13 13:18:18  loetzsch
* Creation of boolean and decimal expressions finished.
*
*/
