/** 
* @file Xabsl2BooleanExpression.h
*
* Definition of Xabsl2BooleanExpression and derivates
* 
* @author Martin Ltzsch
*/

#ifndef __Xabsl2BooleanExpression_h_
#define __Xabsl2BooleanExpression_h_

#include "Xabsl2DecimalExpression.h"

// class prototype needed for the subsequentOption paramer of some classes
class Xabsl2Option;

/** 
* @class Xabsl2BooleanExpression
* 
* Base class for all boolean expressions inside an option graph.
*
* @author Martin Ltzsch
*/
class Xabsl2BooleanExpression
{
public:
  /** Evaluates the boolean expression. */
  virtual bool getValue() = 0;
  
  /**
  * Creates a boolean expression depending on the input.
  * @param input An input source for the intermediate code. It must be opened and read until 
  *              A position where a boolean expression starts.
  * @param subsequentOption The subsequent option of the state. 0 if the subsequent behavior is a basic behavior
  * @param errorHandler A reference to a Xabsl2ErrorHandler instance
  * @param parameters The parameters of the option
  * @param symbols All available symbols
  * @param timeOfOptionExecution The time how long the option is already active
  * @param timeOfStateExecution The time how long the state is already active
  */
  static Xabsl2BooleanExpression* create(Xabsl2InputSource& input, 
    Xabsl2Option* subsequentOption,
    Xabsl2ErrorHandler& errorHandler,
    Xabsl2Array<double>& parameters,
    Xabsl2Symbols& symbols,
    unsigned long& timeOfOptionExecution,
    unsigned long& timeOfStateExecution);
  
  /** Destructor */
  virtual ~Xabsl2BooleanExpression() = 0;
private:
/** 
* Creates a boolean expression depending on the input. 
* Used by the create() function to create boolean operands.
* @param operand The expression to be created
* @param input An input source for the intermediate code. It must be opened and read until 
*              A position where a boolean operand starts.
* @param subsequentOption The subsequent option of the state. 0 if the subsequent behavior is a basic behavior
* @param errorHandler A reference to a Xabsl2ErrorHandler instance
* @param parameters The parameters of the option
* @param symbols All available symbols
* @param timeOfOptionExecution The time how long the option is already active
* @param timeOfStateExecution The time how long the state is already active
* @return If the creation was successful
  */
  static bool createOperand(
    Xabsl2BooleanExpression*& operand,
    Xabsl2InputSource& input, 
    Xabsl2Option* subsequentOption,
    Xabsl2ErrorHandler& errorHandler,
    Xabsl2Array<double>& parameters,
    Xabsl2Symbols& symbols,
    unsigned long& timeOfOptionExecution,
    unsigned long& timeOfStateExecution);
};

/** 
* @class Xabsl2AndOperator
*
* Represents an 'and' element of the option graph 
*
* @author Martin Ltzsch
*/
class Xabsl2AndOperator : public Xabsl2BooleanExpression
{
public:
  /** Constructor */
  Xabsl2AndOperator();
  
  /** Destructor. Deletes the two operands */
  ~Xabsl2AndOperator();
  
  /** Evaluates the boolean expression.*/
  virtual bool getValue();
  
  /** Adds an operand to the operands array */
  void addOperand(Xabsl2BooleanExpression* operand);

private:
  /** the 2+n operands of the operator */
  Xabsl2Array<Xabsl2BooleanExpression*> operands;
};

/** 
* @class Xabsl2OrOperator
*
* Represents an 'or' element of the option graph 
*
* @author Martin Ltzsch
*/
class Xabsl2OrOperator : public Xabsl2BooleanExpression
{
public:
  /** Constructor */
  Xabsl2OrOperator();
  
  /** Destructor. Deletes the two operands */
  ~Xabsl2OrOperator();
  
  /** Evaluates the boolean expression. */
  virtual bool getValue();
  
  /** Adds an operand to the operands array */
  void addOperand(Xabsl2BooleanExpression* operand);

private:
  /** the 2+n operands of the operator */
  Xabsl2Array<Xabsl2BooleanExpression*> operands;
};


/** 
* @class Xabsl2NotOperator
*
* Represents an 'not' element of the option graph 
*
* @author Martin Ltzsch
*/
class Xabsl2NotOperator : public Xabsl2BooleanExpression
{
public:
/** 
* Constructor. Creates the element.
* @param operand1 A boolean expression
  */
  Xabsl2NotOperator(Xabsl2BooleanExpression* operand1);
  
  /** Destructor. Deletes the operand */
  ~Xabsl2NotOperator();
  
  /** Evaluates the boolean expression. */
  virtual bool getValue();
  
private:
  /** operand 1 */
  Xabsl2BooleanExpression* operand1;
};

/** 
* @class Xabsl2BooleanInputSymbolRef
*
* Represents an 'boolean-input-symbol-ref' element of the option graph 
*
* @author Martin Ltzsch
*/
class Xabsl2BooleanInputSymbolRef : public Xabsl2BooleanExpression
{
public:
/** 
* Constructor. Creates the element.
* @param input An input source for the intermediate code. It must be opened and read until 
*              A position where the symbol starts.
* @param errorHandler A reference to a Xabsl2ErrorHandler instance
* @param symbols All available symbols
  */
  Xabsl2BooleanInputSymbolRef(Xabsl2InputSource& input,
    Xabsl2ErrorHandler& errorHandler, Xabsl2Symbols& symbols);
  
  /** Evaluates the boolean expression. */
  virtual bool getValue();
  
private:
  /** The referenced symbol */
  Xabsl2BooleanInputSymbol* symbol;
};

/** 
* @class Xabsl2subsequentOptionReachedTargetStateCondition
*
* Represents an 'subsequent-option-reached-target-state' element of the option graph
*
* @author Martin Ltzsch
*/
class Xabsl2subsequentOptionReachedTargetStateCondition : public Xabsl2BooleanExpression
{
public:
/** 
* Constructor. Creates the element.
* @param subsequentOption The subsequent option of the state. 0 if the subsequent behavior is a basic behavior
* @param errorHandler A reference to a Xabsl2ErrorHandler instance
  */
  Xabsl2subsequentOptionReachedTargetStateCondition(
    Xabsl2Option* subsequentOption,
    Xabsl2ErrorHandler& errorHandler);
  
  /** Evaluates the boolean expression. */
  virtual bool getValue();
  
private:
  /** The subsequent option of that state */
  Xabsl2Option* subsequentOption;
};

/** 
* @class Xabsl2EnumeratedInputSymbolComparison
*
* Represents an 'enumerated-input-symbol-comparison' element of the option graph 
*
* @author Martin Ltzsch
*/
class Xabsl2EnumeratedInputSymbolComparison : public Xabsl2BooleanExpression
{
public:
/** 
* Constructor. Creates the element.
* @param input An input source for the intermediate code. It must be opened and read until 
*              A position where the element starts.
* @param errorHandler A reference to a Xabsl2ErrorHandler instance
* @param symbols All available symbols
  */
  Xabsl2EnumeratedInputSymbolComparison(Xabsl2InputSource& input,
    Xabsl2ErrorHandler& errorHandler, Xabsl2Symbols& symbols);
  
  /** Evaluates the boolean expression. */
  virtual bool getValue();
  
private:
  /** The referenced symbol */
  Xabsl2EnumeratedInputSymbol* symbol;
  
  /** The value to compare */
  int value;
};

/**
* @class Xabsl2RelationalAndEqualityOperator
*
* Base class for the operators <, <=, >, >=, == and !=
* 
* @author Martin Ltzsch
*/
class Xabsl2RelationalAndEqualityOperator : public Xabsl2BooleanExpression
{
public:
/** 
* Creates the element.
* @param operand1 A decimal expression
* @param operand2 A decimal expression
  */
  void create(Xabsl2DecimalExpression* operand1, 
    Xabsl2DecimalExpression* operand2);
  
  /** Destructor. Deletes the two operands */
  ~Xabsl2RelationalAndEqualityOperator();
  
  /** Evaluates the boolean expression.*/
  virtual bool getValue() = 0;
  
protected:
  /** operand 1 */
  Xabsl2DecimalExpression* operand1;
  
  /** operand 2 */
  Xabsl2DecimalExpression* operand2;
};

/** 
* @class Xabsl2EqualToOperator
*
* Represents an 'equal-to' element of the option graph 
*
* @author Martin Ltzsch
*/
class Xabsl2EqualToOperator : public Xabsl2RelationalAndEqualityOperator
{
public:
  /** Evaluates the boolean expression.*/
  virtual bool getValue();
};

/** 
* @class Xabsl2NotEqualToOperator
*
* Represents an 'not-equal-to' element of the option graph 
*
* @author Martin Ltzsch
*/
class Xabsl2NotEqualToOperator : public Xabsl2RelationalAndEqualityOperator
{
public:
  /** Evaluates the boolean expression.*/
  virtual bool getValue();
};

/** 
* @class Xabsl2LessThanOperator
*
* Represents an 'less-than' element of the option graph 
*
* @author Martin Ltzsch
*/
class Xabsl2LessThanOperator : public Xabsl2RelationalAndEqualityOperator
{
public:
  /** Evaluates the boolean expression.*/
  virtual bool getValue();
};

/** 
* @class Xabsl2LessThanOrEqualToOperator
*
* Represents an 'less-than-or-equal-to' element of the option graph 
*
* @author Martin Ltzsch
*/
class Xabsl2LessThanOrEqualToOperator : public Xabsl2RelationalAndEqualityOperator
{
public:
  /** Evaluates the boolean expression.*/
  virtual bool getValue();
};

/** 
* @class Xabsl2GreaterThanOperator
*
* Represents an 'greater-than' element of the option graph 
*
* @author Martin Ltzsch
*/
class Xabsl2GreaterThanOperator : public Xabsl2RelationalAndEqualityOperator
{
public:
  /** Evaluates the boolean expression.*/
  virtual bool getValue();
};

/** 
* @class Xabsl2GreaterThanOrEqualToOperator
*
* Represents an 'greater-than-or-equal-to' element of the option graph 
*
* @author Martin Ltzsch
*/
class Xabsl2GreaterThanOrEqualToOperator : public Xabsl2RelationalAndEqualityOperator
{
public:
  /** Evaluates the boolean expression.*/
  virtual bool getValue();
};


#endif //__Xabsl2BooleanExpression_h_

/*
* Change Log:
*
* $Log: Xabsl2BooleanExpression.h,v $
* Revision 1.1  2003/10/07 10:13:25  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.6  2003/09/30 10:51:10  dueffert
* typos fixed
*
* Revision 1.5  2003/09/20 16:34:15  loetzsch
* renamed "following-option-..." to "subsequent-option-.." and
* "following-basic-behavior-.." to "subsequent-basic-behavior-.."
* for consistency with publications
*
* Revision 1.4  2003/09/16 13:27:21  loetzsch
* changed all occurrences of "option tree" to "option graph"
*
* Revision 1.3  2003/07/23 22:25:52  loetzsch
* implemented question mark operator
*
* Revision 1.2  2003/07/21 19:18:06  loetzsch
* - Xabsl2 AND and OR operators now can contain more than 2 operands
* - speed improvements and cleanup
*
* Revision 1.1.1.1  2003/07/02 09:40:29  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.3  2003/05/05 17:47:55  loetzsch
* added in Xabsl2:
* - marking of states as target-states with attribute is-target-state="true"
* - boolean expression "subsequent-option-reached-target-state"
* - common-decision-tree
*
* Revision 1.2  2003/01/15 01:18:46  timrie
* corrected doxygen comments
*
* Revision 1.1  2003/01/13 13:18:18  loetzsch
* Creation of boolean and decimal expressions finished.
*
*/
