/** 
* @file Xabsl2DecimalExpression.h
*
* Definition of Xabsl2DecimalExpression and derivates
* 
* @author Martin Ltzsch
*/

#ifndef __Xabsl2DecimalExpression_h_
#define __Xabsl2DecimalExpression_h_

#include "Xabsl2Symbols.h"

// class prototype that is used by the conditional expression
class Xabsl2BooleanExpression;

// class prototype that is used for the subsequent option parameter 
class Xabsl2Option;

/** 
* @class Xabsl2DecimalExpression
* 
* Base class for all decimal expressions inside an option graph.
*
* @author Martin Ltzsch
*/
class Xabsl2DecimalExpression
{
public:
  /** Calculates the value of the decimal expression. */
  virtual double getValue() = 0;
  
  /**
  * Creates a decimal expression depending on the input.
  * @param input An input source for the intermediate code. It must be opened and read until 
  *              A position where a decimal expression starts.
  * @param subsequentOption The subsequent option of the state. 0 if the subsequent behavior is a basic behavior
  * @param errorHandler A reference to a Xabsl2ErrorHandler instance
  * @param parameters The parameters of the option
  * @param symbols All available symbols
  * @param timeOfOptionExecution The time how long the option is already active
  * @param timeOfStateExecution The time how long the state is already active
  */
  static Xabsl2DecimalExpression* create(Xabsl2InputSource& input, 
    Xabsl2Option* subsequentOption,
    Xabsl2ErrorHandler& errorHandler,
    Xabsl2Array<double>& parameters,
    Xabsl2Symbols& symbols,
    unsigned long& timeOfOptionExecution,
    unsigned long& timeOfStateExecution);

  /** Destructor */
  virtual ~Xabsl2DecimalExpression() = 0;
  
  /** 
  * Creates a decimal expression depending on the input. 
  * Used by the create() function to create decimal operands.
  * @param operand The expression to be created
  * @param input An input source for the intermediate code. It must be opened and read until 
  *              A position where a decimal operand starts.
* @param subsequentOption The subsequent option of the state. 0 if the subsequent behavior is a basic behavior
  * @param errorHandler A reference to a Xabsl2ErrorHandler instance
  * @param parameters The parameters of the option
  * @param symbols All available symbols
  * @param timeOfOptionExecution The time how long the option is already active
  * @param timeOfStateExecution The time how long the state is already active
  * @return If the creation was successful
  */
  static bool createOperand(
    Xabsl2DecimalExpression*& operand,
    Xabsl2InputSource& input, 
    Xabsl2Option* subsequentOption,
    Xabsl2ErrorHandler& errorHandler,
    Xabsl2Array<double>& parameters,
    Xabsl2Symbols& symbols,
    unsigned long& timeOfOptionExecution,
    unsigned long& timeOfStateExecution);
};

/** 
* @class Xabsl2DecimalInputSymbolRef
* 
* Represents a reference to a decimal input symbol.
*
* @author Martin Ltzsch
*/
class Xabsl2DecimalInputSymbolRef : public Xabsl2DecimalExpression
{
public:
/**
* Constructor. Creates the reference 
* @param input An input source for the intermediate code. It must be opened and read until 
*              A position where a decimal operand starts.
* @param errorHandler A reference to a Xabsl2ErrorHandler instance
* @param symbols All available symbols
  */
  Xabsl2DecimalInputSymbolRef(Xabsl2InputSource& input, 
    Xabsl2ErrorHandler& errorHandler,
    Xabsl2Symbols& symbols);
  
  /** Calculates the value of the decimal expression. */
  virtual double getValue();
  
private:
  /** The referenced decimal input symbol */
  Xabsl2DecimalInputSymbol* symbol;
};

/** 
* @class Xabsl2DecimalValue
* 
* Represents a decimal value.
*
* @author Martin Ltzsch
*/
class Xabsl2DecimalValue : public Xabsl2DecimalExpression
{
public:
/**
* Constructor. Creates the value
* @param input An input source for the intermediate code. It must be opened and read until 
*              A position where a value starts.
* @param errorHandler A reference to a Xabsl2ErrorHandler instance
* @param symbols All available symbols
  */
  Xabsl2DecimalValue(Xabsl2InputSource& input, 
    Xabsl2ErrorHandler& errorHandler,
    Xabsl2Symbols& symbols);
  
  /** Calculates the value of the decimal expression. */
  virtual double getValue();
  
private:
  /** The value */
  double value;
};

/** 
* @class Xabsl2OptionParameterRef
* 
* Represents a reference to an option parameter.
*
* @author Martin Ltzsch
*/
class Xabsl2OptionParameterRef : public Xabsl2DecimalExpression
{
public:
/**
* Constructor. Creates the reference 
* @param input An input source for the intermediate code. It must be opened and read until 
*              A position where the expression starts.
* @param errorHandler A reference to a Xabsl2ErrorHandler instance
* @param parameters The parameters of the option
  */
  Xabsl2OptionParameterRef(Xabsl2InputSource& input, 
    Xabsl2ErrorHandler& errorHandler,
    Xabsl2Array<double>& parameters);
  
  /** Calculates the value of the decimal expression. */
  virtual double getValue();
  
private:
  /** A pointer to the parameter */
  double* parameter;
};

/** 
* @class Xabsl2ArithmeticOperator
* 
* Base class for the +, -, *, / and % operator.
*
* @author Martin Ltzsch
*/
class Xabsl2ArithmeticOperator : public Xabsl2DecimalExpression
{
public:
/**
* Creates the operator
* @param operand1 The first operand
* @param operand2 The second operand
  */
  void create(Xabsl2DecimalExpression* operand1, Xabsl2DecimalExpression* operand2);
  
  /** Calculates the value of the decimal expression. */
  virtual double getValue() = 0;
  
  /** Destructor. Deletes the operands */
  ~Xabsl2ArithmeticOperator();
  
protected:
  /** The first operand */
  Xabsl2DecimalExpression* operand1;
  
  /** The second operand */
  Xabsl2DecimalExpression* operand2;
};

/** 
* @class Xabsl2PlusOperator
*
* Represents a + operator in the option graph 
* 
* @author Martin Ltzsch
*/
class Xabsl2PlusOperator : public Xabsl2ArithmeticOperator
{
public:
  /** Calculates the value of the decimal expression. */
  virtual double getValue();
};

/** 
* @class Xabsl2MinusOperator
*
* Represents a - operator in the option graph 
* 
* @author Martin Ltzsch
*/
class Xabsl2MinusOperator : public Xabsl2ArithmeticOperator
{
public:
  /** Calculates the value of the decimal expression. */
  virtual double getValue();
};


/** 
* @class Xabsl2MultiplyOperator
*
* Represents a * operator in the option graph
* 
* @author Martin Ltzsch
*/
class Xabsl2MultiplyOperator : public Xabsl2ArithmeticOperator
{
public:
  /** Calculates the value of the decimal expression. */
  virtual double getValue();
};

/** 
* @class Xabsl2DivideOperator
*
* Represents a / operator in the option graph 
* 
* @author Martin Ltzsch
*/
class Xabsl2DivideOperator : public Xabsl2ArithmeticOperator
{
public:
  /** Calculates the value of the decimal expression. */
  virtual double getValue();
};

/** 
* @class Xabsl2ModOperator
*
* Represents a % operator in the option graph 
* 
* @author Martin Ltzsch
*/
class Xabsl2ModOperator : public Xabsl2ArithmeticOperator
{
public:
  /** Calculates the value of the decimal expression. */
  virtual double getValue();
};

/** 
* @class Xabsl2TimeRef
*
* Represents a time-of-option-execution or time-of-state-execution element in the option graph 
* 
* @author Martin Ltzsch
*/
class Xabsl2TimeRef : public Xabsl2DecimalExpression
{
public:
  /** 
  * Constructor
  * @param errorHandler A reference to a Xabsl2ErrorHandler instance
  * @param time the referenced time
  */
  Xabsl2TimeRef(Xabsl2ErrorHandler& errorHandler,
    unsigned long& time);
  
  /** Calculates the value of the decimal expression. */
  virtual double getValue();
  
private:
  /** The referenced time */
  unsigned long& time;
};

/** 
* @class Xabsl2DecimalInputFunctionCall
* 
* Represents a reference to a decimal input function.
*
* @author Martin Ltzsch
*/
class Xabsl2DecimalInputFunctionCall : public Xabsl2DecimalExpression
{
public:
  /** Calculates the value of the decimal expression. */
  virtual double getValue();
  
  /**
  * Constructor. Creates the function call depending on the input.
  * @param input An input source for the intermediate code. It must be opened and read until 
  *              A position where the function reference starts.
  * @param subsequentOption The subsequent option of the state. 0 if the subsequent behavior is a basic behavior
  * @param errorHandler A reference to a Xabsl2ErrorHandler instance
  * @param parameters The parameters of the option
  * @param symbols All available symbols
  * @param timeOfOptionExecution The time how long the option is already active
  * @param timeOfStateExecution The time how long the state is already active
  */
  Xabsl2DecimalInputFunctionCall(Xabsl2InputSource& input, 
    Xabsl2Option* subsequentOption,
    Xabsl2ErrorHandler& errorHandler,
    Xabsl2Array<double>& parameters,
    Xabsl2Symbols& symbols,
    unsigned long& timeOfOptionExecution,
    unsigned long& timeOfStateExecution);
  
  /** Destructor */
  ~Xabsl2DecimalInputFunctionCall();

private:
  /** The referenced function */
  Xabsl2DecimalInputFunction* function;

  /** Decimal expressions for the parameters */
  Xabsl2Array<Xabsl2DecimalExpression*> parameterValues;

  /** Pointers to the function parameters */
  Xabsl2Array<double*> functionParameters;
};


/** 
* @class Xabsl2ConditionalExpression
* 
* Represents an ANSI C (condition?expression:expression) question mark operator
*
* @author Martin Ltzsch
*/
class Xabsl2ConditionalExpression : public Xabsl2DecimalExpression
{
public:
/**
* Constructor. Creates the expression
* @param input An input source for the intermediate code. It must be opened and read until 
*              A position where a expression starts.
  * @param subsequentOption The subsequent option of the state. 0 if the subsequent behavior is a basic behavior
  * @param errorHandler A reference to a Xabsl2ErrorHandler instance
  * @param parameters The parameters of the option
  * @param symbols All available symbols
  * @param timeOfOptionExecution The time how long the option is already active
  * @param timeOfStateExecution The time how long the state is already active
  */
  Xabsl2ConditionalExpression(Xabsl2InputSource& input, 
    Xabsl2Option* subsequentOption,
    Xabsl2ErrorHandler& errorHandler,
    Xabsl2Array<double>& parameters,
    Xabsl2Symbols& symbols,
    unsigned long& timeOfOptionExecution,
    unsigned long& timeOfStateExecution);  

  /** Destructor */
  ~Xabsl2ConditionalExpression();

  /** Calculates the value of the decimal expression. */
  virtual double getValue();

private:
  /** The condition */
  Xabsl2BooleanExpression* condition;

  /** The expression that is returned when the condition evaluates true */
  Xabsl2DecimalExpression* expression1;

  /** The expression that is returned when the condition evaluates false */
  Xabsl2DecimalExpression* expression2;
};




#endif //__Xabsl2DecimalExpression_h_

/*
* Change Log:
*
* $Log: Xabsl2DecimalExpression.h,v $
* Revision 1.1  2003/10/07 10:13:25  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.6  2003/09/30 10:51:11  dueffert
* typos fixed
*
* Revision 1.5  2003/09/20 16:34:16  loetzsch
* renamed "following-option-..." to "subsequent-option-.." and
* "following-basic-behavior-.." to "subsequent-basic-behavior-.."
* for consistency with publications
*
* Revision 1.4  2003/09/16 13:27:21  loetzsch
* changed all occurrences of "option tree" to "option graph"
*
* Revision 1.3  2003/08/09 12:19:05  loetzsch
* renamed question-mark-operator to conditional-expression
*
* Revision 1.2  2003/07/23 22:25:52  loetzsch
* implemented question mark operator
*
* Revision 1.1.1.1  2003/07/02 09:40:29  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.3  2003/02/22 18:09:40  loetzsch
* changed comments
*
* Revision 1.2  2003/01/15 01:18:57  timrie
* corrected doxygen comments
*
* Revision 1.1  2003/01/13 13:18:18  loetzsch
* Creation of boolean and decimal expressions finished.
*
*/
