/**
* @file Xabsl2Tools.h
*
* Definition of several helper classes for the XabslEngine.
*
* @author Matthias Jngel
* @author Martin Ltzsch
*/

#ifndef __Xabsl2Tools_h_
#define __Xabsl2Tools_h_

#include "Xabsl2Array.h"

/**
* A Xabsl2Engine helper class for reading input data from files or from memory.
*/
class Xabsl2InputSource
{
public:
  /** opens the source that contains the intermediate code */
  virtual bool open() = 0;
  
  /** closes the source */
  virtual void close() = 0;

  /** reads a numeric value from the input source */
  virtual double readValue() = 0;

  /** 
  * reads a string from the input source
  * @param destination The position where to write the string
  * @param maxLength the maximum length of the string
  * @return if the read succeded
  */
  virtual bool readString(char* destination, int maxLength) = 0;
};

/**
* A Xabsl2Engine helper class for handling errors and debug messages
*/
class Xabsl2ErrorHandler
{
public:
  /** constructor */
  Xabsl2ErrorHandler() : errorsOccurred(false) {};

  /** 
  * Prints out an error
  * @param text The text to display
  */
  virtual void printError(const char* text) = 0;

  /**
  * Prints out a message
  * @param text The text to display
  */
  virtual void printMessage(const char* text) = 0;

  /** 
  * Formats a error message and calls the printError() function.
  * @param format Format string as used by printf defined in stdio.h.
  * @param ... See printf in stdio.h.
  */
  void error(const char* format, ...);

  /** 
  * Formats a message and calls the printMessage() function 
  * @param format Format string as used by printf defined in stdio.h.
  * @param ... See printf in stdio.h.
  */
  void message(const char* format, ...);

  /** if errors occurred */
  bool errorsOccurred;

private:
  /** a buffer for errors and debug messages */
  char messageBuffer[300];
};

/**
* @typedef TimeFunction
* A pointer to a function that returns the current system time.
*/
typedef unsigned long (*TimeFunction)();

// If that variable is defined, the engine prints a lot of debug messages during initialization 
// #define XABSL2_DO_DEBUG_INIT

/** Expressions inside that macro are only executed if XABSL2_DEBUG_INIT is defined */
#ifdef XABSL2_DO_DEBUG_INIT
#define XABSL2_DEBUG_INIT(expression) expression
#else
#define XABSL2_DEBUG_INIT(expression) /**/
#endif


#endif //__Xabsl2Tools_h_

/*
* Change Log:
*
* $Log: Xabsl2Tools.h,v $
* Revision 1.1  2003/10/07 10:13:25  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.3  2003/09/30 10:51:11  dueffert
* typos fixed
*
* Revision 1.2  2003/08/09 14:53:10  dueffert
* files and docu beautified
*
* Revision 1.1.1.1  2003/07/02 09:40:29  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.4  2003/01/13 13:19:38  loetzsch
* switched off the XABSL2_DEBUG_INIT macro
*
* Revision 1.3  2002/12/06 21:13:37  loetzsch
* Decimal input symbols can now be registered at the engine
*
* Revision 1.2  2002/12/02 19:56:32  loetzsch
* - Xabsl2Array now seems to work for more than 1 element
* - Basic behaviors now can be registered at the engine
* - Basic behaviors have access to their parameters
*
* Revision 1.1  2002/12/01 17:54:29  loetzsch
* continued Xabsl2Engine: Xabsl2Array seems to work now
*
*/
